# Copyright (c) 2003-2006 Ingeniweb SAS

# This software is subject to the provisions of the GNU General Public
# License, Version 2.0 (GPL).  A copy of the GPL should accompany this
# distribution.  THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL
# EXPRESS OR IMPLIED WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT
# LIMITED TO, THE IMPLIED WARRANTIES OF TITLE, MERCHANTABILITY,
# AGAINST INFRINGEMENT, AND FITNESS FOR A PARTICULAR PURPOSE

# More details in the ``LICENSE`` file included in this package.

"""Plone Article migration module
$Id: Migrator.py,v 1.3 2005/06/15 15:41:32 pjgrizel Exp 
"""
__author__ = ''
__docformat__ = 'restructuredtext'

# Python imports
from StringIO import StringIO
import sys

# Zope imports
from OFS.Image import File

# Archetypes imports
from Products.Archetypes.interfaces.base import IBaseUnit

# CMF imports
from Products.CMFCore.utils import getToolByName

class ZAttachmentAttribute:
    """Fake ZAttachmentAttribute class. Used to get attachment"""
    __underlyingFile__ = None
    title = ''
    filename = ''
    content_type = ''
    
class ZDummyAttachment:
    """Fake ZDummyAttachment class. Used to get attachment data"""
    content = ''

class Migrator:
    """Migration class
    """
    def generateUniqueId(self, type_name = None):
        """generateUniqueId(self, type_name = None) => generate a unique id for the content
        """
        from DateTime import DateTime
        from random import random
        
        now=DateTime()
        time='%s.%s' % (now.strftime('%Y-%m-%d'), str(now.millis())[7:])
        rand=str(random())[2:6]
        prefix=''
        suffix=''

        if type_name is not None:
            prefix = type_name.replace(' ', '_')+'.'
        prefix=prefix.lower()

        return prefix+time+rand+suffix

    def migrate20To30(self, portal, article_brains, stdout, options):
        """Migrate PloneArticle 2.x to PloneArticle 3.x
        """
        
        portal_article = getToolByName(portal, 'portal_article')
        image_type = portal_article.getAllowedImageType()
        attachment_type = portal_article.getAllowedAttachmentType()
        dry_run = options.get('dry_run', False)
        external_images = options.get('external_images', False)
        external_files = options.get('external_files', False)
        limit = options.get('convert_limit', 0)

        stdout = sys.stdout             # XXX TMP
        if stdout is not None:
            stdout.write('Migrate PloneArticle v2.x to v3.x.\r\n')
        
        nb_articles = 0
        already = 0
        
        # If we have a 'limit' option set, we just try to convert <limit> articles maximum.
        # This is so just to avoid Memory Errors.
        if limit:
            limit = min(len(article_brains), limit)
        else:
            limit = len(article_brains)

        stdout.write("There are %d PloneArticle objects.\r\n" % len(article_brains))
        stdout.write("Migrating at most %d articles.\r\n" % limit)

        # Get all PloneArticle in v2.x
        for article_brain in article_brains:
            article = article_brain.getObject()

            # Check limit
            if nb_articles >= limit:
                break

            # Check if it has not been done before
            if not hasattr(article, 'images_ids'):
                already += 1
                continue
                
            # This is an article from v2.x
            parent = article.getParentNode()
            nb_articles += 1

            # Dry run: we simulate
            if not dry_run:
                image_ids = article.images_ids
                attachment_ids = article.attachments_ids

                # Migrate images
                stdout.write("Migrating article at '%s'.\r\n" % article.absolute_url())
                for image_id in image_ids:
                    stdout.write('  ...migrating article image...\r\n')
                    image = getattr(article, image_id)
                    title = image.title_or_id()
                    setattr(image, 'filename', image.getId())

                    if external_images:
                        new_id = self.generateUniqueId()
                        parent.invokeFactory(image_type, new_id, title=title, image=image)
                        new_obj = getattr(parent, new_id)
                        article.addImageFromObject(new_obj)
                    else:
                        article.addImage(title=title, description='', image=image)

                # Migrate attachments
                for attachment_id in attachment_ids:
                    stdout.write('  ...migrating article attachment...\r\n')
                    attachment = getattr(article, attachment_id)
                    attachment.__module__ = 'ZAttachmentAttribute'
                    content_type = attachment.content_type
                    attachment_file = attachment.__underlyingFile__
                    attachment_file.__module__ = 'ZDummyAttachment'
                    title = attachment.title
                    data = attachment_file.content
                    new_file = File('new_file', title, data, content_type)
                    setattr(new_file, 'filename', attachment.filename)

                    if external_images:
                        new_id = self.generateUniqueId()
                        parent.invokeFactory(attachment_type, new_id, title=title, file=new_file)
                        new_obj = getattr(parent, new_id)
                        article.addAttachmentFromObject(new_obj)
                    else:
                        article.addAttachment(title=title, description='', file=new_file)

                # Migrate models
                root_model_name = 'PloneArticleModel'
                model_nb = article.model[len(root_model_name):]
                article.model = 'plonearticle_model%s' % model_nb

                # Delete old attributes
                article.manage_delObjects(image_ids)
                article.manage_delObjects(attachment_ids)
                del article.images_ids
                del article.attachments_ids

                # Try to remove the article from live memory
                article = None
            
        if stdout is not None:
            stdout.write('%s article(s) migrated.\r\n' % nb_articles)
            stdout.write('%s article(s) were already migrated.\r\n' % already)
            stdout.write('Migration from PloneArticle v2.x to v3.x finished.\r\n')
            
    def migrate(self, portal, stdout=None, options={}):
        cat_tool = getToolByName(portal, 'portal_catalog')
        article_brains = cat_tool(portal_type='PloneArticle')
        self.migrate20To30(portal, article_brains, stdout, options)
        article_brains = None


# Others migration script

# Migrate PA created with AT 1.3
# Mimetype of text is text/plain. Change it to text/html

def change_text_mimetype(self, mimetype='text/html', portal_types=None):
    """Migrate mimetype of text_field"""
    
    out = StringIO()
    cat_tool = getToolByName(self, 'portal_catalog')
    
    if portal_types is None:
        portal_types = 'PloneArticle'
    
    article_brains = cat_tool(portal_type=portal_types)
    
    for article_brain in article_brains:
        article = article_brain.getObject()
        
        # Get base unit 
        text_field = article.getField('text')
        bu = text_field.getBaseUnit(article)
        if IBaseUnit.isImplementedBy(bu):
            bu.mimetype = mimetype
            out.write("%s (%s) converted.\n" % (article.absolute_url_path(), article.portal_type))

    out.write('Mimetypes changed.\n')
    return out.getvalue()
