#####################################################################
# 
# Xron -- Zope Scheduled Event Product
# Copyright (C) 2000 Loren Stafford
#  
# Derived from the ZScheduler product version 0.0.7 in accord 
# with the terms of the ZScheduler's license.
#
# Based on code by Martijn Pieters, (c) 1999 Antraciet B.V.
# Used with explicit permission.
# 
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License  
# along with this program; if not, write to the 
#     Free Software Foundation, Inc., 
#     59 Temple Place - Suite 330, 
#     Boston, MA  02111-1307, USA.
# 
#####################################################################
""" Dispatcher for Xron Events """

#The Dispatcher runs as a separate thread
# It uses the Schedule as its primary data structure
# It knows which event is next
# It sleeps until that event or a change in the Schedule
# It fires an event and logs its output

import Loggerr
loggerr=Loggerr.loggerr
#import pdb; pdb.set_trace()

import AccessControl.SecurityManagement, AccessControl.User
import ZODB, ZODB.ZApplication
import Globals, OFS.Application
try:
  from Globals import DateTime
except ImportError:
  from DateTime import DateTime
import sys, string

maxwait=float(10)  # max time to wait between wake-ups in seconds

  
from threading import Thread
import ThreadedAsync

class Dispatcher(Thread):

  def __init__(self, group=None, target=None, name=None, args=(), kwargs={}):
    Thread.__init__(self,group=group,target=target,name=name,args=args,kwargs=kwargs)
    self.setDaemon(1) # so thread will terminate with parent process
    ThreadedAsync.register_loop_callback(self.becomeAsync)

  def becomeAsync(self, map):
    self.start()

def Timer(ScheduleID, ScheduleChange, rpc):     # aka Dispatcher
  #loggerr(0, 'Dispatcher thread started.')
  import Globals
  DB = Globals.DB
  
  # "Log on" as system user
  AccessControl.SecurityManagement.newSecurityManager(
    None, AccessControl.User.system)

  # Set up the "application" object that automagically opens
  # connections
  app=ZODB.ZApplication.ZApplicationWrapper(
    DB, 'Application', OFS.Application.Application, (),
    Globals.VersionNameName)

  # "Log off" as system user
  AccessControl.SecurityManagement.noSecurityManager()

  # infinite loop
  while 1:
    # Good morning. We just woke up.
    # The first thing we need is a new connection.
    loop_app = app()
    try: 
      Schedule = getattr(loop_app, ScheduleID, None)
    except:
      loggerr(301, 'Cannot access catalog. Suspending operation.')
      break

    interval=maxwait # Default sleep time. May be recalculated below
    try: 
      (atime, aurl)=Schedule.armed_event() # Get next armed event
    except:
      loggerr(302,'Cannot access catalog. Suspending operation.')
      break # out of infinite loop
    if atime is None: 
      #loggerr(0, 'No armed events.') # debug
      pass     # Sleep some more
    elif atime.isFuture(): # The next armed event is not yet ready
      #   calculate how long we have to wait
      ainterval=atime.timeTime() - DateTime().timeTime()
      if ainterval < float(0): ainterval=float(0) # Is negative bad? 
      #loggerr(0, 'Next armed event: %s, %s, %s' % (atime, aurl, ainterval)) # debug
      interval=ainterval # Comment out for debugging to limit to maxwait 
      #break              # Sleep some more
    else:                        # This event is ready now.
      # Fire event, and log its output
      emsg= '\nTrigger event: %s\nTrigger time: %s' % (aurl, atime)
      #import pdb; pdb.set_trace()
      furl=string.join((aurl, 'trigger'), '/')
      try:
        (headers,response)=rpc(furl) # Fire event
        dmsg='%s\n' % response
        loggerr(0, emsg, detail=dmsg) # Log the event and its output
      except:
        type, val, tb = sys.exc_info()
        dmsg="Failed to trigger event.\nType=%s\nVal=%s\n" % (type, val)
        loggerr(100, emsg, detail=dmsg)
        del type, val, tb, dmsg
        try: 
          rpc('%s/%s' % (aurl, 'disarm')) # Attempt to disarm
          loggerr(100, 'Disarmed event', detail='')
        except: 
          # aurl is probably pointing to an event that no longer exists
          # or the url doesn't resolve correctly
          loggerr(100, "Failed to disarm event", detail='')
          # Let's just kick it out of the catalog
          # Otherwise, this event will come back to haunt us
          try:
            Schedule.exterminate(aurl) 
            get_transaction().commit()
          except:
            pass
      # Finished processing one event
      loop_app._p_jar.sync() # see ZODB/Connection.py 
      # There may be more events waiting
      #continue  # Go get next event without sleeping
      # We'll just take a cat nap
      interval=float(0)

    # End of inner while 1 loop
    # We're going to sleep now; so, free the connection
    loop_app._p_jar.close()
    del loop_app
    #  Sleep for predetermined interval
    #loggerr(0, 'Going to sleep for %s seconds' % (interval)) # debug
    ScheduleChange.wait(interval) # in seconds (float)

    if ScheduleChange.isSet():
      #loggerr(0, 'Awakened by set event.') # debug
      ScheduleChange.clear()
      # Schedule has changed, we woke up early
      # Loop back to the top and check for
      # an earlier event than we were waiting for
    #else:
      #loggerr(0, 'Awakened by timeout.') # debug
      # We timed out, so there must be a ready event
      # Loop back to the top and trigger the next ready event. 
      # That's probably the one we were waiting for.
      #pass

  # End of outer while 1 loop.
  # Something bad happened, let's clean up before quitting for good
  try:
    loop_app._p_jar.close()
    del loop_app
  except:
    pass
  loggerr(100, 'Dispatcher thread is terminating.')

