###################################################################################################
# _fileutil.py
#
# $Id: _fileutil.py,v 1.6 2004/02/24 19:57:21 dnordmann Exp $
# $Name:  $
# $Author: dnordmann $
# $Revision: 1.6 $
#
# Implementation of file utilities.
# 
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
###################################################################################################

# Imports.
from App.Common import package_home
from OFS.content_types import guess_content_type
import os
import stat
import tempfile


# -------------------------------------------------------------------------------------------------
#  _fileutil.importZexp:
#
# Import file from specified path.
# -------------------------------------------------------------------------------------------------
def importZexp(self, path, filename):
  src_filename = path + filename
  dst_filename = INSTANCE_HOME + '/import/' + filename
  if src_filename != dst_filename:
    try: 
      os.stat(getOSPath(dst_filename)) 
    except OSError: 
      cpFile(src_filename,dst_filename)
  self.manage_importObject(filename)
  remove(dst_filename)


# -------------------------------------------------------------------------------------------------
#  _fileutil.extractFileName:
#
#  Extract filename from path.
#  IN:	path
# -------------------------------------------------------------------------------------------------
def extractFileName(path):
  items = getOSPath(path).split(os.sep)
  lastitem = items[len(items)-1]
  return lastitem


# -------------------------------------------------------------------------------------------------
#  _fileutil.extractFileExt:
# 
#  Extract fileextension from path.
#  IN:	path
#  OUT:	extension
# -------------------------------------------------------------------------------------------------
def extractFileExt(path):
  items = path.split('.')
  lastitem = items[len(items)-1]
  return lastitem


# -------------------------------------------------------------------------------------------------
#  _fileutil.getOSPath:
# 
#  Return path with OS separators.
# -------------------------------------------------------------------------------------------------
def getOSPath(path, sep=None):
  if sep is None: sep = os.sep
  path = path.replace('\\',sep)
  path = path.replace('/',sep)
  for ch in range(256):
    if ch < 32 or ch in [34,60,62] or ch > 126: 
      path = path.replace(chr(ch),'')
  return path


# -------------------------------------------------------------------------------------------------
#  _fileutil.getFilePath:
#
#  Extract filepath from path (cut-off filename).
#  IN:	path
#  OUT:	filepath
# -------------------------------------------------------------------------------------------------
def getFilePath(path):
  items = getOSPath(path).split(os.sep)
  filepath = ''
  for i in range(len(items)-1):
    filepath = filepath + items[i] + os.sep
  if len(filepath) > 0:
    if filepath[-1] == os.sep:
      filepath = filepath[:-1]
  return filepath


# -------------------------------------------------------------------------------------------------
#  _fileutil.findExtension:
#
#  Searches path and all subdirectories for file with extension and returns complete filepath.
#  Returns None if no file with specified extension exists.
#  IN:  extension
#       path
#  OUT: filepath
# -------------------------------------------------------------------------------------------------
def findExtension(extension, path, deep=1):
  rtn = None
  path = getOSPath(path)
  for file in os.listdir(path):
    filepath = path + os.sep + file 
    if extractFileExt(file).lower() == extension:
      return filepath
    elif deep:
      mode = os.stat(filepath)[stat.ST_MODE]
      if stat.S_ISDIR(mode):
        rtn = findExtension(extension,filepath,deep)
        if rtn is not None:
          return rtn
  return rtn


# -------------------------------------------------------------------------------------------------
#  _fileutil.cpFile:
#
#  Copies src to dst.
# -------------------------------------------------------------------------------------------------
def cpFile(src, dst):
  #++ print "[cpFile]: %s to %s"%(src,dst)
  f_src = open(getOSPath(src),'rb')
  f_dst = open(getOSPath(dst),'wb')
  f_dst.write(f_src.read())
  f_dst.close()
  f_src.close()


# -------------------------------------------------------------------------------------------------
#  _fileutil.readPath:
#
#  Reads path.
# -------------------------------------------------------------------------------------------------
def readPath(path):
  l = []
  path = getOSPath(path)
  mode = os.stat(path)[stat.ST_MODE]
  if stat.S_ISDIR(mode):
    for filename in os.listdir(path):
      filepath = path + os.sep + filename
      mode = os.stat(filepath)[stat.ST_MODE]
      if stat.S_ISDIR(mode):
        l.extend(readPath(filepath))
      else:
        f = open(filepath,'rb')
        data = f.read()
        filename = extractFileName(filepath)
        mt, enc  = guess_content_type(filename,data)
        f.close()
        d = {}
        d['filepath']=filepath
        d['filename']=filename
        d['data']=data
        d['content_type']=mt
        l.append(d)
  return l


# -------------------------------------------------------------------------------------------------
#  _fileutil.importPath:
#
#  Imports path.
# -------------------------------------------------------------------------------------------------
def importPath(self, path):
  path = getOSPath(path)
  mode = os.stat(path)[stat.ST_MODE]
  if stat.S_ISDIR(mode):
    for filename in os.listdir(path):
      filepath = path + os.sep + filename
      mode = os.stat(filepath)[stat.ST_MODE]
      if stat.S_ISDIR(mode):
        folder = self.manage_addFolder(id=filename)
        folder = getattr(self,filename)
        importPath(folder,filepath)
      else: 
        f = open(filepath,'rb')
        file = self.manage_addFile(id=filename,file=f.read())
        f.close()


# -------------------------------------------------------------------------------------------------
#  _fileutil.remove:
#
#  Removes path (and all its subdirectories if deep==1).
# -------------------------------------------------------------------------------------------------
def remove(path, deep=0):
  path = getOSPath(path)
  mode = os.stat(path)[stat.ST_MODE]
  if stat.S_ISDIR(mode):
    if deep == 1:
      for filename in os.listdir(path):
        filepath = path + os.sep + filename
        mode = os.stat(filepath)[stat.ST_MODE]
        if stat.S_ISDIR(mode):
          remove(filepath,deep)
        else: 
          os.remove(filepath)
    os.rmdir(path)
  else:
    os.remove(path)


# -------------------------------------------------------------------------------------------------
#  _fileutil.getDataSizeStr: 
#
#  Display string for file-size.
# -------------------------------------------------------------------------------------------------
def getDataSizeStr(len):
  s = ''
  try:
    mod = 0
    while len > 1024.0 and mod < 3:
      len = len / 1024.0
      mod = mod + 1
    s = str(int(len))
    if mod == 0:
      s = "%sBytes"%s
    elif mod == 1:
      s = "%sKB"%s
    elif mod == 2:
      s = "%sMB"%s
    elif mod == 3:
      s = "%sGB"%s
  except:
    pass
  return s


# -------------------------------------------------------------------------------------------------
#  _fileutil.executeCommand:
# -------------------------------------------------------------------------------------------------
def executeCommand(path, command):
  os.chdir(path)
  os.system(command)

    
# -------------------------------------------------------------------------------------------------
#  _fileutil.exportObj:
# -------------------------------------------------------------------------------------------------
def exportObj(obj, filename):
    
  #-- Try to create directory-tree.
  filename = getOSPath(filename)
  filepath = getFilePath(filename)
  mkDir(filepath)
        
  #-- Get object data.   
  try: # ImageFile        
    f = open(obj.path,'rb')
    data = f.read()
    f.close()
  except:
    try: # Image / File
      data = obj.data
      if len(data) == 0:
        data = obj.getData()
    except: 
      try:
        data = obj.raw # DTML Method
      except:
        try:
	  data = obj.read() # REQUEST.enctype multipart/form-data
	except:
          data = str(obj)

  #-- Save to file.
  if data:
    objfile = open(filename,'wb')
    if type(data) is type(''):
      objfile.write(data) 
    else:
      while data is not None:
        objfile.write(data.data)
        data=data.next
    objfile.close()


# -------------------------------------------------------------------------------------------------
#  _fileutil.mkDir:
#
#  Make directory.
# -------------------------------------------------------------------------------------------------
def mkDir(path):
  try:
    os.makedirs(path)
  except:
    pass


# -------------------------------------------------------------------------------------------------
#  _fileutil.readDir
# -------------------------------------------------------------------------------------------------
def readDir(path):
  obs = []
  path = getOSPath(path)
  for file in os.listdir(path):
    ob = {}
    ob['path'] = path + os.sep
    ob['file'] = file
    filepath = path + os.sep + file 
    ob['mtime'] = os.path.getmtime(filepath)
    ob['size'] = os.path.getsize(filepath)
    mode = os.stat(filepath)[stat.ST_MODE]
    if stat.S_ISDIR(mode): 
      ob['type'] = 'd'
    else:
      ob['type'] = 'f'
    obs.append((ob['type'],ob))
  obs.sort()
  return map(lambda ob: ob[1],obs)


"""
###################################################################################################
###
###  P I L   I M A G I N G 
###
###################################################################################################
"""

# -------------------------------------------------------------------------------------------------
#  _fileutil.createThumbnail:
#
#  Creates thumbnail of given image.
#  IN:  img
#  OUT: thumb
# -------------------------------------------------------------------------------------------------
def createThumbnail(img, maxdim=100):
  
  # Save image in temp-folder.
  tempfolder = tempfile.mktemp()
  filename = getOSPath('%s/%s'%(tempfolder,img.filename))
  exportObj(img,filename)
  
  # Call PIL to generate thumbnail.
  command = 'python ' + getOSPath(package_home(globals())+'/conf/pil/img_conv.py') + ' ' + str(maxdim) + ' ' + filename
  os.system(command)
  
  # Read thumbnail from file-system.
  lang_sffx = '_' + img.lang
  thumb_sffx = '_thumbnail'
  filename = filename[:-(len(extractFileExt(filename))+1)]
  f = open(filename + thumb_sffx + '.jpg','rb')
  thumb_data = f.read()
  thumb_filename = filename
  if len(thumb_filename)>len(lang_sffx) and thumb_filename[-len(lang_sffx):]==lang_sffx: thumb_filename = thumb_filename[:-len(lang_sffx)]
  thumb_filename = extractFileName(thumb_filename + thumb_sffx + lang_sffx + '.jpg')
  thumb = {'data':thumb_data,'filename':thumb_filename}
  f.close()
    
  # Remove temp-folder and images.
  remove(tempfolder,deep=1)
  
  # Returns thumbnail.
  return thumb


# -------------------------------------------------------------------------------------------------
#  _fileutil.resizeImage:
#
#  Resize given image.
#  IN:  img
#       perc
#  OUT: resized
# -------------------------------------------------------------------------------------------------
def resizeImage(img, perc):
  
  # Save image in temp-folder.
  tempfolder = tempfile.mktemp()
  filename = getOSPath('%s/%s'%(tempfolder,img.filename))
  exportObj(img,filename)
  
  # Call PIL to generate resized image.
  command = 'python ' + getOSPath(package_home(globals())+'/conf/pil/img_resize.py') + ' ' + str(img.width) + ' ' + str(img.height) + ' ' + filename
  os.system(command)
  
  # Read resized image from file-system.
  f = open(filename,'rb')
  resized_data = f.read()
  resized_filename = extractFileName(filename)
  resized = {'data':resized_data,'filename':resized_filename}
  f.close()
  
  # Remove temp-folder and images.
  remove(tempfolder,deep=1)
  
  # Returns resized image.
  return resized


"""
###################################################################################################
###
###  Z I P   A R C H I V E 
###
###################################################################################################
"""

# -------------------------------------------------------------------------------------------------
#  _fileutil.getZipArchive:
#
#  Extract files from zip-archive and return list of extracted files.
# -------------------------------------------------------------------------------------------------
def getZipArchive(f):
  l = []
  
  # Saved zip-file in temp-folder.
  tempfolder = tempfile.mktemp()
  filename = getOSPath('%s/%s'%(tempfolder,f.filename))
  exportObj(f,filename)
  
  # Unzip zip-file.
  extractZipArchive(filename)
  
  # Remove zip-file.
  remove(filename)
  
  # Read extracted files.
  l = readPath(tempfolder)
  
  # Remove temp-folder.
  remove(tempfolder,deep=1)
  
  # Return list of files.
  return l


# -------------------------------------------------------------------------------------------------
#  _fileutil.extractZipArchive:
#
#  Unpack ZIP-Archive.
# -------------------------------------------------------------------------------------------------
def extractZipArchive(sZipFilename):
  
  # DOS, Win32:
  # pkzip -extract -direct zipfile destinationpath : Extract files from .ZIP
  if os.name in ['nt', 'dos']:
    command = package_home(globals()) + '/bin/win32/unzip.exe %s -d %s'%(sZipFilename,getFilePath(sZipFilename))
    command = getOSPath(command)
  
  # UNIX, Linux, Solaris:
  # unzip -d zipfile : Extract files from .ZIP
  #   -d	extract files into exdir
  else:
    command = 'unzip -d %s %s'%(getFilePath(sZipFilename),sZipFilename)
  
  # Execute command.
  os.system(command)


# -------------------------------------------------------------------------------------------------
#  _fileutil.buildZipArchive:
#
#  Create ZIP-Archive and returns data.
# -------------------------------------------------------------------------------------------------
def buildZipArchive(files):
  
  # Create temporary zip-archive.
  zipfilename = tempfile.mktemp() + '.zip'
  
  # DOS, Win32 (www.info-zip.org):
  #   zip
  #   -r	recurse into directories
  if os.name in ['nt','dos']:
    folder = getFilePath(files)
    command = package_home(globals()) + '/bin/win32/zip -r -b ' + folder + ' ' + zipfilename + ' ' + files
  
  # UNIX, Linux, Solaris:
  #   zip
  #   -r	recurse into directories
  else:
    folder = getFilePath(files)
    files = getOSPath(files)
    files = files.replace('*.*','*')
    if len(folder) > 0:
      files = files.replace(folder, '.')
      command = 'cd %s;' %folder
    else:
      command = ''
    command = '%s zip -r %s %s'%(command,zipfilename,files)
  
  # Execute command.
  command = getOSPath(command)
  os.system(command)
  
  # Read data of zip-archive as return value.
  f = open(zipfilename,'rb')
  data = f.read()
  f.close()
  
  # Remove temporary zip-archive.
  os.remove(zipfilename)
  
  # Returns data of zip-archive.
  return data

###################################################################################################
