###################################################################################################
# _zcatalogmanager.py
#
# $Id: _zcatalogmanager.py,v 1.8 2004/11/30 20:03:17 dnordmann Exp $
# $Name:  $
# $Author: dnordmann $
# $Revision: 1.8 $
#
# Implementation of classes ZCatalogItem and ZCatalogManager (see below).
# 
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
###################################################################################################


# Imports.
from Products.ZCatalog import ZCatalog, Vocabulary, CatalogAwareness
# Product Imports.
import _globals


# -------------------------------------------------------------------------------------------------
#  _zcatalogmanager.zcat_meta_types
# -------------------------------------------------------------------------------------------------
zcat_meta_types = ['ZMS','ZMSRubrik','ZMSDocument','ZMSCustom','ZMSSysFolder','ZMSFile']


# -------------------------------------------------------------------------------------------------
#  _zcatalogmanager.search_encode:
#
#  Encodes given string.
# -------------------------------------------------------------------------------------------------
def search_encode(s):
  try:
    s = unicode( s, 'utf-8').encode( 'latin-1')
    # German Umlauts in capital letters.
    mapping = {
	'\xc4':'\xe4', # Ae
  	'\xd6':'\xf6', # Oe
  	'\xdc':'\xfc', # Ue
    }
    for capital in mapping.keys():
      letter = mapping[ capital]
      while s.find( capital) >= 0:
        s = s.replace( capital, letter)
  except ( UnicodeDecodeError, UnicodeEncodeError):
    pass
  return s
  

# -------------------------------------------------------------------------------------------------
#  _zcatalogmanager.search_string:
#
#  Search string of given value.
# -------------------------------------------------------------------------------------------------
def search_string(v):
  s = ''
  if v is not None:
    if type(v) is type('') and len(v) > 0:
      s += v + ' '
    elif type(v) is list:
      for i in v:
        s += search_string(i)
    elif type(v) is dict:
      for k in v.keys():
        i = v[k]
        s += search_string(i)
  return s

# -------------------------------------------------------------------------------------------------
#  _zcatalogmanager.search_quote:
#
#  Remove HTML-Tags from given string.
# -------------------------------------------------------------------------------------------------
def search_quote(s, maxlen=255, tag='&middot;'):
  # remove all tags.
  stop = False
  while not stop:
    iStartTag = s.find('<')
    iEndTag = s.find('>',iStartTag)
    if iStartTag >= 0 and iEndTag > iStartTag:
      s = s[:iStartTag] + tag + s[iEndTag+1:]
    else:
      stop = True
  # limit characters.
  if len(s) > maxlen:
    blank = s.find(' ',maxlen)
    s = s[:blank] + '&middot;&middot;&middot;'
  # search quote.
  while s.find('>') >= 0:
    i = s.find('>')
    s = s[:i] + '&gt;' + s[i+1:]
  while s.find('<') >= 0:
    i = s.find('<')
    s = s[:i] + '&lt;' + s[i+1:]
  # return quoted search string.
  return s


###################################################################################################
###################################################################################################
###
###   class ZCatalogItem
###
###################################################################################################
###################################################################################################
class ZCatalogItem(CatalogAwareness.CatalogAware): 

    # ---------------------------------------------------------------------------------------------
    #	ZCatalogItem.search_quote:
    #
    #	Remove HTML-Tags.
    # ---------------------------------------------------------------------------------------------
    def search_quote(self, s, maxlen=255, tag='&middot;'):
      return search_quote(s,maxlen,tag)

    
    # ---------------------------------------------------------------------------------------------
    #	ZCatalogItem.getCatalogNavUrl:
    #
    #	Returns catalog-navigation url.
    # ---------------------------------------------------------------------------------------------
    def getCatalogNavUrl(self, REQUEST):
      return self.url_inherit_params(REQUEST['URL'],REQUEST,['qs'])


    # ---------------------------------------------------------------------------------------------
    #  ZCatalogItem.catalogText:
    #
    #  Catalog text.
    # ---------------------------------------------------------------------------------------------
    def catalogText(self, REQUEST):
      v = ''
      if self.isVisible(REQUEST):
        for key in self.getObjAttrs().keys():
          obj_attr = self.getObjAttr(key)
          if obj_attr['xml']:
            datatype = obj_attr['datatype_key']
            if datatype in _globals.DT_STRINGS or \
               datatype == _globals.DT_DICTIONARY or \
               datatype == _globals.DT_LIST:
              value = self.getObjAttrValue(obj_attr,REQUEST)
              value = search_string(value)
              # Increase weight of title-attributes!
              if obj_attr['key'] in ['title','titlealt','titleshort']:
                value = value + value + value
              v += value
        for ob in self.getChildNodes(REQUEST,self.PAGEELEMENTS):
          v += ob.catalogText(REQUEST)
      return v


    # ---------------------------------------------------------------------------------------------
    #  ZCatalogItem.catalogData:
    #
    #  Catalog data.
    # ---------------------------------------------------------------------------------------------
    def catalogData(self, REQUEST):
      v = ''
      if self.isVisible(REQUEST):
        keys = []
        for key in self.getObjAttrs().keys():
          obj_attr = self.getObjAttr(key)
          datatype = obj_attr['datatype_key']
          if datatype == _globals.DT_FILE:
            keys.append( key)
        if len( keys) == 1:
          file = self.getObjProperty( key, REQUEST)
          if file is not None:
            v = file.getData()
      return v


    # ---------------------------------------------------------------------------------------------
    #  ZCatalogItem.zcat_data:
    #
    #  !!! TODO: We need a more generic approach here !!!
    # ---------------------------------------------------------------------------------------------
    def zcat_data_ger( self): return self.catalogData( {'lang':'ger'})
    def zcat_data_eng( self): return self.catalogData( {'lang':'eng'})
    def zcat_data_fra( self): return self.catalogData( {'lang':'fra'})
    def zcat_data_ita( self): return self.catalogData( {'lang':'ita'})
    def zcat_data_rus( self): return self.catalogData( {'lang':'rus'})
    def zcat_data_de( self): return self.catalogData( {'lang':'de'})
    def zcat_data_en( self): return self.catalogData( {'lang':'en'})


    # ---------------------------------------------------------------------------------------------
    #  ZCatalogItem.zcat_text:
    #
    #  !!! TODO: We need a more generic approach here !!!
    # ---------------------------------------------------------------------------------------------
    def zcat_text_ger( self): return search_encode( self.catalogText( {'lang':'ger'}))
    def zcat_text_eng( self): return search_encode( self.catalogText( {'lang':'eng'}))
    def zcat_text_fra( self): return search_encode( self.catalogText( {'lang':'fra'}))
    def zcat_text_ita( self): return search_encode( self.catalogText( {'lang':'ita'}))
    def zcat_text_rus( self): return search_encode( self.catalogText( {'lang':'rus'}))
    def zcat_text_de( self): return search_encode( self.catalogText( {'lang':'de'}))
    def zcat_text_en( self): return search_encode( self.catalogText( {'lang':'en'}))


    # ---------------------------------------------------------------------------------------------
    #  ZCatalogItem.synchronizeSearch:
    # ---------------------------------------------------------------------------------------------
    def synchronizeSearch(self, REQUEST, forced=0):
      if self.getConfProperty('ZMS.CatalogAwareness.active',1) or forced:
        ob = self
        for ref_by in self.getRefByObjs(REQUEST):
          ref_ob = self.getLinkObj(ref_by,REQUEST)
          if ref_ob is not None and ref_ob. meta_type == 'ZMSLinkElement':
            if not forced or ref_ob.getHome().id != self.getHome().id:
              ref_ob.synchronizeSearch( REQUEST=REQUEST, forced=1)
        lang = REQUEST.get( 'lang', self.getPrimaryLanguage())
        while ob is not None:
          if ob.isCatalogItem():
            #  !!! We need some more generic approach here !!!
            if lang in [ 'ger', 'eng', 'fra', 'ita', 'de', 'en']:
              for key in [ 'text', 'data']:
                attr = 'zcat_%s_%s'%(key,lang)
                try: delattr( ob, attr)
                except: pass
            else:
              setattr( ob, 'zcat_text_%s'%lang, search_encode( ob.catalogText(REQUEST)))
              if self.getConfProperty('ZCatalog.TextIndexNG',0)==1:
                setattr( ob, 'zcat_data_%s'%lang, ob.catalogData(REQUEST))
            ob.default_catalog = 'catalog'
            ob.reindex_object()
            break
          ob = ob.getParentNode()


    # ---------------------------------------------------------------------------------------------
    #  ZCatalogItem.isCatalogItem:
    #
    #  Returns true if this is a catalog item.
    # ---------------------------------------------------------------------------------------------
    def isCatalogItem(self):
      b = False
      b = b or self.isPage() 
      b = b or self.meta_type == 'ZMSFile'
      return b


    # ---------------------------------------------------------------------------------------------
    #  ZCatalogItem.reindexCatalogItem:
    #
    #  Reindex catalog item.
    # ---------------------------------------------------------------------------------------------
    def reindexCatalogItem(self, REQUEST):
      message = ''
      if self.isCatalogItem():
	for lang in self.getLangIds():
	  REQUEST.set('lang',lang)
	  self.synchronizeSearch(REQUEST=REQUEST,forced=1)
      lang = REQUEST.get( 'lang', self.getPrimaryLanguage())
      zcatalog = self.getCatalog()
      for index_name in zcatalog.indexes():
        if index_name.find( 'zcat_') != 0:
          key = index_name
          i = key.find( '_%s'%lang)
          if i > 0:
            key = key[ :i]
          if key in self.getObjAttrs().keys():
            obj_attr = self.getObjAttr( key)
            value = self.getObjProperty( key, REQUEST)
            if value is not None:
              datatype = obj_attr['datatype_key']
              if datatype in _globals.DT_DATETIMES:
                from DateTime.DateTime import DateTime
                value = DateTime(self.getLangFmtDate(value))
            setattr( self, index_name, value)
            self.default_catalog = 'catalog'
            self.reindex_object()
      for ob in self.getChildNodes():
        ob.reindexCatalogItem(REQUEST)
      # Return with message.
      return message


###################################################################################################
###################################################################################################
###
###   class ZCatalogManager
###
###################################################################################################
###################################################################################################
class ZCatalogManager:

    # ---------------------------------------------------------------------------------------------
    #  ZCatalogManager.getCatalog:
    #
    #  Returns catalog.
    # ---------------------------------------------------------------------------------------------
    def getCatalog(self):
      context = self.getDocumentElement()
      obs = context.objectValues(['ZCatalog'])
      if len(obs) == 0:
        return self.recreateCatalog()
      return obs[0]


    # ---------------------------------------------------------------------------------------------
    #  ZCatalogManager.recreateCatalog:
    #
    #  Recreates catalog.
    # ---------------------------------------------------------------------------------------------
    def recreateCatalog(self):
      message = ''
      context = self.getDocumentElement()
      
      #-- Get catalog
      obs = context.objectValues( [ 'ZCatalog'])
      if len( obs) == 0:
        cat_id = 'catalog'
        cat_title = 'Catalog'
        vocab_id = 'create_default_catalog_'
        zcatalog = ZCatalog.ZCatalog(cat_id, cat_title, vocab_id, context)
        context._setObject(zcatalog.id, zcatalog)
        zcatalog = self.getCatalog()
      else:
        zcatalog = obs[ 0]
      
      #-- Remove Default-Vocabulary
      ids = zcatalog.objectIds(['Vocabulary'])
      if len(ids) > 0:
        zcatalog.manage_delObjects(ids,None)
      
      #-- Add ISO 8859-1 Vocabulary
      # see ZCatalogs with Umlauts
      # http://www.zope.org/Members/strobl/HowTos/Iszcatalog
      id = "Vocabulary"
      title = id
      globbing = 1
      splitter = "ISO_8859_1_Splitter"
      oVocabulary = Vocabulary.Vocabulary(id,title,globbing,splitter)
      zcatalog._setObject(oVocabulary.id,oVocabulary)
      
      #-- Clear catalog
      zcatalog.manage_catalogClear()
      
      #-- Delete indexes from catalog
      index_names = []
      l = []
      l.extend( zcatalog.schema())
      l.extend( zcatalog.indexes())
      for index_name in l:
        if index_name not in index_names and \
           index_name.find( 'zcat_') == 0:
          try:
            zcatalog.manage_delColumn( [ index_name])
          except:
            _globals.writeException(self,"[recreateCatalog]: Can't delete column '%s' from catalog"%index_name)
          try:
            zcatalog.manage_delIndex( [ index_name])
          except:
            _globals.writeException(self,"[recreateCatalog]: Can't delete index '%s' from catalog"%index_name)
          index_names.append( index_name)
      
      #-- (Re-)create indexes on catalog
      index_types = []
      for index in zcatalog.Indexes.filtered_meta_types():
        index_types.append(index['name'])
      for lang in self.getLangIds():
        message += "Index ["
        index_name = 'zcat_text_%s'%lang
        index_type = 'TextIndex'
        index_extras = None
        zcatalog.manage_addColumn(index_name)
        zcatalog.manage_addIndex(index_name,index_type,index_extras)
        message += index_name
        index_type = None
        for k in [ 'TextIndexNG2', 'TextIndexNG3']:
          if k in index_types: index_type = k
        if self.getConfProperty('ZCatalog.TextIndexNG',0)==1 and index_type is not None:
          index_name = 'zcat_data_%s'%lang
          index_extras = _globals.MyClass()
          setattr(index_extras,'default_encoding','latin-1')
          setattr(index_extras,'indexed_fields',index_name)
          setattr(index_extras,'near_distance',5)
          setattr(index_extras,'splitter_casefolding',1)
          setattr(index_extras,'splitter_max_len',64)
          setattr(index_extras,'splitter_separators','.+-_@')
          setattr(index_extras,'splitter_single_chars',0)
          if index_type == 'TextIndexNG2':
            setattr(index_extras,'use_converters',1)
            setattr(index_extras,'use_normalizer','')
            # setattr(index_extras,'use_stemmer','')
            setattr(index_extras,'use_stopwords','')
          elif index_type == 'TextIndexNG3':
            setattr(index_extras,'use_converters',True)
            setattr(index_extras,'use_normalizer',False)
            setattr(index_extras,'use_stemmer',False)
            setattr(index_extras,'use_stopwords',False)
          zcatalog.manage_addIndex(index_name,index_type,index_extras)
          message += ", "+index_name+"("+index_type+")"
        message += "] created for language <i>"+self.getLanguageLabel(lang)+'</i><br/>'
      
      #-- Return message.
      return message


    # ---------------------------------------------------------------------------------------------
    #  ZCatalogManager.reindexCatalog:
    #
    #  Reindex catalog.
    # ---------------------------------------------------------------------------------------------
    def reindexCatalog(self, REQUEST):
      message = ''
      
      #-- Recreate catalog.
      message += self.recreateCatalog()
      
      #-- Find items to catalog.
      message += self.reindexCatalogItem(REQUEST)
      
      # Return with message.
      message += 'Catalog indexed successfully.'
      return message


    # ---------------------------------------------------------------------------------------------
    #	ZCatalogManager.getCatalogQueryString:
    # ---------------------------------------------------------------------------------------------
    def getCatalogQueryString(self, raw, option='AND'):
      qs = ''
      raw = raw.replace(' and ',' AND ')
      raw = raw.replace(' or ',' OR ')
      if raw.find(' AND ')>=0 and raw.find(' OR ')>=0:
        for raw_item in raw.split(' '):
          if len(raw_item)>0:
            if raw_item in ['AND','OR'] or raw_item[-1]=='*':
              qs += raw_item + ' '
            else:
              qs += raw_item + '* '
      else:
        q = 0
        c = 0
        for raw_item in raw.split(' '):
          if len(raw_item)>0:
            if q==0 and c>0:
              qs += option + ' '
            if raw_item[0]=='*':
              q=1
            if q==1:
              qs += raw_item +' '
            else:
              qs += raw_item + '* '
            if raw_item[-1]=='*':
              q=0
            c=c+1
      qs = qs.replace('-','* AND *')
      qs = qs.strip()
      return qs


    # ---------------------------------------------------------------------------------------------
    #  ZCatalogManager.getCatalogPathObject:
    #
    #  Returns object from catalog-path.
    # ---------------------------------------------------------------------------------------------
    def getCatalogPathObject(self, path):
      ob = self.getHome()
      l = path.split( '/')
      if ob.id not in l:
        docElmnt = self.getDocumentElement()
        if docElmnt.id not in l:
          ob = docElmnt
      else:
        l = l[ l.index(ob.id)+1:]
      for id in l:
         if len( id) > 0 and ob is not None:
          ob = getattr(ob,id,None)
      return ob


    # ---------------------------------------------------------------------------------------------
    #  ZCatalogManager.submitCatalogQuery:
    #
    #  Submits query to catalog.
    # ---------------------------------------------------------------------------------------------
    def submitCatalogQuery(self, search_query, search_order_by, search_meta_types, search_clients, REQUEST):

      #-- Initialize local variables.
      message = ''
      rtn = []
    
      #-- Process clients.
      if search_clients == 1:
        for portalClient in self.getPortalClients():
          rtn.extend(portalClient.submitCatalogQuery( search_query, search_order_by, search_meta_types, search_clients, REQUEST))
          
      #-- Numbers are not cataloged.
      lnum = []
      qs = ''
      lq = search_query.split(' ')
      for i in range(len(lq)):
        try: 
          lnum.append(int(lq[i].strip()))
        except:
          qs += lq[i] + ' '
      
      #-- Search catalog.
      lang = REQUEST['lang']
      zcatalog = self.getCatalog()
      items = []
      for zcindex in zcatalog.indexes():
        if zcindex.find('zcat_')==0 and zcindex.rfind('_'+lang)==len(zcindex)-len('_'+lang):
          d = {}
          d['meta_type'] = zcat_meta_types
          if zcindex.find('zcat_data')==0:
            d[zcindex] = qs
          else:
            d[zcindex] = search_encode( qs)
          if _globals.debug( self):
            _globals.writeLog( self, "[submitCatalogQuery]: %s=%s"%(zcindex,d[zcindex]))
          qr = zcatalog(d)
          if _globals.debug( self):
            _globals.writeLog( self, "[submitCatalogQuery]: qr=%i"%len( qr))
          items.extend( qr)
      
      #-- Sort order.
      if int(search_order_by)==1:
        order_by = 'score'
      else:
        order_by = 'time'
      
      #-- Process results.
      results = []
      for item in items:
        # Get object by path.
        data_record_id = item.data_record_id_
        path = zcatalog.getpath(data_record_id)
        ob = self.getCatalogPathObject( path)
        # Uncatalog object.
        uncatalog = ob is None
        if not uncatalog:
          for path_ob in ob.breadcrumbs_obj_path():
            if not uncatalog:
              uncatalog = uncatalog or path_ob == self.getTrashcan()
              uncatalog = uncatalog or not path_ob.isVisible(REQUEST)
        if uncatalog:
          try:
            zcatalog.uncatalog_object(path)
          except:
            pass
          ob = None
        # Check for valid result.
        append = ob is not None and ob not in map(lambda x: x[1]['ob'], results)
        if append:
          # Handle Pages.
          append = ob.isPage() or ob.meta_type=='ZMSFile'
          # Handle Meta-Types.
          if len(search_meta_types) > 0:
            if ob.meta_type == 'ZMSCustom':
              append = append and ob.meta_id in search_meta_types
            else:
              append = append and ob.meta_type in search_meta_types
          # Handle Numbers.
          zctext = getattr(ob,'zcat_text_%s'%lang,'')
          for num in lnum:
            append = append and zctext.find(str(num))>=0
        # Append to valid results.
        if append:
          sum_time = ob.getObjProperty('attr_dc_date',REQUEST)
          if sum_time is None or len(str(sum_time))==0:
            sum_time = ob.getObjProperty('change_dt',REQUEST)
          try:
            sum_score = int(item.data_record_score_)
          except:
            sum_score = 0
          if ob.meta_type == 'ZMSFile':
            id = ob.id+'/'
            sum_url = ob.getObjProperty('file',REQUEST).getHref(REQUEST)
            sum_url = sum_url[sum_url.find(id)+len(id):]
          else:
            sum_url = 'index_%s.html'%lang
            if REQUEST.get('preview','')=='preview':
              sum_url = self.url_append_params(sum_url,{'preview':'preview'})
            if ob.meta_type=='ZMSCustom':
              sum_url = self.url_append_params(sum_url,{'ZMS_PARAM0':'search','ZMS_PARAM1':search_query})
          result = {}
          result['ob'] = ob
          result['title'] = ob.getTitle(REQUEST)
          result['description'] = ob.getDCDescription(REQUEST)
          result['score'] = sum_score
          result['time'] = sum_time
          result['url'] = ob.getDeclUrl(REQUEST) + '/' + sum_url
          results.append((result[order_by],result))
      
      #-- Sort objects.
      results.sort()
      results.reverse()

      #-- Append objects.
      rtn.extend(map(lambda ob: ob[1],results))
      
      #-- Return objects in correct sort-order.
      return rtn

###################################################################################################
