"""
The main Zwiki module. See README.txt.

(c) 1999-2003 Simon Michael <simon@joyful.com> for the zwiki community.
Wikiwikiweb formatting by Tres Seaver <tseaver@zope.com>
Parenting code and regulations by Ken Manheimer <klm@zope.com>
Initial Zope CMF integration by Chris McDonough <chrism@zope.com>
Full credits are at http://zwiki.org/ZwikiContributors .

This product is available under the GNU GPL.  All rights reserved, all
disclaimers apply, etc.
"""

__version__="$Revision: 0.347 $"[11:-2]

import os, sys, re, string, time, math, traceback
from DateTime import DateTime
from string import split,join,find,lower,rfind,atoi,strip
from types import *
from AccessControl import getSecurityManager, ClassSecurityInfo
import Acquisition
from App.Common import absattr, rfc1123_date, aq_base
import DocumentTemplate
from urllib import quote, unquote
import ZODB
import Globals
from Globals import MessageDialog, package_home
from Products.MailHost.MailHost import MailBase
from OFS.CopySupport import CopyError
from OFS.content_types import guess_content_type
from OFS.Document import Document
from OFS.DTMLDocument import DTMLDocument
import OFS.Image
import StructuredText
from WWML import translate_WMML


from Products.ZWiki import __version__
from Defaults import DEFAULT_PAGE_TYPE, DISABLE_JAVASCRIPT, \
     LARGE_FILE_SIZE, AUTO_UPGRADE, \
     default_standard_wiki_header, default_standard_wiki_footer, \
     default_editform, default_backlinks, default_subscribeform, \
     default_wikipage
import Permissions
from Regexps import url, bracketedexpr, footnoteexpr, wikiname1, \
     wikiname2, simplewikilink, wikilink, interwikilink, remotewikiurl, \
     protected_line, javascriptexpr, dtmlorsgmlexpr, zwikiidcharsexpr, \
     anywikilinkexpr, localwikilink, spaceandlowerexpr, \
     untitledwikilinkexpr, htmlheaderexpr, htmlfooterexpr
from Utils import thunk_substituter, within_literal, html_quote, \
     html_unquote, DLOG, myDocument, ZOPEVERSION, withinSgmlOrDtml, \
     parseHeadersBody
#from Rendering import Rendering
#from Editing import Editing
from UI import UI
from Parents import ParentsSupport
from Diff import DiffSupport
from Mail import MailSupport
from CatalogAwareness import CatalogAwareness
from Tracker import TrackerSupport, ISSUE_FORM
from Regulations import RegulationsSupport
from CMF import CMFAwareness
from Fit import FitSupport
# i18n support
from LocalizerSupport import LocalDTMLFile, _, N_
DTMLFile = LocalDTMLFile
del LocalDTMLFile


class ZWikiPage(
    DTMLDocument,
    UI,
    ParentsSupport,
    DiffSupport,
    MailSupport,
    CatalogAwareness,
    TrackerSupport,
    RegulationsSupport,
    CMFAwareness,
    FitSupport,
    ):
    #Rendering,
    #Editing,
    """
    A ZWikiPage is essentially a DTML Document which knows how to render
    itself in various wiki styles, and can function inside or outside a
    CMF site. A lot of utility methods are provided to support
    wiki-building.

    Mixins are used to organize functionality into distinct modules.
    Initialization, rendering, editing and miscellaneous methods remain in
    the base class.

    RESPONSIBILITIES: (old)

      - render itself

      - provide edit/create forms, backlinks, table of contents

      - accept edit/create requests, with authentication

      - store metadata such as permissions, time, last author, parents etc

      - manage subscriber lists for self & parent folder

    """

    #XXX calculate dynamically ?
    ZWIKI_PAGE_TYPES = (
        'stxprelinkdtmlfitissuehtml',
        'stxprelinkdtmlhtml',
        'stxdtmllinkhtml',
        'dtmlstxlinkhtml',
        'stxprelinkhtml',
        'stxlinkhtml',
        'stxprelink',
        'stxlink',
        'wwmlprelink',
        'wwmllink',
        'prelinkdtmlhtml',
        'dtmllinkhtml',
        'prelinkhtml',
        'linkhtml',
        'dtmlhtml',
        'html',
        'textlink',
        'plaintext',
        'issuedtml',
    )

    meta_type = "ZWiki Page"
    icon      = "misc_/ZWiki/ZWikiPage_icon"
    creator = ''
    creator_ip = ''
    creation_time = ''
    last_editor = ''
    last_editor_ip = ''
    last_edit_time = ''
    last_log = ''
    page_type = DEFAULT_PAGE_TYPE
    _prerendered = '' # cached partially rendered text
    _prelinked = []     # cached links & rendered text regions
    _links = []         # cached unique links

    # properties visible in the ZMI
    # would rather append to the superclass' _properties here
    # DocumentTemplate.inheritedAttribute('_properties'),...) ?
    _properties=(
        {'id':'title', 'type': 'string', 'mode':'w'},
        {'id':'page_type', 'type': 'selection', 'mode': 'w',
         'select_variable': 'ZWIKI_PAGE_TYPES'}, # shows all types
        {'id':'creator', 'type': 'string', 'mode': 'r'},
        {'id':'creator_ip', 'type': 'string', 'mode': 'r'},
        {'id':'creation_time', 'type': 'string', 'mode': 'r'},
        {'id':'last_editor', 'type': 'string', 'mode': 'r'},
        {'id':'last_editor_ip', 'type': 'string', 'mode': 'r'},
        {'id':'last_edit_time', 'type': 'string', 'mode': 'r'},
        {'id':'last_log', 'type': 'string', 'mode': 'r'},
        ) \
        + ParentsSupport._properties \
        + MailSupport._properties \
        + CatalogAwareness._properties

    security = ClassSecurityInfo()
    security.declareObjectProtected('View')
    # set some permissions for superclass methods
    security.declareProtected(Permissions.Change, 'manage_upload')
    # needed ?
    security.declareProtected(Permissions.FTP, 'manage_FTPstat')
    security.declareProtected(Permissions.FTP, 'manage_FTPlist')
    # make sure this appears in the security screen
    security.declareProtected(Permissions.ChangeType, 'dummy')
    def dummy(self):
        pass

    ######################################################################
    # initialization

    def __init__(self, source_string='', mapping=None, __name__=''):
        """
        Initialise this instance, including it's CMF data if applicable.

        Ugly, but putting CMFAwareness before DTMLDocument in the
        inheritance order creates problems.
        """
        if self.supportsCMF():
            CMFAwareness.__init__(self,
                                  source_string=source_string,
                                  mapping=mapping,
                                  __name__=__name__,
                                  )
        else:
            DTMLDocument.__init__(self,
                                  source_string=source_string,
                                  mapping=mapping,
                                  __name__=__name__,
                                  )

    ######################################################################
    # rendering

    security.declareProtected(Permissions.View, '__call__')
    def __call__(self, client=None, REQUEST={}, RESPONSE=None, **kw):
        """
        Render this zwiki page, upgrading it on the fly if needed

        Similar situation to __init__
        """
        if AUTO_UPGRADE: self.upgrade(REQUEST)
        if self.supportsCMF() and self.inCMF():
            return apply(CMFAwareness.__call__,
                         (self,client,REQUEST,RESPONSE),kw)
        else:
            body = apply(self._render,(client,REQUEST,RESPONSE),kw)
            if RESPONSE is not None:
                RESPONSE.setHeader('Content-Type', 'text/html')
                #RESPONSE.setHeader('Last-Modified', rfc1123_date(self._p_mtime)) 
                #causes browser caching problems ? 
            return body

    def _render(self, client=None, REQUEST={}, RESPONSE=None, **kw):
        """
        Render the body of this zwiki page according to it's page_type
        """
        method = self._renderMethod()
        return apply(method,(self, REQUEST, RESPONSE), kw)

    def _renderMethod(self):
        r = 'render_' + self.page_type
        if hasattr(self, r):
            method = getattr(self,r)
        else:
            method = self.render_plaintext
        return method

    def _preRender(self,clear_cache=0):
        """
        Make sure any applicable pre-rendering for this page has been done
        
        each render_* method knows how to do it's own prerendering..
        pass the pre_only flag in kw to preserve the standard dtml method
        signature
        if clear_cache is 1, blow away any cached data
        """
        if clear_cache:
            self.clearCache()
        return apply(self._render,(self, {}, None), {'pre_only':1})

    security.declareProtected(Permissions.View, 'clearCache')
    def clearCache(self,REQUEST=None):
        """
        forcibly clear out any cached render data for this page
        """
        self._prerendered = ''
        self._prelinked = []
        self._links = []
        if REQUEST:
            REQUEST.RESPONSE.redirect(self.page_url())

    security.declareProtected(Permissions.View, 'addStandardLayoutTo')
    def addStandardLayoutTo(self,body,**kw):
        """
        Add standard wiki page layout to the rendered page body.

        Tries these alternatives:
        
        1. if a wikipage page template is found in the zodb (in this
           folder or acquired, use that

        2. if a standard_wiki_header or standard_wiki_footer dtml method
        is found in the zodb, use that. If one is missing, use the default
        value for the other. Note the defaults for these methods were last
        updated just before 0.10.0.
        
        3. otherwise, use the default page template from the filesystem.
.
        Also checks for bare/noheader/nofooter keywords.
        """
        #XXX let CMF handle skinning
        if self.supportsCMF() and self.inCMF():
            return body

        REQUEST = getattr(self,'REQUEST',None)
        RESPONSE = getattr(REQUEST,'RESPONSE',None)
        folder = self.folder()
        if (hasattr(folder,'wikipage') and
            not hasattr(REQUEST,'bare') and
            not kw.has_key('bare') and
            getattr(folder.wikipage,'meta_type',None) == 'Page Template'):
            if kw:
                kw['body'] = body
            else:
                kw = {'body':body}
            return apply(folder.wikipage.__of__(self),(),kw)
                                        # XXX REQUEST ?
        elif ((hasattr(folder,'standard_wiki_header') and
               getattr(folder.standard_wiki_header,
                       'meta_type',None) == 'DTML Method') or
              (hasattr(folder,'standard_wiki_footer') and
               getattr(folder.standard_wiki_footer,
                       'meta_type',None) == 'DTML Method')):
            header = apply(self._renderHeaderOrFooter,('header',REQUEST,RESPONSE),kw)
            footer = apply(self._renderHeaderOrFooter,('footer',REQUEST,RESPONSE),kw)
            return header + body + footer
        elif (self.wikipage and
              not hasattr(REQUEST,'bare') and
              not kw.has_key('bare')):
            if kw:
                kw['body'] = body
            else:
                kw = {'body':body}
            #return apply(default_wikipage.__of__(self),(),kw)
            return apply(self.wikipage,(),kw)
        else:
            return body

    def _renderHeaderOrFooter(self, hdrOrFtr, REQUEST, RESPONSE, **kw):
        """
        Generate the standard wiki header or footer for this page.

        Also check for various options which may disable this.
        hdrOrFtr should be 'header' or 'footer'
        """
        # actually a flag in REQUEST is not much use, try a keyword arg
        # why doesn't <dtml-var WikiPage bare> work yet ?
        if (REQUEST is not None and
            not (hasattr(REQUEST,'bare') or
                 hasattr(REQUEST,'no'+hdrOrFtr) or
                 kw.has_key('bare') or
                 kw.has_key('no'+hdrOrFtr)) and
            (hdrOrFtr == 'header' or hdrOrFtr == 'footer')):
            method = getattr(self,'standard_wiki_'+hdrOrFtr)
            return apply(method,(self, REQUEST, RESPONSE), kw)
        else:
            return ''

    # override cook so DTML will use our pre-rendered data if available
    import thread
    def cook(self,
             cooklock=thread.allocate_lock(),
             ):
        cooklock.acquire()
        try:
            self._v_blocks=self.parse(self._prerendered or self.read())
            self._v_cooked=None
        finally:
            cooklock.release()

    # built-in render methods (page types)
    #
    # add new ones as dtml/python/external methods. I inlined all these
    # for maximum control. Perhaps these should become objects, but no
    # hurry; they still have the signature of a DTML method, which seems
    # possibly useful
    #
    # for 0.9.10, added pre-rendering (and pre-linking)
    # same goal as in WikiForNow, different implementation
    # each render method does as much pre-rendering as it can and caches
    # it in _prerendered; this is updated if needed, and also when the
    # page text or type is changed.
    # this allows flexibility for the different page types and
    # keeps the rendering code together
    # NB _prerendered == '' is assumed to mean this page has not been
    # pre-rendered, which may not be true
    # All render methods may be called with pre_only:1 in kw to do only
    # the pre-rendering
    #
    # Could this be simpler ? Perhaps. We are exploring a range of
    # combinations of rendering rules and strategies, while meeting the
    # constraints of DTML, Structured Text etc. and aiming for blazing
    # performance. It's a bit of a dance.

    security.declareProtected(Permissions.View, 'render_stxprelinkdtmlfitissuehtml')
    def render_stxprelinkdtmlfitissuehtml(self, client=None, REQUEST={},
                                          RESPONSE=None, **kw):
        """
        Render the page with all available zwiki bells and whistles.

        That is: do structured text (with pre-formatting) and wiki links
        (with pre-linking), execute any DTML, execute any fit test tables,
        add an issue properties form (if indicated by the page name), and
        display the result as HTML.
        """
        # pre render
        t = str(self.read())
        if not self._prerendered:
            get_transaction().note('prerender')
            t = self.applyLineEscapesIn(t)
            t = self.stxToHtml(t)
            self._prerendered = t or '\n'
            self.cook()      # pre-parse DTML
            self._preLink()  # pre-parse links
        if kw.get('pre_only',0): return
        # final render
        self._renderWithLinks()
        t = apply(DTMLDocument.__call__,(self, client, REQUEST, RESPONSE), kw)
        t = self._renderLateLinkTitles(t)
        if self.hasFitTests():
            t = self.runFitTestsIn(t)
        if self.isIssue():
            t = self.stxToHtml(apply(ISSUE_FORM.__call__,(self, REQUEST),kw)) + t
        t = apply(self.addStandardLayoutTo,(t,),kw)
        return t

    security.declareProtected(Permissions.View, 'render_stxprelinkdtmlhtml')
    def render_stxprelinkdtmlhtml(self, client=None, REQUEST={},
                                  RESPONSE=None, **kw):
        """
        render the page using structured text + wiki links + DTML + HTML

        As well as stx formatting, this one does pre-linking - as much as
        possible of the linking work - ahead of time.
        """
        # pre render
        t = str(self.read())
        if not self._prerendered:
            get_transaction().note('prerender')
            t = self.applyLineEscapesIn(t)
            t = self.stxToHtml(t)
            self._prerendered = t or '\n'
            self.cook()      # pre-parse DTML
            self._preLink()  # pre-parse links
        if kw.get('pre_only',0): return
        # final render
        self._renderWithLinks()
        t = apply(DTMLDocument.__call__,(self, client, REQUEST, RESPONSE), kw)
        t = self._renderLateLinkTitles(t)
        t = apply(self.addStandardLayoutTo,(t,),kw)
        return t

    security.declareProtected(Permissions.View, 'render_stxdtmllinkhtml')
    def render_stxdtmllinkhtml(self, client=None, REQUEST={},
                               RESPONSE=None, **kw):
        """
        render the page using structured text + DTML + wiki links + HTML

        Behaviour change since 0.9.9: switches stx/dtml rendering order.
        This one caches the structured text formatting ahead of time.
        """
        # pre render
        t = str(self.read())
        if not self._prerendered:
            get_transaction().note('prerender')
            t = self.applyLineEscapesIn(t)
            t = self.stxToHtml(t)
            self._prerendered = t or '\n'
            self.cook()
        if kw.get('pre_only',0): return
        # final render
        t = apply(DTMLDocument.__call__,(self, client, REQUEST, RESPONSE), kw)
        t = self.renderLinksIn(t)
        t = apply(self.addStandardLayoutTo,(t,),kw)
        return t

    security.declareProtected(Permissions.View, 'render_dtmlstxlinkhtml')
    def render_dtmlstxlinkhtml(self, client=None, REQUEST={},
                               RESPONSE=None, **kw):
        """
        render the page using DTML + structured text + wiki links + HTML

        This is the old DTML-first non-caching version used up till 0.9.9.
        """
        # pre render
        t = str(self.read())
        if not self._prerendered:
            get_transaction().note('prerender')
            self._prerendered = t or '\n'
            self.cook()
        if kw.get('pre_only',0): return
        # final render
        t = apply(DTMLDocument.__call__,(self, client, REQUEST, RESPONSE), kw)
        t = self.applyLineEscapesIn(t)
        t = self.stxToHtml(t)
        t = self.renderLinksIn(t)
        t = apply(self.addStandardLayoutTo,(t,),kw)
        return t

    security.declareProtected(Permissions.View, 'render_stxprelinkhtml')
    def render_stxprelinkhtml(self, client=None, REQUEST={},
                              RESPONSE=None, **kw):
        """
        render the page using structured text + wiki links + HTML, prelinking
        """
        # pre render
        t = str(self.read())
        if not self._prerendered:
            get_transaction().note('prerender')
            t = self.applyLineEscapesIn(t)
            t = self.stxToHtml(t)
            self._prerendered = t or '\n'
            self._preLink() 
        if kw.get('pre_only',0): return
        # final render
        self._renderWithLinks()
        t = self._prerendered
        t = self._renderLateLinkTitles(t)
        t = apply(self.addStandardLayoutTo,(t,),kw)
        return t

    security.declareProtected(Permissions.View, 'render_stxlinkhtml')
    def render_stxlinkhtml(self, client=None, REQUEST={},
                           RESPONSE=None, **kw):
        """
        render the page using structured text + wiki links + HTML
        """
        # pre render
        t = str(self.read())
        if not self._prerendered:
            get_transaction().note('prerender')
            t = self.applyLineEscapesIn(t)
            t = self.stxToHtml(t) or '\n'
            self._prerendered = t
        if kw.get('pre_only',0): return
        # final render
        t = self._prerendered
        t = self.renderLinksIn(t)
        t = apply(self.addStandardLayoutTo,(t,),kw)
        return t

    security.declareProtected(Permissions.View, 'render_stxprelink')
    def render_stxprelink(self, client=None, REQUEST={},
                          RESPONSE=None, **kw):
        """
        render the page using structured text + wiki links, prelinking
        """
        # pre render
        t = str(self.read())
        if not self._prerendered:
            get_transaction().note('prerender')
            t = html_quote(t)
            t = self.applyLineEscapesIn(t)
            t = self.stxToHtml(t)
            self._prerendered = t or '\n'
            self._preLink() 
        if kw.get('pre_only',0): return
        # final render
        self._renderWithLinks()
        t = self._prerendered
        t = self._renderLateLinkTitles(t)
        t = apply(self.addStandardLayoutTo,(t,),kw)
        return t

    security.declareProtected(Permissions.View, 'render_stxlink')
    def render_stxlink(self, client=None, REQUEST={},
                       RESPONSE=None, **kw):
        """
        render the page using structured text + wiki links
        """
        # pre render
        t = str(self.read())
        if not self._prerendered:
            get_transaction().note('prerender')
            t = html_quote(t)
            t = self.applyLineEscapesIn(t)
            t = self.stxToHtml(t)
            self._prerendered = t or '\n'
        if kw.get('pre_only',0): return
        # final render
        t = self._prerendered
        t = self.renderLinksIn(t)
        t = apply(self.addStandardLayoutTo,(t,),kw)
        return t

    security.declareProtected(Permissions.View, 'render_prelinkdtmlhtml')
    def render_prelinkdtmlhtml(self, client=None, REQUEST={},
                               RESPONSE=None, **kw):
        """
        render the page using DTML + wiki links + HTML, prelinking
        """
        # pre render
        t = str(self.read())
        if not self._prerendered:
            get_transaction().note('prerender')
            t = self.applyLineEscapesIn(t)
            self._prerendered = t or '\n'
            self.cook()
            self._preLink()
        if kw.get('pre_only',0): return
        # final render
        self._renderWithLinks()
        t = apply(DTMLDocument.__call__,(self, client, REQUEST, RESPONSE), kw)
        t = self._renderLateLinkTitles(t)
        t = apply(self.addStandardLayoutTo,(t,),kw)
        return t

    security.declareProtected(Permissions.View, 'render_dtmllinkhtml')
    def render_dtmllinkhtml(self, client=None, REQUEST={},
                            RESPONSE=None, **kw):
        """
        render the page using DTML + wiki links + HTML
        """
        # pre render
        t = str(self.read())
        if not self._prerendered:
            get_transaction().note('prerender')
            t = self.applyLineEscapesIn(t)
            self._prerendered = t or '\n'
            self.cook()
        if kw.get('pre_only',0): return
        # final render
        t = apply(DTMLDocument.__call__,(self, client, REQUEST, RESPONSE), kw)
        t = self.renderLinksIn(t)
        t = apply(self.addStandardLayoutTo,(t,),kw)
        return t

    security.declareProtected(Permissions.View, 'render_prelinkhtml')
    def render_prelinkhtml(self, client=None, REQUEST={}, RESPONSE=None, **kw):
        """
        render the page using wiki links + HTML, prelinking
        """
        # pre render
        t = str(self.read())
        if not self._prerendered:
            get_transaction().note('prerender')
            t = self.applyLineEscapesIn(t)
            self._prerendered = t or '\n'
            self._preLink()
        if kw.get('pre_only',0): return
        # final render
        self._renderWithLinks()
        t = self._prerendered
        t = self._renderLateLinkTitles(t)
        t = apply(self.addStandardLayoutTo,(t,),kw)
        return t

    security.declareProtected(Permissions.View, 'render_linkhtml')
    def render_linkhtml(self, client=None, REQUEST={}, RESPONSE=None, **kw):
        """
        render the page using wiki links + HTML
        """
        # pre render
        t = str(self.read())
        if not self._prerendered:
            get_transaction().note('prerender')
            t = self.applyLineEscapesIn(t)
            self._prerendered = t or '\n'
        if kw.get('pre_only',0): return
        # final render
        t = self._prerendered
        t = self.renderLinksIn(t)
        t = apply(self.addStandardLayoutTo,(t,),kw)
        return t

    security.declareProtected(Permissions.View, 'render_dtmlhtml')
    def render_dtmlhtml(self, client=None, REQUEST={}, RESPONSE=None, **kw):
        """
        render the page using DTML + HTML, nothing else
        """
        # pre render
        t = str(self.read())
        if not self._prerendered:
            get_transaction().note('prerender')
            self._prerendered = t or '\n'
            self.cook()
        if kw.get('pre_only',0): return
        # final render
        t = apply(DTMLDocument.__call__,(self, client, REQUEST, RESPONSE), kw)
        t = apply(self.addStandardLayoutTo,(t,),kw)
        return t

    security.declareProtected(Permissions.View, 'render_html')
    def render_html(self, client=None, REQUEST={}, RESPONSE=None, **kw):
        """
        render the page using HTML with no surprises
        """
        # pre render
        t = str(self.read())
        if not self._prerendered:
            get_transaction().note('prerender')
            self._prerendered = t or '\n'
        if kw.get('pre_only',0): return
        # final render
        t = self._prerendered
        t = apply(self.addStandardLayoutTo,(t,),kw)
        return t

    security.declareProtected(Permissions.View, 'render_wwmlprelink')
    def render_wwmlprelink(self, client=None, REQUEST={},
                           RESPONSE=None, **kw):
        """
        render the page with WikiWikiWeb formatting + (z)wiki links, prelinking
        """
        # pre render
        t = str(self.read())
        if not self._prerendered:
            get_transaction().note('prerender')
            t = html_quote(t)
            t = self.applyLineEscapesIn(t)
            t = translate_WMML(t)
            self._prerendered = t or '\n'
            self._preLink()
        if kw.get('pre_only',0): return
        # final render
        self._renderWithLinks()
        t = self._prerendered
        t = self._renderLateLinkTitles(t)
        t = apply(self.addStandardLayoutTo,(t,),kw)
        return t

    security.declareProtected(Permissions.View, 'render_wwmllink')
    def render_wwmllink(self, client=None, REQUEST={}, RESPONSE=None, **kw):
        """
        render the page using WikiWikiWeb formatting + (z)wiki links
        """
        # pre render
        t = str(self.read())
        if not self._prerendered:
            get_transaction().note('prerender')
            t = html_quote(t)
            t = self.applyLineEscapesIn(t)
            t = translate_WMML(t)
            self._prerendered = t or '\n'
        if kw.get('pre_only',0): return
        # final render
        t = self._prerendered
        t = self.renderLinksIn(t)
        t = apply(self.addStandardLayoutTo,(t,),kw)
        return t

    security.declareProtected(Permissions.View, 'render_textlink')
    def render_textlink(self, client=None, REQUEST={}, RESPONSE=None, **kw):
        """
        render the page using fixed-width plain text plus zwiki links
        """
        # pre render
        t = str(self.read())
        if not self._prerendered:
            get_transaction().note('prerender')
            t = html_quote(t)
            self._prerendered = t or '\n'
        if kw.get('pre_only',0): return
        # final render
        t = self._prerendered
        t = self.renderLinksIn(t)
        t = "<pre>\n" + t + "\n</pre>\n"
        t = apply(self.addStandardLayoutTo,(t,),kw)
        return t

    security.declareProtected(Permissions.View, 'render_plaintext')
    def render_plaintext(self, client=None, REQUEST={}, RESPONSE=None, **kw):
        """
        render the page using fixed-width plain text with no surprises
        """
        # pre render
        t = str(self.read())
        if not self._prerendered:
            get_transaction().note('prerender')
            t = "<pre>\n" + html_quote(t) + "\n</pre>\n"
            self._prerendered = t or '\n'
        if kw.get('pre_only',0): return
        # final render
        t = self._prerendered
        t = apply(self.addStandardLayoutTo,(t,),kw)
        return t

    security.declareProtected(Permissions.View, 'stxToHtml')
    def stxToHtml(self, text):
        """
        Render some structured text into html, with our customizations.
        """
        text = str(text)        
        if ZOPEVERSION < (2,4):
            # final single-line paragraph becomes a heading if there are
            # trailing blank lines - strip them
            text = re.sub(r'(?m)\n[\n\s]*$', r'\n', text)

        # an initial single word plus period becomes a numeric bullet -
        # prepend a temporary marker to prevent
        # XXX use locale/wikichars from Regexps.py instead of A-z
        text = re.sub(r'(?m)^([ \t]*)([A-z]\w*\.)',
                      r'\1<!--NOSTX-->\2',
                      text)

        # :: quoting fails if there is whitespace after the :: - remove it
        text = re.sub(r'(?m)::[ \t]+$', r'::', text)

        # suppress stx footnote handling so we can do it our way
        text = re.sub(footnoteexpr,r'<a name="ref\1">![\1]</a>',text)
        text = re.sub(r'(?m)\[',r'[<!--NOSTX-->',text)

        # generate html
        if ZOPEVERSION < (2,4):
            text = str(StructuredText.HTML(text,level=2))
        else:
            text = StructuredText.HTMLNG(
                myDocument(StructuredText.Basic(str(text))),
                level=2)

        # clean up
        text = re.sub(r'(<|&lt;)!--NOSTX--(>|&gt;)', r'', text)

        # strip html & body added by some zope versions
        text = re.sub(
            r'(?sm)^<html.*<body.*?>\n(.*)</body>\n</html>\n',r'\1',text)

        return text

    def _renderLateLinkTitles(self,text):
        """
        fill in titles for any untitled wiki links in text
    
        In pre-linking modes, we do this in a final pass after linking &
        dtml so that we can have fast-changing link titles without messing
        up our caching. See _renderWithLinks for more.
        #    The regexp needs to match _renderLink's link format.  We can
        assume the link will be all on one line and that there's no raw
        dtml lying around.
        """
        return re.sub(untitledwikilinkexpr,self._replaceUntitledWikilink,text)

    def _replaceUntitledWikilink(self, match):
        """
        Replace a untitledwikilink occurence with a suitably titled link.
        """
        link = match.group()
        pagename = match.group('page')
        # old wiki page urls may need un-quoting to get the id ?
        page = self.pageWithName(unquote(pagename))
        if page:
            linktitle = page.linkTitle()
        else:
            # this shouldn't normally happen now
            linktitle = ""
        newlink = re.sub(r' title="">', r' title="%s">' % (linktitle), link)
        return newlink

    security.declareProtected(Permissions.View, 'supportsStx')
    def supportsStx(self):
        """does this page do Structured Text formatting ?"""
        return re.search(r'(?i)(structuredtext|stx)',
                         self.page_type) is not None

    security.declareProtected(Permissions.View, 'supportsWiki')
    def supportsWiki(self):
        """does this page do wiki linking ?"""
        return re.search(r'(?i)plain',self.page_type) is None

    security.declareProtected(Permissions.View, 'supportsHtml')
    def supportsHtml(self):
        """does this page render embedded HTML ?"""
        return not self.page_type in \
               ['structuredtextonly','classicwiki','plaintext']

    security.declareProtected(Permissions.View, 'supportsDtml')
    def supportsDtml(self):
        """does this page support embedded DTML ?"""
        return re.search(r'(?i)dtml',self.page_type) is not None

    security.declareProtected(Permissions.View, 'hasDynamicContent')
    def hasDynamicContent(self):
        """does this page contain dynamic content ?"""
        return (self.supportsDtml() and
                re.search(r'(?i)(<!--|<dtml|&dtml)',self.read()) is not None)

    security.declareProtected(Permissions.View, 'supportsPreLinking')
    def supportsPreLinking(self):
        """does this page do prelinking ?"""
        return re.search(r'prelink',self.page_type) is not None

    # from WFN - not used yet, may come in handy
    security.declareProtected(Permissions.View, 'prep_citation')
    def prep_citation(self, rfind=string.rfind, strip=string.strip):
        """Quote text for use in literal citations.

        We prepend '>' to each line, splitting long lines (propagating
        existing citation and leading whitespace) when necessary.
        """
        got = []
        for line in string.split(self._st_data or self.xread(), '\n'):
            pref = '> '
            if len(line) < 79:
                got.append(pref + line)
                continue
            m = cite_prefixexp.match(line)
            if m is None:
                pref = '> %s'
            else:
                if m.group(1):
                    pref = pref + m.group(1)
                    line = line[m.end(1)+1:]
                    if m.end(1) > 60:
                        # Too deep quoting - collapse it:
                        pref = '> >> '
                        lencut = 0
                pref = pref + '%s'
                leading_space = m.group(2)
                if leading_space:
                    pref = pref + leading_space
                    line = line[len(leading_space):]
            lenpref = len(pref)
            continuation_padding = ''
            lastcurlen = 0
            while 1:
                curlen = len(line) + lenpref
                if curlen < 79 or (lastcurlen and lastcurlen <= curlen):
                    # Small enough - we're done - or not shrinking - bail out
                    if line: got.append((pref % continuation_padding) + line)
                    break
                else:
                    lastcurlen = curlen
                splitpoint = max(rfind(line[:78-lenpref], ' '),
                                 rfind(line[:78-lenpref], '\t'))
                if not splitpoint or splitpoint == -1:
                    if strip(line):
                        got.append((pref % continuation_padding) +
                                   line)
                    line = ''
                else:
                    if strip(line[:splitpoint]):
                        got.append((pref % continuation_padding) +
                                   line[:splitpoint])
                    line = line[splitpoint+1:]
                if not continuation_padding:
                    # Continuation lines are indented more than intial - just
                    # enough to line up past, eg, simple bullets.
                    continuation_padding = '  '
        return string.join(got, '\n')

    ######################################################################
    # linking & link rendering

    # Two ways to get stuff linked:
    #
    # 1. link arbitrary text
    # renderLinksIn(text)
    #  replaces links in text
    #   calls _renderLink for each
    #
    # 2. link this page's text, with caching
    # _preLink() - called at edit time
    #  searches for valid links in self._prerendered
    #  caches them with text regions in self._prelinked (also self._links)
    # _renderWithLinks() - called at view time
    #  calls _renderedLinks()
    #   calls _renderLink for each cached link
    #  interpolates rendered links with cached text regions
    #  saves final text back in self._prerendered (dtml: re-cook if changed)

    security.declareProtected(Permissions.View, 'wikilink')
    def wikilink(self, text):
        """
        api utility method for wiki-linking an arbitrary text snippet
        """
        return self.renderLinksIn(text,doLineEscapes=1)

    security.declareProtected(Permissions.View, 'applyLineEscapesIn')
    def applyLineEscapesIn(self, text):
        """
        implement wikilink-escaping in lines in text which begin with !
        """
        return re.sub(protected_line, self._protectLine, text)
        
    def _protectLine(self, match):
        """
        return the string represented by match with all it's wikilinks escaped
        """
        return re.sub(wikilink, r'!\1', match.group(1))

    def _replaceInterWikilink(self, match, allowed=0, state=None, text='',
                              protect=1):
        """
        Replace an occurrence of interwikilink with a suitable hyperlink.

        To be used as a re.sub repl function *and* get a proper value
        for literal context, 'allowed', etc, enclose this function
        with the value using 'thunk_substituter'.
        """
        # matches beginning with ! should be left alone
        #if re.match('^!',match.group(0)): return match.group(1)
        # NB this is a bit naughty, but: since we know this text will likely
        # be scanned for ordinary wiki links right after this pass, leave
        # the ! in place for it to find. Otherwise the localname will
        # get wiki-linked.
        if re.match('^!',match.group(0)): return match.group(0)

        localname  = match.group('local')
        remotename = match.group('remote')

        # NB localname could be [bracketed]
        if re.match(bracketedexpr,localname):
            localname = re.sub(bracketedexpr, r'\1', localname)

        # look for a RemoteWikiURL definition
        if hasattr(self.folder(), localname): 
            localpage = getattr(self.folder(),localname)
            # local page found - search for "RemoteWikiUrl: url"
            m = re.search(remotewikiurl, str(localpage))
            if m is not None:
                remoteurl = html_unquote(m.group(1)) # NB: pages are 
                                                     # stored html-quoted
                                                     # XXX eh ? they are ?
                                                     # something's not
                                                     # right
                                                     # somewhere.. 
                                                     # I have lost my
                                                     # grip on this
                                                     # whole quoting
                                                     # issue.
                
                # we have a valid inter-wiki link
                link = '<a href="%s%s">%s:%s</a>' % \
                       (remoteurl, remotename, localname, remotename)
                if protect:
                    # normally, need to protect it from later wiki-izing passes
                    link = re.sub(wikilink, r'!\1', link)
                return link

        # otherwise, leave alone
        return match.group(0)

    def _preLink(self):
        """
        find links in _prerendered & prepare for a fast _renderWithLinks

        saves a list of alternating text extents and link names in
        _prelinked and the unique link names in _links
        """
        get_transaction().note('prelink')
        text = self._prerendered or ''
        #XXX regexp alert - possibly slower than before with large pages
        r = anywikilinkexpr
        self._links = []
        self._prelinked = []
        lastpos = textstart = 0
        state = {'lastend':0,'inpre':0,'incode':0,'intag':0,'inanchor':0}
        while 1:
            m = r.search(text,lastpos)
            if m:
                link = m.group()
                linkstart,linkend = m.span()
                #if (not within_literal(m.start(1),m.end(1)-1,
                if (not within_literal(linkstart,linkend-1,
                                       state,text) and
                    #XXX more tests; easier than adapting within_literal
                    not withinSgmlOrDtml(m.span(),text) and
                    not re.match('^!',link)):
                    # found a link - save it and the text extent before it
                    self._links.append(link)
                    self._prelinked.append(text[textstart:linkstart])
                    self._prelinked.append(text[linkstart:linkend])
                    lastpos = textstart = linkend
                else:
                    # found the link pattern but it's escaped or inside a
                    # stx literal or sgml tag
                    # leave it as part of the ordinary text
                    # and strip the ! from escaped links
                    if re.match('^!',link):
                        text = text[:linkstart] + text[linkstart+1:]
                        linkend = linkend - 1
                    lastpos = linkend
            else:
                # no more links - save the final text extent & quit
                self._prelinked.append(text[textstart:])
                break

    def _renderWithLinks(self):
        """
        Regenerate this page's pre-rendered data with up-to-date links.

        Called on each view for pre-linking page types.  Assumes _preLink
        was called at edit time to cache this page's links.  Renders all
        cached links based on latest wiki contents and interpolates these
        with the static text extents previously identified. If this
        rendering has changed since last time, save it in _prerendered
        (also re-cook dtml pages, so dtml knows what's where).

        premature optimisation:

        This method is expensive for large pages (due to _renderedLinks()).
        (True ? 10-20s on zwiki.org with 90K page, 1400 pages. Because
        it's called on each view this is significant.)
        We don't know if the result will be different from what's already
        in self._prerendered, we just do it and find out.  We could try to
        predict this. Assuming the source text is unchanged, the
        prerendered text is dependent on what ?

        1. creation of wiki pages we link to
        2. deletion of wiki pages we link to 
        3. renaming of wiki pages we link to 
        4. edits to wiki pages we link to (because of info in link titles)

        On the theory that 4 is relatively frequent and the link titles
        are cheap to add at view time, we currently exclude them from the
        prelinking process and fill them in at the last moment with
        _renderLateLinkTitles.

        For 1/2/3 we could note the link targets existing and not existing
        when we render and check these next time.

        The result would be: large pages often but not always render
        faster. Better if we could just make stx formatting and link
        rendering fast, or turn off freeform names.
        
        """
        if not self._prelinked:
            return
        renderedlinks = self._renderedLinks()
        t = ''
        prelinked = self._prelinked
        for i in range(0,len(prelinked)-1,2):
            text,link = prelinked[i],prelinked[i+1]
            t = t + text
            t = t + renderedlinks[link]
        t = t + self._prelinked[-1]
        if t != self._prerendered:
            get_transaction().note('renderwithlinks')
            self._prerendered = t
            if self.supportsDtml():
                self.cook()

    def _renderedLinks(self):
        """
        return a mapping of our cached links to their current renderings
        """
        rendered = {}
        for link in self._links:
            rendered[link] = self._renderLink(link,render_title=0)
        return rendered

    def _renderLink(self,link,allowed=0,state=None,text='',render_title=1):
        """
        render a link appropriately depending on the current page/wiki context

        Can be called directly (link should be a string)
        or from re.sub via thunk substituter (link will be a match object).
        """
        if type(link) is not StringType:
            # we are being called from re.sub via thunk_substituter
            # do the within_literal and other checks that _preLink would
            # normally do
            match = link
            link = match.group()
            if (within_literal(match.start(),match.end()-1,state,text) or
                withinSgmlOrDtml(match.span(),text) or
                re.match('^!',link)):
                if re.match('^!',link):
                    link = link[1:]
                return link
        else:
            text = getattr(self,'_prerendered','') or ''

        folder = self.folder()
        m = re.match(interwikilink,link)
        if m:
            return self._replaceInterWikilink(m,protect=0)
        else:
            m = morig = link

        # if it's a bracketed phrase,
        if re.match(bracketedexpr,m):

            # strip the enclosing []'s
            m = re.sub(bracketedexpr, r'\1', m)

            # look for a page with a name matching this
            pagewiththisname = self.pageWithFuzzyName(m,ignore_case=1)
            if pagewiththisname:
                m = pagewiththisname.getId()
                # and fall through to normal link processing
                # XXX but m might get hassled by the next two if clauses

            else:

                # XXX old stuff - may restrict [] to wiki pages later

                # extract a (non-url) path if there is one
                # XXX could also be a [] link containing /
                # try it as a path & link if the thing exists, 
                # otherwise treat this as a [] link  ?
                # or leave out path linking altogether ?
                # yes, let's simplify []'s functionality for the moment
                #pathmatch = re.match(r'(([^/]*/)+)([^/]+)',m)
                #if pathmatch:
                #    path,id = pathmatch.group(1), pathmatch.group(3)
                #else:
                #    path,id = '',m

                # if it looks like an image link, inline it
                #if guess_content_type(id)[0][0:5] == 'image':
                #    return '<img src="%s%s">' % (path,id)

                # or if there was a path assume it's to some non-wiki
                # object and skip the usual existence checking for
                # simplicity. Could also attempt to navigate the path in
                # zodb to learn more about the destination
                #if path:
                #    #return '<a href="%s%s">%s</a>' % (path,id,id)
                #    return '<a href="%s%s">%s%s</a>' % (path,id,path,id)

                # otherwise fall through to normal link processing
                pass
        
        # if it's an ordinary url, link to it
        if re.match(url,m):
            # except, if preceded by " or = it should probably be left alone
            if re.match('^["=]',m):                                   # "
                return m
            else:
                return '<a href="%s">%s</a>' % (m, m)

        # it might be a structured text footnote ?
        # try to avoid matching accidentally
        # XXX needs to be in docs
        # XXX stx uses ref prefix now.. from what version ? what about older
        # zopes ?
        elif (morig[-1] == ']' and
              re.search(r'(?s)<a name="ref%s"' % (re.escape(m)),text)):
            return '<a href="#ref%s" title="footnote %s">[%s]</a>' % (m,m,m)

        # a wikiname - if a page of this name exists in
        # this folder, link to it
        #elif (hasattr(folder.aq_base, m) and
        #      self.isZwikiPage(getattr(folder.aq_base,m))):
        # XXX new: if international characters have been enabled for wiki
        # names but not page ids (see Regexps.py), wikinames too may
        # differ from their page id.  Are we sure we want to allow this ?
        # check quick case first.. premature optimization, all this will
        # be refactored
        elif self.pageWithNameOrId(m):
            #return '<a href="%s">%s</a>' % (quote(m), m)
            # wiki links were made absolute for robustness in editform etc.
            # a relative_urls property can re-enable the old behaviour
            if not hasattr(folder.aq_base,m): m = self.pageWithNameOrId(m).getId()
            morig = re.sub(r'\[(.*)\]',r'\1',morig)
            if render_title:
                linktitle = folder[m].linkTitle()
            else:
                linktitle = ''
            if getattr(folder,'relative_urls',None):
                return '<a href="%s" title="%s">%s</a>' % \
                       (quote(m),linktitle,morig)
            else:
                return '<a href="%s/%s" title="%s">%s</a>' % \
                       (self.wiki_url(),quote(m),linktitle,morig)

        # if it's acquired from above, link it differently to facilitate
        # sub-wikis (one level).
        # XXX NB pages whose wikiname and id differ as described above
        # (accented pages, freeform pages in parent folder) won't be linked.
        elif (hasattr(folder,'aq_parent') and
              hasattr(folder.aq_parent, m) and
              self.isZwikiPage(getattr(folder.aq_parent,m))):
            if render_title:
                linktitle = getattr(folder.aq_parent,m).linkTitle()
            else:
                linktitle = ''
            if getattr(folder,'relative_urls',None):
                return \
                  '<a href="../%s" title="page in parent wiki, %s">../%s</a>' \
                  % (quote(m),linktitle,morig)
            else:
                return \
                  '<a href="%s/../%s" title="page in parent wiki, %s">../%s</a>'\
                  % (self.wiki_url(),quote(m),linktitle,morig)

        # otherwise, provide a "?" creation link
        else:
            if getattr(folder,'relative_urls',None):
                return '%s<a class="new" href="%s/editform?page=%s" title="create this page">?</a>' % \
                       (morig, quote(self.id()), quote(m))
            else:
                return '%s<a class="new" href="%s/%s/editform?page=%s" title="create this page">?</a>' % \
                       (morig, self.wiki_url(), quote(self.id()), quote(m))

    security.declareProtected(Permissions.View, 'renderLinksIn')
    def renderLinksIn(self,text,doLineEscapes=0): #XXX default to 1 ?
        """
        Return text with all links rendered as for the context of this page.
        """
        t = text
        if doLineEscapes:
            t = self.applyLineEscapesIn(t)
        t = re.sub(anywikilinkexpr,
                   thunk_substituter(self._renderLink, t, 1),
                   t)
        return t

    security.declareProtected(Permissions.View, 'links')
    def links(self):
        """
        list the links occurring on this page - useful for cataloging

        includes urls & interwiki links
        """
        # Make sure the _links cache is up to date.
        # It's ok to call _preLink even for pages which don't do
        # pre-linking. It expects pre-rendering to have been called first.
        if not self._prerendered:
            self._preRender()
        if not self._prelinked:
            self._preLink()
        return self._links

    security.declareProtected(Permissions.View, 'canonicalLinks')
    def canonicalLinks(self):
        """
        list the canonical id form of the wiki links occurring on this page

        useful for calculating backlinks.
        """
        clinks = []
        localwikilinkexpr = re.compile(localwikilink)
        for link in self.links():
            if localwikilinkexpr.match(link):
                if link[0] == r'[' and link[-1] == r']':
                    link = link[1:-1]
                clink = self.canonicalIdFrom(link)
                clinks.append(clink)
        return clinks

    ######################################################################
    # page naming and lookup

    security.declareProtected(Permissions.View, 'canonicalIdFrom')
    def canonicalIdFrom(self,name):
        """
        Convert a free-form page name to a canonical url- and zope-safe id.

        Constraints for zwiki page ids:
        - it needs to be a legal zope object id
        - to simplify linking, we will require it to be a valid url
        - it should be unique for a given name (ignoring whitespace)
        - we'd like it to be as similar to the name and as simple to read
          and work with as possible
        - we'd like to encourage serendipitous linking between free-form
          and wikiname links & pages

        So this version
        - discards non-word-separating punctuation (')
        - converts remaining punctuation to spaces
        - capitalizes and joins whitespace-separated words into a wikiname
        - converts any non-zope-and-url-safe characters and _ to _hexvalue
        - if the above results in an id beginning with _, prepends X
          (XXX this breaks the uniqueness requirement, better ideas ?)

        performance-sensitive
        """
        # remove punctuation, preserving word boundaries.
        # ' is not considered a word boundary.
        name = re.sub(r"'",r"",name)
        name = re.sub(r'[%s]+'%re.escape(string.punctuation),r' ',name)
        
        # capitalize whitespace-separated words (preserving existing
        # capitals) then strip whitespace
        id = ' '+name
        id = spaceandlowerexpr.sub(lambda m:string.upper(m.group(1)),id)
        id = string.join(string.split(id),'')

        # quote any remaining unsafe characters (international chars)
        safeid = ''
        for c in id:
            if zwikiidcharsexpr.match(c):
                safeid = safeid + c
            else:
                safeid = safeid + '_%02x' % ord(c)

        # zope ids may not begin with _
        if len(safeid) > 0 and safeid[0] == '_':
            safeid = 'X'+safeid
        return safeid

    security.declareProtected(Permissions.View, 'canonicalId')
    def canonicalId(self):
        """
        Give the canonical id of this page.
        """
        return self.canonicalIdFrom(self.title_or_id())

    security.declareProtected(Permissions.View, 'pages')
    def pages(self):
        """
        Return a list of all pages in this wiki.
        """
        return self.folder().objectValues(spec=self.meta_type)

    security.declareProtected(Permissions.View, 'pageIds')
    def pageIds(self):
        """
        Return a list of all page ids in this wiki.
        """
        return self.folder().objectIds(spec=self.meta_type)

    security.declareProtected(Permissions.View, 'pageNames')
    def pageNames(self):
        """
        Return a list of all page names in this wiki.
        """
        return map(lambda x:x.title_or_id(),self.pages())

    security.declareProtected(Permissions.View, 'pageIdsStartingWith')
    def pageIdsStartingWith(self,text):
        #from __future__ import nested_scopes
        #return filter(lambda x:x[:len(text)]==text,self.pageIds())
        ids = []
        for i in self.pageIds():
            if i[:len(text)] == text:
                ids.append(i)
        return ids

    security.declareProtected(Permissions.View, 'pageNamesStartingWith')
    def pageNamesStartingWith(self,text):
        #from __future__ import nested_scopes
        #return filter(lambda x:x[:len(text)]==text,self.pageNames())
        names = []
        for n in self.pageNames():
            if n[:len(text)] == text:
                names.append(n)
        return names

    security.declareProtected(Permissions.View, 'firstPageIdStartingWith')
    def firstPageIdStartingWith(self,text):
        return (self.pageIdsStartingWith(text) or [None])[0]

    security.declareProtected(Permissions.View, 'firstPageNameStartingWith')
    def firstPageNameStartingWith(self,text):
        return (self.pageNamesStartingWith(text) or [None])[0]

    security.declareProtected(Permissions.View, 'pageWithId')
    def pageWithId(self,id,url_quoted=0,ignore_case=0):
        """
        Return the page in this folder which has this id, or None.

        Can also do a case-insensitive id search,
        and optionally unquote id.
        """
        # we don't want to acquire, but be careful to return right
        # acquisition context..
        # getattr(f.aq_base.id).__of__(f) isn't good enough
        # (see getPhysicalPath)
        if url_quoted:
            id = unquote(id)
        f = self.folder()
        if hasattr(f.aq_base,id) and self.isZwikiPage(f[id]):
            return f[id]
        elif ignore_case:
            id = string.lower(id)
            # use catalog ?
            for i in self.pageIds():
                p = f[i]
                if id == string.lower(p.id()):
                    return p
        else:
            return None

    security.declareProtected(Permissions.View, 'pageWithName')
    def pageWithName(self,name,url_quoted=0):
        """
        Return the page in this folder which has this name, or None.

        page name may be different from page id, and if so is stored in
        the title property. Ie page name is currently defined as
        the value given by title_or_id().

        As of 0.17, page ids and names always follow the invariant
        id == canonicalIdFrom(name).
        """
        return (self.pageWithId(self.canonicalIdFrom(name),url_quoted))

    security.declareProtected(Permissions.View, 'pageWithNameOrId')
    def pageWithNameOrId(self,name,url_quoted=0):
        """
        Return the page in this folder with this as it's name or id, or None.
        """
        return (self.pageWithId(name,url_quoted) or 
                self.pageWithName(name,url_quoted))
        
    security.declareProtected(Permissions.View, 'pageWithFuzzyName')
    def pageWithFuzzyName(self,name,url_quoted=0,
                           allow_partial=0,ignore_case=1):
        """
        Return the page in this folder for which name is a fuzzy link, or None.

        A fuzzy link ignores whitespace, capitalization & punctuation.  If
        there are multiple fuzzy matches, return the page whose name is
        alphabetically first.  The allow_partial flag allows even fuzzier
        matching. As of 0.17 ignore_case is not used and kept only for
        backward compatibility.

        performance-sensitive
        """
        if url_quoted:
            name = unquote(name)
        p = self.pageWithName(name)
        if p: return p
        id = self.canonicalIdFrom(name)
        idlower = string.lower(id)
        ids = self.pageIds()
        ids.sort()
        for i in ids:
            ilower = string.lower(i)
            if (ilower == idlower or 
                (allow_partial and ilower[:len(idlower)] == idlower)):
                return self.pageWithId(i)
        return None
        
    ######################################################################
    # page editing/creation/deletion/file upload methods

    def _checkPermission(self, permission, object):
        return getSecurityManager().checkPermission(permission,object)

    security.declareProtected(Permissions.View, 'create') 
    # really Permissions.Add, but keep our informative unauthorized message
    def create(self,page,text=None,type=None,title='',REQUEST=None,log=''):
        #XXX title argument no longer used, remove
        """
        Create a new wiki page and redirect there if appropriate; can
        upload a file at the same time.  Normally edit() will call
        this for you.

        Assumes page has been url-quoted. If it's not a url-safe name, we
        will create the page with a url-safe id that's similar. We assume
        this id won't match anything already existing (zwiki would have
        linked it instead of offering to create it).
        """
        # do we have permission ?
        if not self._checkPermission(Permissions.Add,self.folder()):
            raise 'Unauthorized', (
                _('You are not authorized to add ZWiki Pages here.'))

        name = unquote(page)
        id = self.canonicalIdFrom(name)

        # make a new (blank) page object, situate it
        # in the parent folder and get a hold of it's
        # normal acquisition wrapper
        # newid should be the same as id, but don't assume
        p = ZWikiPage(source_string='', __name__=id)
        newid = self.folder()._setObject(id,p)
        p = getattr(self.folder(),newid)

        p.title = name
        p._setCreator(REQUEST)
        p._setLastEditor(REQUEST)
        p._setLastLog(log)
        p._setOwnership(REQUEST)
        p.parents = [self.title_or_id()]

        # inherit type from the previous (parent) page if not specified
        # or overridden via folder attribute,
        # special case: don't inherit issuedtml page type
        # XXX standard_page_type prevents creating issue pages.
        # What to do ? For now nothing, assume the tracker add form
        # has permission to change the page type afterward
        if hasattr(self.folder(),'standard_page_type'):
            p.page_type = self.folder().standard_page_type
        elif type:
            p.page_type = type
        elif self.page_type != 'issuedtml':
            p.page_type = self.page_type
        else:
            p.page_type = DEFAULT_PAGE_TYPE

        # set initial page text as edit() would, with cleanups and dtml
        # validation
        p._setText(text or '',REQUEST)

        # if a file was submitted as well, handle that
        p._handleFileUpload(REQUEST)

        # we got indexed after _setObject,
        # but do it again with our text in place
        p.index_object()
        
        if p.usingRegulations():
            # initialize regulations settings.
            p.setRegulations(REQUEST,new=1)

        # generate a mailout
        p.sendMailToSubscribers(
            p.read(),
            REQUEST=REQUEST,
            subjectSuffix='',
            subject='(new) '+log)

        # redirect browser if needed
        if REQUEST is not None:
            try:
                u=REQUEST['URL2']
                REQUEST.RESPONSE.redirect(u + '/' + quote(newid))
            except KeyError:
                pass

    security.declareProtected(Permissions.Comment, 'comment')
    def comment(self, text='', username='', time='', note='',
                use_heading=0, REQUEST=None, subject_heading='',
                do_mailout=1):
        """
        A handy method, like append but

        - adds a standard comment heading with user name, time & note

        - allows those fields to be specified for some flexibility
        with mail-ins etc

        - does some html prettification for web display; nb other
        parts try to strip this same html (sendMailSubscriber)

        As a convenience for standard_wiki_footer, unless use_heading
        is true we act exactly like append. Should think of something
        better here.

        The subject_heading argument is so named to avoid a clash with
        some existing zope/cmf subject attribute.
        """
        # gather various bits and pieces
        subject = subject_heading # "subject" clashes with some zope attribute
        if not username:
            username = self.zwiki_username_or_ip(REQUEST)
            if re.match(r'^[0-9\.\s]*$',username): 
                username = ''
        if not time:
            time = self.ZopeTime().strftime('%Y/%m/%d %H:%M %Z')
        oldtext = self.read()

        # add comment to page
        # testing support - process but don't add comments with subject
        # [test] (except on TestPage)
        if subject != '[test]' or self.getId() == 'TestPage':
            # italicize citations
            formattedtext = re.sub(r'(?m)^>(.*)',r'<br>><i>\1</i>',text)
            if use_heading or subject:
                # generate comment heading
                # XXX should be made configurable
                heading = '\n\n'
                if self.inCMF(): heading = heading + \
                  '<img src="discussionitem_icon.gif" style="border:none; margin:0"/>'
                heading = heading + '<b>%s</b> --' % (subject or '...')
                if username: heading = heading + '%s, ' % (username)
                heading = heading + html_quote(time)
                heading = heading + '<br>\n'
            else:
                heading = '\n\n'
            self.append(formattedtext,heading,REQUEST=REQUEST,log=subject)

        # if mailout policy is comments only, send it now
        # (otherwise append has done it)
        if ((not hasattr(self.folder(),'mailout_policy')
             or self.folder().mailout_policy == 'comments') and
            (text or subject) and
            do_mailout):
            # get the username in there
            if REQUEST: REQUEST.cookies['zwiki_username'] = username
            self.sendMailToSubscribers(
                #self.textDiff(a=oldtext,b=self.read(),verbose=0),
                #may as well use the original !
                text, 
                REQUEST,
                "",
                subject=subject)

    security.declareProtected(Permissions.View, 'subjectAndTextFrom')
    def subjectAndTextFrom(self, text):
        """
        Separate an inline subject from a web comment.

        An inline subject is a short, bold, single-line initial paragraph.
        This allows a subject to be entered even when there is no separate
        field for it. NB this also cleans up the text.

        Example:
        if subject == None:
            subject,text = self.subjectAndTextFrom(text)

        No longer used.
        """
        text = self._cleanupText(text)  # strip ^M's, etc
        m = re.search(r'(?si)^\s*(\*\*|<b>)(?P<subject>[^\n]+)(\*\*|</b>)\s*\n\n(?P<body>.*)',text)
        if m and len(m.group('subject')) < 100:
            return (m.group('subject'), m.group('body'))
        else:
            return ('', text)

    security.declareProtected(Permissions.Comment, 'append')
    def append(self, text='', separator='\n\n', REQUEST=None, log=''):
        """
        Appends some text to an existing zwiki page by calling edit;
        may result in mail notifications to subscribers.
        """
        oldtext = self.read()
        text = str(text)
        if text:
            # usability hack: scroll to bottom after adding a comment
            if REQUEST:
                REQUEST['URL1'] = REQUEST['URL1'] + '#bottom'
            self.edit(text=oldtext+separator+text, REQUEST=REQUEST,log=log)

    security.declarePublic('edit')      # check permissions at runtime
    def edit(self, page=None, text=None, type=None, title='', 
             timeStamp=None, REQUEST=None, 
             subjectSuffix='', log='', check_conflict=1): # temp
        """
        General-purpose method for editing & creating zwiki pages.
        Changes the text and/or markup type of this (or the specified)
        page, or creates the specified page (name or id allowed) if it
        does not exist.
        
        Other special features:

        - Usually called from a time-stamped web form; we use
        timeStamp to detect and warn when two people attempt to work
        on a page at the same time. This makes sense only if timeStamp
        came from an editform for the page we are actually changing.

        - The username (authenticated user or zwiki_username cookie)
        and ip address are saved in page's last_editor, last_editor_ip
        attributes if a change is made

        - If the text begins with "DeleteMe", move this page to the
        recycle_bin subfolder.

        - If file has been submitted in REQUEST, create a file or
        image object and link or inline it on the current page.

        - May also cause mail notifications to be sent to subscribers

        This code has become more complex to support late page
        creation, but the api should now be more general & powerful
        than it was.  Doing all this stuff in one method simplifies
        the layer above I think.
        """
        #self._validateProxy(REQUEST)   # XXX correct ? don't think so
                                        # do zwiki pages obey proxy roles ?

        if page: page = unquote(page)
        # are we changing this page ?
        if page is None:
            p = self
        # changing another page ?
        elif self.pageWithNameOrId(page):
            p = self.pageWithNameOrId(page)
        # or creating a new page
        else:
            return self.create(page,text,type,title,REQUEST,log)

        # ok, changing p. We may be doing several things here;
        # each of these handlers checks permissions and does the
        # necessary. Some of these can halt further processing.
        # todo: tie these in to mail notification, along with 
        # other changes like reparenting
        if check_conflict and self.checkEditConflict(timeStamp, REQUEST):
            return self.editConflictDialog()
        if check_conflict and hasattr(self,'wl_isLocked') and self.wl_isLocked():
            return self.davLockDialog()
        if p._handleDeleteMe(text,REQUEST,log):
            return
        p._handleEditPageType(type,REQUEST,log)
        p._handleEditText(text,REQUEST,subjectSuffix,log)
        p._handleFileUpload(REQUEST,log)
        if self.usingRegulations():
            p._handleSetRegulations(REQUEST)

        # update catalog if present
        try:
            p.index_object()
        except:
            # XXX show traceback in log
            DLOG('failed to index '+p.id())
            try:
                p.reindex_object()
            except:
                DLOG('failed to reindex '+p.id()+', giving up')
                pass

        # redirect browser if needed
        if REQUEST is not None:
            u=REQUEST.get('redirectURL',REQUEST['URL1'])
            REQUEST.RESPONSE.redirect(u)

    def _handleSetRegulations(self,REQUEST):
        if REQUEST.get('who_owns_subs',None) != None:
            # do we have permission ?
            if not self._checkPermission(Permissions.ChangeRegs,self):
                raise 'Unauthorized', (
                  _("You are not authorized to set this ZWiki Page's regulations."))
            self.setRegulations(REQUEST)
            self._preRender(clear_cache=1)
            #self._setLastEditor(REQUEST)

    def _handleEditPageType(self,type,REQUEST=None,log=''):
        # is the new page type valid and different ?
        if (type is not None and
            type != self.page_type):
            # do we have permission ?
            if not self._checkPermission(Permissions.ChangeType,self):
                raise 'Unauthorized', (
                    _("You are not authorized to change this ZWiki Page's type."))
            # change it
            self.page_type = type
            self._preRender(clear_cache=1)
            self._setLastEditor(REQUEST)
            self._setLastLog(log)

    def _setLastLog(self,log):
        """\
        Note log message, if provided.
        """
        if log and string.strip(log):
            log = string.strip(log)
            get_transaction().note('"%s"' % log)
            self.last_log = log
        else:
            self.last_log = ''


    def _handleEditText(self,text,REQUEST=None, subjectSuffix='', log=''):
        # is the new text valid and different ?
        if (text is not None and
            self._cleanupText(text) != self.read()):
            # do we have permission ?
            if (not
                (self._checkPermission(Permissions.Change, self) or
                 (self._checkPermission(Permissions.Append, self)
                  and find(self._cleanupText(text),self.read()) == 0))):
                raise 'Unauthorized', (
                    _('You are not authorized to edit this ZWiki Page.'))

            # change it
            oldtext = self.read()
            self._setText(text,REQUEST)
            self._setLastEditor(REQUEST)
            self._setLastLog(log)

            # if mailout policy is "all edits", do it here
            if (hasattr(self.folder(),'mailout_policy')
                and self.folder().mailout_policy == 'edits'):
                self.sendMailToSubscribers(
                    self.textDiff(a=oldtext,b=self.read()),
                    REQUEST=REQUEST,
                    subject=log)

    def _handleDeleteMe(self,text,REQUEST=None,log=''):
        if not text or not re.match('(?m)^DeleteMe', text):
            return 0
        if (not
            (self._checkPermission(Permissions.Change, self) or
             (self._checkPermission(Permissions.Append, self)
              and find(self._cleanupText(text),self.read()) == 0))):
            raise 'Unauthorized', (
                _('You are not authorized to edit this ZWiki Page.'))
        if not self._checkPermission(Permissions.Delete, self):
            raise 'Unauthorized', (
                _('You are not authorized to delete this ZWiki Page.'))
        self._setLastLog(log)
        self._recycle(REQUEST)

        if REQUEST:
            # redirect to first existing parent, or front page
            destpage = ''
            for p in self.parents:
                if hasattr(self.folder(),p):
                    destpage = p
                    break
            REQUEST.RESPONSE.redirect(self.wiki_url()+'/'+quote(destpage))
            # I used to think redirect did not return, guess I was wrong

        # return true to terminate edit processing
        return 1

    security.declareProtected(Permissions.Delete, 'delete')
    def delete(self,REQUEST=None):
        """
        delete (recycle) this page, if permissions allow
        """
        if not (self._checkPermission(Permissions.Delete, self) and
                self.requestHasSomeId(REQUEST)):
            raise 'Unauthorized', (
                _('You are not authorized to delete this ZWiki Page.'))
        # first, transfer any children to our (first) parent to avoid orphans
        for id in self.offspringIdsAsList(REQUEST):
            child = getattr(self.folder(),id)
            try:
                child.parents.remove(self.title_or_id())
            except ValueError:
                pass
            if self.parents:
                child.parents.append(self.parents[0])
            child.index_object()
        self._recycle(REQUEST)
        self.sendMailToSubscribers(
            'This page was deleted.\n',
            REQUEST=REQUEST,
            subjectSuffix='',
            subject='(deleted)')
        if REQUEST:
            # redirect to first existing parent or front page
            destpage = ''
            for p in self.parents:
                if hasattr(self.folder(),p):
                    destpage = p
                    break
            REQUEST.RESPONSE.redirect(self.wiki_url()+'/'+quote(destpage))

    def _recycle(self, REQUEST=None):
        """
        move this page to the recycle_bin subfolder, creating it if necessary.
        """
        # create recycle_bin folder if needed
        f = self.folder()
        if not hasattr(f,'recycle_bin'):
            f.manage_addFolder('recycle_bin', 'deleted wiki pages')
        # & move page there
        id=self.id()
        cb=f.manage_cutObjects(id)

        # update catalog if present
        # don't let manage_afterAdd catalog the new location..
        save = ZWikiPage.manage_afterAdd # not self!
        ZWikiPage.manage_afterAdd = lambda self,item,container: None
        f.recycle_bin.manage_pasteObjects(cb)
        ZWikiPage.manage_afterAdd = save

    security.declareProtected(Permissions.Rename, 'rename')
    def rename(self,pagename,leaveplaceholder=0,updatebacklinks=0,
               sendmail=1,REQUEST=None):
        """
        Rename this page, if permissions allow, with optional fixups:

        - leave a placeholder page

        - update links throughout the wiki. Warning, this replaces all
        occurrences of the old page name (beginning and ending with a word
        boundary), not just links. This tries to do the most sensible
        thing when dealing with freeform page names, but probably won't
        find all possible fuzzy links.

        - if called with the existing name, ensures that id conforms to
        canonicalId(title).
        """
        # rename permission is also declared above.. all due for cleanup
        # ?
        if not (self._checkPermission(Permissions.Rename, self) and
                self.requestHasSomeId(REQUEST)):
            raise 'Unauthorized', (
                _('You are not authorized to rename this ZWiki Page.'))
        newname = pagename
        oldname = self.title_or_id()
        newid = self.canonicalIdFrom(newname)
        oldid = self.getId()
        if not newname or (newname == oldname and newid == oldid):
            return
        if newname != oldname:
            # first, update the parents list of any children
            # I'm assuming any later problems will roll all this back
            for id in self.offspringIdsAsList(REQUEST):
                child = getattr(self.folder(),id)
                try: child.parents.remove(oldname)
                except ValueError: pass
                child.parents.append(newname)
                child.index_object()
            DLOG('renaming "%s" to "%s"' % (oldname,newname))
            self.title = newname
            if newid == oldid: # let's not do this twice
                self.index_object()
            if updatebacklinks:
                for p in self.backlinksFor(oldname):
                    p._replaceLink(oldname,newname,REQUEST)
        if newid != oldid:
            # manage_renameObject will overwrite these
            # doh.. means we do have to index twice in fact 
            creation_time,creator,creator_ip = \
              self.creation_time,self.creator,self.creator_ip
            self.folder().manage_renameObject(oldid,newid,REQUEST)
            self.creation_time,self.creator,self.creator_ip = \
              creation_time,creator,creator_ip
            self.index_object()
            self = getattr(self.folder(),newid)
            changed = 1
            if leaveplaceholder:
                if newname == newid:
                    link = newname
                else:
                    link = '[' + newname + ']'
                self.create(oldid,_("This page was renamed to %s.\n") % (link))
            if updatebacklinks:
                for p in self.backlinksFor(oldid):
                    p._replaceLink(oldname,newname,REQUEST)
        if sendmail and newname != oldname:
            self.sendMailToSubscribers(
                'This page was renamed from %s to %s.\n'%(oldname,newname),
                REQUEST=REQUEST,
                subjectSuffix='',
                subject='(renamed)')
        if REQUEST:
            REQUEST.RESPONSE.redirect(self.wiki_url()+'/'+quote(newid))

    def _replaceLink(self,old,new,REQUEST=None):
        """
        Replace occurrences of old (that look like links) with new in my text.
        """
        newtext = re.sub(r'\b%s\b' % (old), new, self.read())
        self.edit(text=newtext,REQUEST=REQUEST,log='links updated')

    def _handleFileUpload(self,REQUEST,log=''):
        # is there a file upload ?
        if (REQUEST and
            hasattr(REQUEST,'file') and
            hasattr(REQUEST.file,'filename') and
            REQUEST.file.filename):     # XXX do something

            # figure out the upload destination
            if hasattr(self,'uploads'):
                uploaddir = self.uploads
            else:
                uploaddir = self.folder()

            # do we have permission ?
            if not (self._checkPermission(Permissions.Upload,uploaddir)):# or
                    #self._checkPermission(Permissions.UploadSmallFiles,
                    #                self.folder())):
                raise 'Unauthorized', (
                    _('You are not authorized to upload files here.'))
            if not (self._checkPermission(Permissions.Change, self) or
                    self._checkPermission(Permissions.Append, self)):
                raise 'Unauthorized', (
                    _('You are not authorized to add a link on this ZWiki Page.'))
            # can we check file's size ?
            # yes! len(REQUEST.file.read()), apparently
            #if (len(REQUEST.file.read()) > LARGE_FILE_SIZE and
            #    not self._checkPermission(Permissions.Upload,
            #                        uploaddir)):
            #    raise 'Unauthorized', (
            #        _("""You are not authorized to add files larger than
            #        %s here.""" % (LARGE_FILE_SIZE)))

            # create the File or Image object
            file_id, content_type, size = \
                    self._createFileOrImage(REQUEST.file,
                                            title=REQUEST.get('title', ''),
                                            REQUEST=REQUEST)
            if file_id:
                # link it on the page and finish up
                self._addFileLink(file_id, content_type, size, REQUEST)
                self._setLastLog(log)
                self.index_object()
            else:
                # failed to create - give up (what about an error)
                pass

    def _createFileOrImage(self,file,title='',REQUEST=None,parent=None):
        # based on WikiForNow which was based on
        # OFS/Image.py:File:manage_addFile
        """
        Add a new File or Image object, depending on file's filename
        suffix. Returns a tuple containing the new id, content type &
        size, or (None,None,None).
        """
        # set id & title from filename
        title=str(title)
        id, title = OFS.Image.cookId('', title, file)
        if not id:
            return None, None, None

        # find out where to store files - in the 'uploads'
        # folder if defined, otherwise the wiki folder
        # NB a page might override this with a property
        if hasattr(self,'uploads'):
            folder = self.uploads
        else:
            folder = parent or self.folder() # see create()

        if hasattr(folder,id) and folder[id].meta_type in ('File','Image'):
            pass
        else:
            # First, we create the file or image object without data
            if guess_content_type(file.filename)[0][0:5] == 'image':
                folder._setObject(id, OFS.Image.Image(id,title,''))
            else:
                folder._setObject(id, OFS.Image.File(id,title,''))

        # Now we "upload" the data.  By doing this in two steps, we
        # can use a database trick to make the upload more efficient.
        folder._getOb(id).manage_upload(file)

        return id, folder._getOb(id).content_type, folder._getOb(id).getSize()

    def _addFileLink(self, file_id, content_type, size, REQUEST):
        """
        Add a link to the specified file at the end of this page,
        unless a link already exists.
        If the file is an image and not too big, inline it instead.
        """
        if re.search(r'(src|href)="%s"' % file_id,self.text()): return

        if hasattr(self,'uploads'):
            filepath = 'uploads/'
        else:
            filepath = ''
        if content_type[0:5] == 'image' and \
           not (hasattr(REQUEST,'dontinline') and REQUEST.dontinline) and \
           size <= LARGE_FILE_SIZE :
            linktxt = '\n\n<img src="%s%s">\n' % (filepath,file_id)
        else:
            linktxt = '\n\n<a href="%s%s">%s</a>\n' % (filepath,file_id,file_id)
        self._setText(self.read()+linktxt,REQUEST)
        self._setLastEditor(REQUEST)

    def _setOwnership(self, REQUEST=None):
        """
        set up the zope ownership of a new page appropriately

        depends on whether we are using regulations or not
        """
        if not self.usingRegulations():
            # To help control executable content, make sure the new page
            # acquires it's owner from the parent folder.
            self._deleteOwnershipAfterAdd()
        else:
            self._setOwnerRole(REQUEST)
            
    def _setText(self, text='', REQUEST=None):
        """
        Change the page text, with cleanups and perhaps DTML validation.
        """
        self.raw = self._cleanupText(text)
        self._preRender(clear_cache=1)
        # _renderWithLinks is called on every page view so I think there's
        # no benefit to doing it here ?
        # oho, it avoids an extra transaction & empty diff
        if self.supportsPreLinking():
            self._renderWithLinks()
        # XXX here we should do DTML validation to catch & roll back
        # errors, but were getting authorization problems
        #if self.supportsDtml():
            # a commit didn't seem to help
            #get_transaction().commit()
            # so we skipped it for the moment
            #DTMLDocument.__call__(self,self,REQUEST,REQUEST.RESPONSE)

    def _cleanupText(self, t):
        """do some cleanup of a page's new text
        """
        # strip any browser-appended ^M's
        t = re.sub('\r\n', '\n', t)

        # convert international characters to HTML entities for safekeeping
        #for c,e in intl_char_entities:
        #    t = re.sub(c, e, t)
        # assume today's browsers will not harm these.. if this turns out
        # to be false, do some smarter checking here

        # here's the place to strip out any disallowed html/scripting elements
        # XXX there are updates for this somewhere on zwiki.org
        if DISABLE_JAVASCRIPT:
            t = re.sub(javascriptexpr,r'&lt;disabled \1&gt;',t)

        # strip out HTML document header/footer if added
        t = re.sub(htmlheaderexpr,'',t)
        t = re.sub(htmlfooterexpr,'',t)

        return t

    def _setLastEditor(self, REQUEST=None):
        """
        record my last_editor & last_editor_ip
        """
        if REQUEST:
            self.last_editor_ip = REQUEST.REMOTE_ADDR
            self.last_editor = self.zwiki_username_or_ip(REQUEST)
        else:
            # this has been fiddled with before
            # if we have no REQUEST, at least update last editor
            self.last_editor_ip = ''
            self.last_editor = ''
        self.last_edit_time = DateTime(time.time()).ISO()

    def _setCreator(self, REQUEST=None):
        """
        record my creator, creator_ip & creation_time
        """
        self.creation_time = DateTime(time.time()).ISO()
        if REQUEST:
            self.creator_ip = REQUEST.REMOTE_ADDR
            self.creator = self.zwiki_username_or_ip(REQUEST)
        else:
            self.creator_ip = ''
            self.creator = ''

    security.declareProtected(Permissions.View, 'checkEditConflict')
    def checkEditConflict(self, timeStamp, REQUEST):
        """
        Warn if this edit would be in conflict with another.

        Edit conflict checking based on timestamps -
        
        things to consider: what if
        - we are behind a proxy so all ip's are the same ?
        - several people use the same cookie-based username ?
        - people use the same cookie-name as an existing member name ?
        - no-one is using usernames ?

        strategies:
        0. no conflict checking

        1. strict - require a matching timestamp. Safest but obstructs a
        user trying to backtrack & re-edit. This was the behaviour of
        early zwiki versions.

        2. semi-careful - record username & ip address with the timestamp,
        require a matching timestamp or matching non-anonymous username
        and ip.  There will be no conflict checking amongst users with the
        same username (authenticated or cookie) connecting via proxy.
        Anonymous users will experience strict checking until they
        configure a username.

        3. relaxed - require a matching timestamp or a matching, possibly
        anonymous, username and ip. There will be no conflict checking
        amongst anonymous users connecting via proxy. This is the current
        behaviour.
        """
        username = self.zwiki_username_or_ip()
        if (timeStamp is not None and
            timeStamp != self.timeStamp() and
            (not hasattr(self,'last_editor') or
             not hasattr(self,'last_editor_ip') or
             username != self.last_editor or
             REQUEST.REMOTE_ADDR != self.last_editor_ip)):
            return 1
        else:
            return 0

    security.declareProtected(Permissions.View, 'timeStamp')
    def timeStamp(self):
        return str(self._p_mtime)
    
    security.declareProtected(Permissions.FTP, 'manage_FTPget')
    def manage_FTPget(self):
        """Get source for FTP download.
        """
        #candidates = list(self.ZWIKI_PAGE_TYPES)
        #types = "%s (alternatives:" % self.page_type
        #if self.page_type in candidates:
        #    candidates.remove(self.page_type)
        #for i in candidates:
        #    types = types + " %s" % i
        #types = types + ")"
        types = "%s" % self.page_type
        return "Wiki-Safetybelt: %s\nType: %s\nLog: \n\n%s" % (
            self.timeStamp(), types, self.read())

    security.declareProtected(Permissions.Change, 'PUT')
    def PUT(self, REQUEST, RESPONSE):
        """Handle HTTP/FTP/WebDav PUT requests."""
        self.dav__init(REQUEST, RESPONSE)
        self.dav__simpleifhandler(REQUEST, RESPONSE, refresh=1)
        body=REQUEST.get('BODY', '')
        self._validateProxy(REQUEST)

        headers, body = parseHeadersBody(body)
        log = string.strip(headers.get('Log', headers.get('log', ''))) or None
        type = (string.strip(headers.get('Type', headers.get('type', '')))
                or None)
        if type is not None:
            type = string.split(type)[0]
            if type not in self.ZWIKI_PAGE_TYPES[:]:
                # Silently ignore it.
                type = None
        timestamp = string.strip(headers.get('Wiki-Safetybelt', '')) or None
        if timestamp and self.checkEditConflict(timestamp, REQUEST):
            RESPONSE.setStatus(423) # Resource Locked
            return RESPONSE

        #self._setText(body)
        #self._setLastEditor(REQUEST)
        #self.index_object()
        #RESPONSE.setStatus(204)
        #return RESPONSE
        try:
            self.edit(text=body, type=type, timeStamp=timestamp,
                      REQUEST=REQUEST, log=log, check_conflict=0)
        except 'Unauthorized':
            RESPONSE.setStatus(401)
            return RESPONSE
        RESPONSE.setStatus(204)
        return RESPONSE

    ######################################################################
    # miscellaneous

    security.declareProtected(Permissions.View, 'isZwikiPage')
    def isZwikiPage(self,object):
        return getattr(object,'meta_type',None) == self.meta_type

    security.declareProtected(Permissions.View, 'backlinksFor')
    def backlinksFor(self, page):
        """
        Return a list of pages that link to page, in no particular order.

        page may be a name or id.

        Optimisation: this method uses the catalog if present (and
        suitably configured). Otherwise it does a brute-force search of
        all pages, with increased zodb cache activity.  

        The brute force search is not too smart.

        XXX BF should check for fuzzy links
        XXX BF should do a smarter search (eg use _links)
        XXX how do we test the catalog optimization ?
        """
        try:
            cid = self.pageWithNameOrId(page).canonicalId()
            # check for canonicalLinks index, otherwise we'll get the lot
            if not 'canonicalLinks' in self.catalog().indexes():
                raise AttributeError
            backlinks = self.catalog()(canonicalLinks=cid)
            list = []
            for b in backlinks:
                list.append(b.getObject())
        except (AttributeError, TypeError):
            try:
                # check for both [name] and bare wiki links
                id = self.pageWithNameOrId(page).getId()
                regex = re.compile(r'\b(%s|%s)\b'%(page,id))
            except AttributeError:
                regex = re.compile(r'\b%s\b'%(page))
            list = []
            for p in self.pages():
                if regex.search(p.read()):
                    list.append(p)
        return list

    security.declareProtected(Permissions.View, 'zwiki_version')
    def zwiki_version(self):
        """
        Return the zwiki product version.
        """
        return __version__

    # expose these darn things for dtml programmers once and for all!
    # XXX safe ?
    security.declareProtected(Permissions.View, 'htmlquote')
    def htmlquote(self, text):
        return html_quote(text)

    security.declareProtected(Permissions.View, 'htmlunquote')
    def htmlunquote(self, text):
        return html_unquote(text)

    security.declareProtected(Permissions.View, 'urlquote')
    def urlquote(self, text):
        return quote(text)

    security.declareProtected(Permissions.View, 'urlunquote')
    def urlunquote(self, text):
        return unquote(text)

    security.declareProtected(Permissions.View, 'zwiki_username_or_ip')
    def zwiki_username_or_ip(self, REQUEST=None):
        """
        search REQUEST for an authenticated member or a zwiki_username
        cookie or a username passed in by mailin.py

        XXX added REQUEST arg at one point when sending mail
        notification in append() was troublesome - still needed ?
        refactor
        """
        username = None
        REQUEST = REQUEST or getattr(self,'REQUEST',None)
        if REQUEST:
            username = REQUEST.get('MAILIN_USERNAME',None)
            if not username:
                user = REQUEST.get('AUTHENTICATED_USER')
                if user:
                    username = user.getUserName()
                if not username or username == str(user.acl_users._nobody):
                    username = REQUEST.cookies.get('zwiki_username',None)
                    if not username:
                        username = REQUEST.REMOTE_ADDR
        return username or ''

    security.declareProtected(Permissions.View, 'requestHasSomeId')
    def requestHasSomeId(self,REQUEST):
        """
        Check REQUEST has either a non-anonymous user or a username cookie.
        """
        username = self.zwiki_username_or_ip(REQUEST)
        if (username and username != REQUEST.REMOTE_ADDR):
            return 1
        else:
            return 0

    security.declareProtected(Permissions.View, 'text')
    def text(self, REQUEST=None, RESPONSE=None):
        # see also backwards compatibility section
        """
        return this page's raw text
        (a permission-free version of document_src)
        also fiddle the mime type for web browsing
        """
        if RESPONSE is not None:
            RESPONSE.setHeader('Content-Type', 'text/plain')
            #RESPONSE.setHeader('Last-Modified', rfc1123_date(self._p_mtime))
        return self.read()

    # cf _setText, IssueNo0157
    _old_read = DTMLDocument.read
    security.declareProtected(Permissions.View, 'read')
    def read(self):
        return re.sub('<!--antidecapitationkludge-->\n\n?','',
                      self._old_read())

    def __repr__(self):
        return ("<%s %s at 0x%s>"
                % (self.__class__.__name__, `self.id()`, hex(id(self))[2:]))

    # methods for reliable dtml access to page & wiki url
    # these have been troublesome; they need to work with & without
    # virtual hosting. Tests needed.
    # Keep old versions around to help with debugging - see
    # http://zwiki.org/VirtualHostingSummary
    # XXX absolute_url uses REQUEST.SERVER_URL - would be nice for these
    # to work without requiring a live request
    
    security.declareProtected(Permissions.View, 'page_url')
    def page_url(self):
        """return the url path for this wiki page"""
        return self.page_url7()

    security.declareProtected(Permissions.View, 'wiki_url')
    def wiki_url(self):
        """return the base url path for this wiki"""
        return self.wiki_url7()

    security.declareProtected(Permissions.View, 'page_url1')
    def page_url1(self):
        """return the url path for the current wiki page"""
        o = self
        url = []
        while hasattr(o,'id'):
            url.insert(0,absattr(o.id))
            o = getattr(o,'aq_parent', None)
        return quote('/' + join(url[1:],'/'))

    security.declareProtected(Permissions.View, 'wiki_url1')
    def wiki_url1(self):
        """return the base url path for the current wiki"""
        # this code is buried somewhere in cvs
        return '?'

    security.declareProtected(Permissions.View, 'page_url2')
    def page_url2(self):
        """return the url path for the current wiki page"""
        return quote('/' + self.absolute_url(relative=1))

    security.declareProtected(Permissions.View, 'wiki_url2')
    def wiki_url2(self):
        """return the base url path for the current wiki"""
        return quote('/' + self.folder().absolute_url(relative=1))

    security.declareProtected(Permissions.View, 'page_url3')
    def page_url3(self):
        """return the url path for the current wiki page"""
        return self.REQUEST['BASE1'] + '/' + self.absolute_url(relative=1)

    security.declareProtected(Permissions.View, 'wiki_url3')
    def wiki_url3(self):
        """return the base url path for the current wiki"""
        return self.REQUEST['BASE1'] + '/' + self.folder().absolute_url(relative=1)

    security.declareProtected(Permissions.View, 'page_url4')
    def page_url4(self):
        """return the url path for the current wiki page"""
        return self.absolute_url(relative=0)

    security.declareProtected(Permissions.View, 'wiki_url4')
    def wiki_url4(self):
        """return the base url path for the current wiki"""
        return self.folder().absolute_url(relative=0)

    security.declareProtected(Permissions.View, 'page_url5')
    def page_url5(self):
        """return the url path for the current wiki page"""
        return self.absolute_url()

    security.declareProtected(Permissions.View, 'wiki_url5')
    def wiki_url5(self):
        """return the base url path for the current wiki"""
        return self.folder().absolute_url()

    security.declareProtected(Permissions.View, 'page_url6')
    def page_url6(self):
        """return the url path for the current wiki page"""
        return '/' + self.absolute_url(relative=1)

    security.declareProtected(Permissions.View, 'wiki_url6')
    def wiki_url6(self):
        """return the base url path for the current wiki"""
        return '/' + self.aq_inner._my_folder_orig().absolute_url(relative=1)

    def _my_folder_orig(self):
        """
        Obtain parent folder, avoiding potential acquisition recursion.

        ken's original version - keep it around in case
        """
        got = parent = self.folder()
        # Handle bizarred degenerate case - darnit, i've forgotten now where
        # i've seen this occur!
        while (got
               and hasattr(got, 'aq_parent')
               and got.aq_parent
               and (got is not got.aq_parent)
               and (aq_base(got) is aq_base(got.aq_parent))):
            parent = got
            got = got.aq_parent
        return parent

    security.declareProtected(Permissions.View, 'page_url7')
    def page_url7(self):
        """return the url path for the current wiki page"""
        return self.wiki_url() + '/' + quote(self.id())

    security.declareProtected(Permissions.View, 'wiki_url7')
    def wiki_url7(self):
        """return the base url path for the current wiki"""
        try:
            return self.folder().absolute_url()
        except KeyError,AttributeError:
            return '' # makes debugging/testing easier

    security.declareProtected(Permissions.View, 'creationTime')
    def creationTime(self):
        """return our creation time as a DateTime object"""
        if self.creation_time:
            try: return DateTime(self.creation_time)
            except DateTime.SyntaxError: return None
        else:
            return None

    security.declareProtected(Permissions.View, 'lastEditTime')
    def lastEditTime(self):
        """return our last edit time as a DateTime object"""
        if self.last_edit_time:
            try: return DateTime(self.last_edit_time)
            except DateTime.SyntaxError: return None
        else:
            return None

    security.declareProtected(Permissions.View, 'folder')
    def folder(self):
        """
        return this page's containing folder

        We used to use self.aq_parent everywhere, now
        self.aq_inner.aq_parent to ignore acquisition paths.
        Work for pages without a proper acquisition wrapper too.
        """
        return getattr(getattr(self,'aq_inner',self),'aq_parent',None)

    security.declareProtected(Permissions.View, 'age')
    def age(self):
        """
        return a string describing the approximate age of this page
        """
        return self.asAgeString(self.creation_time)

    security.declareProtected(Permissions.View, 'lastEditInterval')
    def lastEditInterval(self):
        """
        return a string describing the approximate interval since last edit
        """
        return self.asAgeString(self.last_edit_time)

    security.declareProtected(Permissions.View, 'asAgeString')
    def asAgeString(self,time):
        """
        return a string describing the approximate elapsed period since time

        time may be a DateTime or suitable string. Returns a blank string
        if there was a problem. Based on the dtml version in ZwikiTracker.
        """
        if not time:
            return 'some time'
        if type(time) is StringType:
            time = DateTime(time)
        # didn't work on a page in CMF, perhaps due to skin acquisition magic
        #elapsed = self.ZopeTime() - time
        elapsed = self.getPhysicalRoot().ZopeTime() - time
        hourfactor=0.041666666666666664
        minutefactor=0.00069444444444444447
        secondsfactor=1.1574074074074073e-05
        days=int(math.floor(elapsed))
        weeks=days/7
        months=days/30
        years=days/365
        hours=int(math.floor((elapsed-days)/hourfactor))
        minutes=int(math.floor((elapsed-days-hourfactor*hours)/minutefactor))
        seconds=int(round((
            elapsed-days-hourfactor*hours-minutefactor*minutes)/secondsfactor))
        if years:
            s = "%d year%s" % (years, years > 1 and 's' or '')
        elif months:
            s = "%d month%s" % (months, months > 1 and 's' or '')
        elif weeks:
            s = "%d week%s" % (weeks, weeks > 1 and 's' or '')
        elif days:
            s = "%d day%s" % (days, days > 1 and 's' or '')
        elif hours:
            s = "%d hour%s" % (hours, hours > 1 and 's' or '')
        elif minutes:
            s = "%d minute%s" % (minutes, minutes > 1 and 's' or '')
        else:
            s = "%d second%s" % (seconds, seconds > 1 and 's' or '')
        return s

    security.declareProtected(Permissions.View, 'linkTitle')
    def linkTitle(self,prettyprint=0):
        """
        return a suitable value for the title attribute of links to this page

        with prettyprint=1, format it for use in the standard header.
        """
        return self.linkTitleFrom(self.last_edit_time,
                                  self.last_editor,
                                  prettyprint=prettyprint)

    # please clean me up
    security.declareProtected(Permissions.View, 'linkTitleFrom')
    def linkTitleFrom(self,last_edit_time=None,last_editor=None,prettyprint=0):
        """
        make a link title string from these last_edit_time and editor strings
        
        with prettyprint=1, format it for use in the standard header.
        """
        interval = self.asAgeString(last_edit_time)
        if not prettyprint:
            s = "last edited %s ago" % (interval)
        else:
            try:
                assert self.REQUEST.AUTHENTICATED_USER.has_permission(
                    'View History',self)
                #XXX do timezone conversion ?
                lastlog = self.lastlog()
                if lastlog: lastlog = ' ('+lastlog+')'
                s = '<b><u>l</u></b>ast edited <a href="%s/diff" title="show last edit%s" accesskey="l">%s</a> ago' % \
                    (self.page_url(), lastlog, interval)
            except:
                s = 'last edited %s ago' % (interval)
        if (last_editor and
            not re.match(r'^[0-9\.\s]*$',last_editor)):
            # escape some things that might cause trouble in an attribute
            editor = re.sub(r'"',r'',last_editor)
            if not prettyprint:
                s = s + " by %s" % (editor)
            else:
                s = s + " by <b>%s</b>" % (editor)
        return s
    
    security.declareProtected(Permissions.Change, 'manage_edit')
    def manage_edit(self, data, title, REQUEST=None):
        """Do standard manage_edit kind of stuff, using our edit."""
        #self.edit(text=data, title=title, REQUEST=REQUEST, check_conflict=0)
        #I think we need to bypass edit to provide correct permissions
        self.title = title
        self._setText(data,REQUEST)
        self._setLastEditor(REQUEST)
        self.reindex_object()
        if REQUEST:
            message="Content changed."
            return self.manage_main(self,REQUEST,manage_tabs_message=message)

    ######################################################################
    # backwards compatibility

    security.declarePublic('upgradeAll') # check permissions at runtime
    def upgradeAll(self,pre_render=1,REQUEST=None):
        """
        Clear cache, upgrade and pre-render all pages

        Normally pages are upgraded/pre-rendered as needed.  An
        administrator may want to call this, particularly after a zwiki
        upgrade, to minimize later delays and to ensure all pages have
        been rendered by the latest code.

        Requires 'Manage properties' permission on the folder.
        Commit every so often an attempt to avoid memory/conflict errors.
        Has problems doing a complete run in large wikis, or when other
        page accesses are going on ?
        """
        if not self._checkPermission('Manage properties',self.folder()):
            raise 'Unauthorized', (
             _('You are not authorized to upgrade all pages.') + \
             _('(folder -> Manage properties)'))
        try: pre_render = int(pre_render)
        except: pre_render = 0
        if pre_render:
            DLOG('upgrading and prerendering all pages:')
        else:
            DLOG('upgrading all pages:')
        n = 0
        for p in self.pages():
            n = n + 1
            p.upgrade(REQUEST)
            p.upgradeId(REQUEST)
            if pre_render:
                p._preRender(clear_cache=1)
            DLOG('page #%d %s'%(n,p.id()))
            if n % 10 == 0:
                DLOG('committing')
                get_transaction().commit()
            # last pages will get committed as this request ends
        DLOG('%d pages processed' % n)

    #security.declarePublic('upgradeId')
    security.declareProtected(Permissions.View, 'upgradeId')
    def upgradeId(self,REQUEST=None):
        """
        Make sure a page's id conforms with it's title, renaming as needed.

        Does not leave a placeholder, so may break incoming links.
        Presently too slow for auto-upgrade, so let people call this
        directly or via upgradeAll (good luck :( )

        updatebacklinks=1 is used even though it's slow, because it's less
        work than fixing up links by hand afterward.

        With legacy pages (not new ones), it may happen that there's a
        clash between two similarly-named pages mapping to the same
        canonical id. In this case we just log the error and move on.
        """
        id, cid = self.getId(), self.canonicalId()
        if id != cid:
            oldtitle = title = self.title_or_id()
            # as a special case, preserve tracker issue numbers in the title
            m = re.match(r'IssueNo[0-9]+$',id)
            if m:
                title = '%s %s' % (m.group(),self.title)
            try:
                self.rename(title,updatebacklinks=1,sendmail=0,REQUEST=REQUEST)
            except CopyError:
                DLOG('failed to rename "%s" (%s) to "%s" (%s) - id clash ?' \
                     % (oldtitle,id,title,self.canonicalIdFrom(title)))

    #security.declarePublic('upgrade')
    security.declareProtected(Permissions.View, 'upgrade')
    def upgrade(self,REQUEST=None):
        """
        Upgrade an old page instance (and possibly the parent folder).

        Called as needed, ie at view time (set AUTO_UPGRADE=0 in
        Default.py to prevent this).  You could also call this on every
        page in your wiki to do a batch upgrade. Affects
        bobobase_modification_time. If you later downgrade zwiki, the
        upgraded pages may not work so well.
        """
        # Note that the objects don't get very far unpickling, some
        # by-hand adjustment via command-line interaction is necessary
        # to get them over the transition, sigh. --ken
        # not sure what this means --SM

        # What happens in the zodb when class definitions change ? I think
        # all instances in the zodb conform to the new class shape
        # immediately on refresh/restart, but what happens to
        # (a) old _properties lists ? not sure, assume they remain in
        # unaffected and we need to add the new properties
        # and (b) old properties & attributes no longer in the class
        # definition ?  I think these lurk around, and we need to delete
        # them.

        changed = 0

        # As of 0.17, page ids are always canonicalIdFrom(title); we'll
        # rename to conform with this where necessary
        # too slow!
        # changed = self.upgradeId()

        # fix up attributes first, then properties
        # don't acquire while doing this
        realself = self
        self = self.aq_base

        # migrate a WikiForNow _st_data attribute
        if hasattr(self, '_st_data'):
            self.raw = self._st_data
            del self._st_data
            changed = 1

        # upgrade old page types
        # choose pre-formatting & pre-linking variants where possible
        changedpagetypes = {
            # early zwiki
            'Structured Text'    :'stxprelinkhtml',
            'structuredtext_dtml':'stxprelinkdtmlfitissuehtml',
            'HTML'               :'prelinkhtml',
            'html_dtml'          :'prelinkdtmlhtml',
            'Classic Wiki'       :'wwmlprelink',
            'Plain Text'         :'plaintext',
            # pre-0.9.10
            'stxprelinkdtml'     :'stxprelinkdtmlfitissuehtml',
            'structuredtextdtml' :'stxprelinkdtmlfitissuehtml',
            'dtmlstructuredtext' :'stxprelinkdtmlfitissuehtml',
            'structuredtext'     :'stxprelinkhtml',
            'structuredtextonly' :'stxprelink',
            'classicwiki'        :'wwmlprelink',
            'htmldtml'           :'prelinkdtmlhtml',
            # I want to reuse the 'html' type, so leave it be.  Old 'html'
            # pages will lose their wikilinks; add to release notes and
            # let people fix manually.
            'plainhtmldtml'      :'dtmlhtml',
            'plainhtml'          :'html',
            # 0.17's stxprelinkdtmlfitissuehtml replaces these two
            'stxprelinkdtmlhtml' :'stxprelinkdtmlfitissuehtml',
            'issuedtml'          :'stxprelinkdtmlfitissuehtml',
            }
        if self.page_type in changedpagetypes.keys():
            self.page_type = changedpagetypes[self.page_type]
            # clear render cache; don't bother prerendering just now
            self.clearCache()
            changed = 1

        # Early zwikipages have a username string property - delete it.
        # Once it's gone, the username method below is revealed and allows
        # old dtml to keep working. 
        # Could attempt to transfer it to last_editor, then you'd need
        # to transfer timestamp as well or it looks confusing. Don't bother.
        if type(self.username) is StringType: 
            delattr(self,'username')
            changed = 1 

        # Pre-0.9.10, creation_time has been a string in custom format and
        # last_edit_time has been a DateTime. Now both are kept as
        # ISO-format strings. Might not be strictly necessary to upgrade
        # in all cases.. will cause a lot of bobobase_mod_time
        # updates.. do it anyway.
        if not self.last_edit_time:
            self.last_edit_time = self.bobobase_modification_time().ISO()
            changed = 1
        elif type(self.last_edit_time) is not StringType:
            self.last_edit_time = self.last_edit_time.ISO()
            changed = 1
        elif len(self.last_edit_time) != 19:
            try: 
                self.last_edit_time = DateTime(self.last_edit_time).ISO()
                changed = 1
            except DateTime.SyntaxError:
                # can't convert to ISO, just leave it be
                pass

        # If no creation_time, just leave it blank for now.
        # we shouldn't find DateTimes here, but check anyway
        if not self.creation_time:
            pass
        elif type(self.creation_time) is not StringType:
            self.creation_time = self.creation_time.ISO()
            changed = 1
        elif len(self.creation_time) != 19:
            try: 
                self.creation_time = DateTime(self.creation_time).ISO()
                changed = 1
            except DateTime.SyntaxError:
                # can't convert to ISO, just leave it be
                pass

        # _wikilinks is now _links
        #if hasattr(self.aq_base,'_wikilinks'): #XXX why doesn't this work
        if hasattr(self,'_wikilinks'):
            self._links = self._wikilinks.keys()[:]
            delattr(self,'_wikilinks')
            changed = 1 

        # update _properties
        # keep in sync with _properties above. Better if we used that as
        # the template (efficiently)
        oldprops = { # not implemented
            'page_type'     :{'id':'page_type','type':'string'},
            }
        newprops = {
            #'page_type'     :{'id':'page_type','type':'selection','mode': 'w',
            #                  'select_variable': 'ZWIKI_PAGE_TYPES'},
            'creator'       :{'id':'creator','type':'string','mode':'r'},
            'creator_ip'    :{'id':'creator_ip','type':'string','mode':'r'},
            'creation_time' :{'id':'creation_time','type':'string','mode':'r'},
            'last_editor'   :{'id':'last_editor','type':'string','mode':'r'},
            'last_editor_ip':{'id':'last_editor_ip','type':'string','mode':'r'},
            'last_edit_time':{'id':'creation_time','type':'string','mode':'r'},
            'last_log'      :{'id':'last_log', 'type': 'string', 'mode': 'r'},
            'NOT_CATALOGED' :{'id':'NOT_CATALOGED', 'type': 'boolean', 'mode': 'w'},
            }
        props = map(lambda x:x['id'], self._properties)
        for p in oldprops.keys():
            if p in props: # and oldprops[p]['type'] != blah blah blah :
                pass
                #ack!
                #self._properties = filter(lambda x:x['id'] != p,
                #                          self._properties)
                #changed = 1
        for p in newprops.keys():
            if not p in props:
                self._properties = self._properties + (newprops[p],)
                changed = 1

        # install issue properties if needed, ie if this page is being
        # viewed as an issue for the first time
        # could do this in isIssue instead
        if (self.isIssue() and not 'severity' in props):
            realself.manage_addProperty('category','issue_categories','selection')
            realself.manage_addProperty('severity','issue_severities','selection')
            realself.manage_addProperty('status','issue_statuses','selection')
            realself.severity = 'normal'
            changed = 1

        if changed:
            # bobobase_modification_time changed - put in a dummy user so
            # it's clear this was not an edit
            # no - you should be looking at last_edit_times, in which case
            # you don't want to see last_editor change for this.
            #self.last_editor_ip = ''
            #self.last_editor = 'UpGrade'
            # do a commit now so the current render will have the
            # correct bobobase_modification_time for display (many
            # headers/footers still show it)
            get_transaction().commit()
            # and log it
            DLOG('upgraded '+self.id())

        # finally, MailSupport does a bit more (merge here ?)
        realself._upgradeSubscribers()

    # some CMF compatibility methods for standard ZWikiPage
    SearchableText = text
    view = __call__
            
    # old API methods to help keep legacy DTML working
    # XXX permissions declarations needed ? Get rid of 'em ?
    wiki_page_url = page_url
    wiki_base_url = wiki_url
    editTimestamp = timeStamp
    checkEditTimeStamp = checkEditConflict
    doLegacyFixups = upgrade
    doSubscriberListFixups = MailSupport._upgradeSubscribers
    doSubscriberFixups = MailSupport._upgradeSubscribers
    # old render methods - these shouldn't be needed after
    # upgrade converts page types, but it may be disabled
    render_stxprelinkdtml     = render_stxprelinkdtmlfitissuehtml
    render_structuredtextdtml = render_stxdtmllinkhtml
    render_dtmlstructuredtext = render_dtmlstxlinkhtml
    render_structuredtext     = render_stxlinkhtml
    render_structuredtextonly = render_stxlink
    render_classicwiki        = render_wwmllink
    render_htmldtml           = render_dtmllinkhtml
    render_plainhtmldtml      = render_dtmlhtml
    render_plainhtml          = render_html
    
    security.declareProtected(Permissions.View, 'username')
    def username(self):
        # backwards compatibility for dtml which tries to display the old
        # username property. NB upgrade may not have been called yet.
        # XXX I think we can be less slavishly compatible at this
        # point. Just return last_editor ?
        for p in self._properties:
            if p['id'] == 'username':
                return username
        if hasattr(self,'last_editor') and self.last_editor:
            return self.last_editor
        if hasattr(self,'last_editor_ip') and self.last_editor_ip:
            return self.last_editor_ip
        return ''
        
    # actually the above is useful to keep around as
    security.declareProtected(Permissions.View, 'last_editor_or_ip')
    last_editor_or_ip = username

    # document_src is the standard zope accessor for raw content
    # inherited from DTMLDocument
    # text/src is a permission-free accessor for the content
    # XXX still need this ? document_src & src equivalent ? clean up
    security.declareProtected(Permissions.View, 'src')
    src = text

Globals.InitializeClass(ZWikiPage) # install permissions


# ZMI page creation form
manage_addZWikiPageForm = DTMLFile('dtml/zwikiPageAdd', globals())

def manage_addZWikiPage(self, id, title='', file='', REQUEST=None,
                        submit=None):
    """
    Add a ZWiki Page object with the contents of file.

    Usually zwiki pages are created by clicking on ? (via create); this
    allows them to be added in the standard ZMI way. These should give
    mostly similar results; refactor the two methods together if possible.
    """
    # create page and initialize in proper order, as in create.
    p = ZWikiPage(source_string='', __name__=id)
    newid = self._setObject(id,p)
    p = getattr(self,newid)
    p.title=title
    p._setCreator(REQUEST)
    p._setLastEditor(REQUEST)
    p._setOwnership(REQUEST)
    if hasattr(self,'standard_page_type'):
        p.page_type = self.standard_page_type
    else:
        p.page_type = DEFAULT_PAGE_TYPE
    text = file
    if type(text) is not StringType: text=text.read()
    p._setText(text or '',REQUEST)
    p.index_object()
    if p.usingRegulations():
        p.setRegulations(REQUEST,new=1)
    if REQUEST is not None:
        try: u=self.DestinationURL()
        except: u=REQUEST['URL1']
        if submit==" Add and Edit ": u="%s/%s" % (u,quote(id))
        REQUEST.RESPONSE.redirect(u+'/manage_main')
    return ''


#python notes
#
#   "The first line should always be a short, concise summary of the
#object's purpose.  For brevity, it should not explicitly state the
#object's name or type, since these are available by other means (except
#if the name happens to be a verb describing a function's operation).
#This line should begin with a capital letter and end with a period.
#
#   If there are more lines in the documentation string, the second line
#should be blank, visually separating the summary from the rest of the
#description.  The following lines should be one or more paragraphs
#describing the object's calling conventions, its side effects, etc."
#
#   "Data attributes override method attributes with the same name; to
#avoid accidental name conflicts, which may cause hard-to-find bugs in
#large programs, it is wise to use some kind of convention that
#...
#verbs for methods and nouns for data attributes."
