<?php

/*
 * A generic table class.  Is is meant to be subclassed by particular
 * table classes.  A table is represented by a name, an array of
 * primary keys, and an array mapping field names to values.
 *
 * This file is part of Zoph.
 *
 * Zoph is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * Zoph is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with Zoph; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

class zoph_table {

    var $table_name;
    var $primary_keys;
    var $fields;
    var $not_null; // Fields that may not be empty

    /*
     * This construnctor should be called from the constructor
     * of a subclass.
     */
    function zoph_table($table_name, $primary_keys, $not_null) {
        $this->table_name = DB_PREFIX . $table_name;
        $this->primary_keys = $primary_keys;
        $this->not_null = $not_null;
        $this->fields = array();
    }

    /*
     * Gets a field.
     */
    function get($name) {
        if(DEBUG>10) {
           echo "<pre><b>GET " . $name . "</b>\n";
           var_dump($this->fields);
           echo "\n\n</pre>";
        }
        if (isset($this->fields[$name])) {
            return $this->fields[$name];
        } else {
            return "";
        }
    }

    /*
     * Sets a field.
     */
    function set($name, $value) {
        $this->fields[$name] = $value;
    }

    /*
     * Sets fields from the given array.  Can be used to set vars
     * directly from a GET or POST.
     */
    function set_fields($vars, $prefix = null, $suffix = null) {

        reset($vars);
        while (list($key, $val) = each($vars)) {

            if (DEBUG > 2) { echo "<b>$key</b> = $val<br>\n"; }

            // ignore empty keys or values unless the field must be set.
            if ((!in_array($key, $this->not_null)) && (empty($key) )) { continue; }

            if ($prefix) {
                if (strpos($key, $prefix) === 0) {
                    $key = substr($key, strlen($prefix));
                }
                else {
                    continue;
                }
            }
            else if ($key[0] == '_') {
                // a leading uderscore signals a non-database field
                continue;
            }

            if ($suffix) {
                $pos = strpos($key, $suffix);
                if (($pos > 0) && (preg_match("/".$suffix."$/", $key))) {
                    $key = substr($key, 0, $pos);
                }
                else {
                    continue;
                }
            }

            // something in ALL CAPS is probably PHP or HTML related
            if (strtoupper($key) == $key) { continue; }

            $this->fields[$key] = stripslashes($val);

        }

    }

    /*
     * Checks to see if the given field is listed as a primary key.
     */
    function is_key($name, $keys = null) {
        if (!$keys) { $keys = $this->primary_keys; }
        return in_array($name, $keys);
    }

    /*
     * Looks up a record.
     */
    function lookup($sql = null) {

        if (!$this->table_name || !$this->primary_keys || !$this->fields) {
            if (DEBUG) { echo "Missing data<br>\n"; }
            return;
        }

        if (!$sql) {
            $constraint = $this->create_constraints($this->primary_keys);

            if (!$constraint) {
                if (DEBUG) { echo "No constraint found<br>\n"; }
                return;
            }

            $sql = "select * from $this->table_name where $constraint";
        }

        if (DEBUG) { echo "$sql<br>\n"; }

        $result = mysql_query($sql)
            or die_with_mysql_error("Lookup failed:", $sql);

        if (mysql_num_rows($result) == 1) {
            $row = mysql_fetch_assoc($result);

            $this->fields = array();

            //$this->set_fields($row);
            $this->fields = array_merge($this->fields, $row);

            return 1;
        }

        return 0;

    }

    /*
     * Inserts a record.  The default behavior is to ignore the
     * primary key field(s) with the assumption that these will
     * be generated by mysql (auto_increment).  Passing a non null
     * parameter causes these fields to be manually inserted.
     */
    function insert($keep_key = null) {

        if (!$this->table_name || !$this->fields) {
            if (DEBUG) { echo "Missing data<br>\n"; }
            return;
        }

        while (list($name, $value) = each($this->fields)) {

            if ($this->primary_keys && !$keep_key && $this->is_key($name)) {
                continue;
            }

            if ($names) {
                $names .= ", ";
                $values .= ", ";
            }

            $names .= $name;

            if ($name == "password") {
                $values .= "password('" . escape_string($value) . "')";
            }
            else if ($value == "now()") {
                /* Lastnotify is normaly set to "now()" and should not be escaped */
                $values .=  $value ;
            } else if ($value =="" && in_array($name, $this->not_null)) {
	    	    die("<p class='error'><b>$name</b> may not be empty</p>");
	        } else if ($value !== "") {
                $values .= "'" . escape_string($value) . "'";
            } else {
                $values .= "null";
            }

        }

        $sql = "insert into $this->table_name ($names) values ($values)";

        if (DEBUG) { echo "$sql<br>\n"; }

        mysql_query($sql) or die_with_mysql_error("Insert failed:", $sql);

        $id = mysql_insert_id();

        if ($this->primary_keys && count($this->primary_keys) == 1) {
            $this->fields[$this->primary_keys[0]] = $id;
        }

        return $id;

    }

    /*
     * Deletes a record.  If extra tables are specified, entries from
     * those tables this match the keys are removed as well.
     */
    function delete($keys = null, $extra_tables = null) {
        if (!$keys) { $keys = $this->primary_keys; }

        if (!$this->table_name || !$keys || !$this->fields) {
            if (DEBUG) { echo "Missing data<br>\n"; }
            return;
        }

        $constraints = $this->create_constraints($keys);

        if (!$constraints) {
            if (DEBUG) { echo "No constraint found<br>\n"; }
            return;
        }

        $sql = "delete from $this->table_name where $constraints";

        if (DEBUG) { echo "$sql<br>\n"; }

        mysql_query($sql) or die_with_mysql_error("Delete failed:", $sql);

        if ($extra_tables) {
            foreach ($extra_tables as $table) {
                $table = DB_PREFIX . $table;
                $sql = "delete from $table where $constraints";
                if (DEBUG) { echo "$sql<br>\n"; }
                mysql_query($sql) or
                    die_with_mysql_error("Delete from $table failed:", $sql);
            }
        }
    }

    /*
     * Updates a record.
     */
    function update($keys = null) {
        if (!$keys) { $keys = $this->primary_keys; }

        if (!$this->table_name || !$keys || !$this->fields) {
            if (DEBUG) { echo "Missing data<br>\n"; }
            return;
        }

        $constraints = $this->create_constraints($keys);

        if (!$constraints) {
            if (DEBUG) { echo "No constraint found<br>\n"; }
            return;
        }

        while (list($name, $value) = each($this->fields)) {

            if ($this->is_key($name, $keys)) { continue; }

            if ($values) { $values .= ", "; }

            if ($name == "password") {
                $values .= "$name = password('" . escape_string($value) . "')";
            } else if ($value == "now()" ) {
                $values .= "$name = " . $value . "";
            } else if ($value == "" && in_array($name, $this->not_null)) {
	    	    die("<p class='error'><b>$name</b> may not be empty</p>");
	        } else if ($value !== "") {
                $values .= "$name = '" . escape_string($value) . "'";
            } else {
                $values .= "$name = null";
            }
        }

        if (!$values) { return; }

        $sql = "update $this->table_name set $values where $constraints";

        if (DEBUG) { echo "$sql<br>\n"; }

        mysql_query($sql) or die_with_mysql_error("Update failed:", $sql);

    }

    /*
     * Creates a constraint clause based on the given keys
     */
    function create_constraints($keys) {
        $constraints="";
        foreach ($keys as $key) {
            $value = $this->fields[$key];
            if (!$value) { continue; }
            if (!empty($constraints)) { $constraints .= " and "; }
            $constraints .= "$key = '" . escape_string($value) . "'";
        }
        return $constraints;
    }

    /*
     * Creates an alphabetized array of field names and values.
     */
    function get_display_array() {
        if (!$this->fields) { return; }

        $keys = array_keys($this->fields);
        sort($keys);
        reset($keys);
        foreach ($keys as $k) {
            if ($this->is_key($k)) { continue; }
            $title = make_title($k);
            $da[$title] = $this->fields[$k];
        }

        return $da;
    }

    /*
     * Creates an alphabetized array of field names and text input blocks.
     */
    function get_edit_array() {
        if (!$this->fields) { return; }

        $field_lengths = get_field_lengths($this->table_name);

        $keys = array_keys($field_lengths);
        sort($keys);
        reset($keys);
        foreach ($keys as $k) {
            if ($this->is_key($k)) { continue; }
            $title = make_title($k);

            $len = $field_lengths[$k];
            $size = min($len, 20);

            $ea[$title] = create_text_input($k, $this->fields[$k], $size, $len);
        }

        return $ea;
    }

    function get_mapping_js($user,$icon,$edit=false) {
        $marker=true;
        $lat=$this->get("lat");
        $lon=$this->get("lon");
        $zoom=$this->get("mapzoom");
        if(!$lat && !$lon) { $marker=false; }
        if(!$lat) { $lat=0; }
        if(!$lon) { $lon=0; }
        if(!$zoom) { $zoom=2; }
        $js="<script type='text/javascript'>\n" .
            "  var center=new LatLonPoint(" .
                $lat . "," .
                $lon . ");\n" .
            "  var zoomlevel=" . $zoom . ";\n" .
            "  mapstraction.setCenterAndZoom(center,zoomlevel);\n";
         if ($marker ) {
            $js.="  createMarker(" . $lat . "," . $lon . ",'" . $icon . "',null, null);\n";
         }
         if ($edit) {
            $js.="  setUpdateHandlers();\n";
         }
            $js.="</script>";
        return $js;
    }

    function get_marker($user, $icon) {
        $lat=$this->get("lat");
        $lon=$this->get("lon");
        if($lat && $lon) {
            $quicklook=$this->get_quicklook($user);
            return "  createMarker(" . $lat . "," . $lon . ", '" . $icon .
                    "','" .  escape_string($title) . "','" . 
                    $quicklook . "');\n";
        } else {
            return null;
        }
    }


}

/*
 * Gets the total count of records in the table for the given class.
 */
function get_count($class) {

    if (class_exists($class)) {
        $obj = new $class;
        $table = $obj->table_name;
    }
    else {
        $table = DB_PREFIX . $class;
    }

    $sql = "select count(*) from $table";

    if (DEBUG > 1) { echo "$sql<br>\n"; }

    return get_count_from_query($sql);
}


/*
 * Executes the given query and returns the result.
 */
function get_count_from_query($sql) {
    if (DEBUG > 1) { echo "$sql<br>\n"; }

    $result = mysql_query($sql)
        or die_with_mysql_error("Unable to get count", $sql);

    return mysql_result($result, 0, 0);
}

/*
 * Gets an array of the records for a table by doing a * "select *"
 * and storing the results in classes of the given type.
 */
function get_records($class, $order = null, $constraints = null,
    $conj = "and", $ops = null) {

    $obj = new $class;
    $sql = "select * from $obj->table_name";
    if ($constraints) {
        while (list($name, $value) = each($constraints)) {
            if ($constraint_string) {
                $constraint_string .= " $conj ";
            }
            else {
                $constraint_string .=  " where ";
            }

            $op = "=";
            if ($ops && $ops["$name"]) {
                $op = $ops["$name"];
            }

            $n = strpos($name, "#");
            if ($n > 1) {
                $name = substr($name, 0, $n);
            }

            if ($value == "null" || $value == "''") {
                // ok
            }
            else {
                $value = "'" . escape_string($value) . "'";
            }

            $constraint_string .= "$name $op $value";
        }
        $sql .= $constraint_string;
    }

    if ($order) {
        $sql .= " order by $order";
    }
    return get_records_from_query($class, $sql);
}

/*
 * Stores the results the the given query in an array of objects of
 * this given type.
 */
function get_records_from_query($class, $sql, $min = 0, $num = 0) {

    if (DEBUG > 1) { echo "$sql<br>\n"; }

    $result = mysql_query($sql)
        or die_with_mysql_error("Unable to get records", $sql);

    if ($min) {
        mysql_data_seek($result, $min);
    }

    if ($num) {
        $limit = true;
    } else {
        $limit = false;
    }

    $objs = array();
    if ($class != null) {
        while ((!$limit || $num-- > 0) && $row = mysql_fetch_assoc($result)) {
            $obj = new $class;
            $obj->set_fields($row);
            $objs[] = $obj;
        }
    }
    else {
        // use to grab ids, for example
        while ((!$limit || $num-- > 0) && $row = mysql_fetch_row($result)) {
            $objs[] = $row[0];
        }
    }

    mysql_free_result($result);
    return $objs;
}

/*
 * Executes a query and returns an array in which each record's
 * link is mapped to its count (dirived by a group by clause).
 */
function get_popular_results($class, $query) {
    $records = get_records_from_query($class, $query);
    foreach ($records as $rec) {
        $pop_array[$rec->get_link()] = $rec->get("count");
    }
    return $pop_array;
}

/*
 * Creates an array to be used in the create_pulldown methods.  The
 * values of the fields in the name_fields parameter are concatentated
 * together to construnct the titles of the selections.
 */
function create_select_array($records, $name_fields) {
    if (!$records || !$name_fields) { return; }

    foreach ($records as $rec) {
        // this only makes sense when there is one key
        $id = $rec->get($rec->primary_keys[0]);

        $name = "";
        foreach ($name_fields as $n) {
            if ($name) { $name .= " "; }
            $name .= $rec->get($n);
        }

        $sa[$id] = $name;
    }

    return $sa;
}

/*
 * Creates an object of the given class and returns the output of
 * its get_link() method.
 */
function get_link($class, $id) {
    if (!$class || !$id) { return ""; }
    $obj = new $class($id);
    $obj->lookup();
    return $obj->get_link();
}

/*
 * Creates a comma separated list of links from the given records.
 * The class of the records must implement the get_link function.
 */
function create_link_list($records) {
    $links = "";
    if ($records) {
        foreach ($records as $rec) {
            if ($links) { $links .= ", "; }
            $links .= $rec->get_link();
        }
    }

    return $links;
}

function get_xml($class, $search) {
    $search=strtolower($search);
    if($class=="location") {
        $class="place";
    } else if($class=="photographer") {
        $class="person";
    } 

    if($class=="person") {
        $tree=false;
    } else {
        $tree=true;
    }

    if($class=="timezone") {
        $tz=new TimeZone("UTC");
        return $tz->get_xml($search);
    }
    if (class_exists($class)) {
        $obj=new $class;
        $rootname=$obj->xml_rootname();
        $nodename=$obj->xml_nodename();
        $idname=$obj->primary_keys[0];

        $xml = new DOMDocument('1.0','UTF-8');
        $rootnode=$xml->createElement($obj->xml_rootname());


        if ($tree) {
            $obj = get_root($class);
            $obj->lookup();
            $tree=$obj->get_xml_tree($xml, $search);
            $rootnode->appendChild($tree);
        } else {
            if($class="person") {
                $constraints=array("lower(first_name)" => $search . "%", "lower(last_name)"=> $search . "%", "lower(concat(first_name, \" \", last_name))" => $search . "%");
                $conj="or";
                $order="last_name";
                $ops=array("lower(first_name)" => "like", "lower(last_name)" => "like", "lower(concat(first_name, \" \", last_name))" => "like");
            }
            $records=get_records($class, $order, $constraints, $conj, $ops);

            foreach($records as $record) {
                $newchild=$xml->createElement($nodename);
                $key=$xml->createElement("key");
                $title=$xml->createElement("title");
                $key->appendChild($xml->createTextNode($record->get($idname)));
                $title->appendChild($xml->createTextNode($record->get_name()));
                $newchild->appendChild($key);
                $newchild->appendChild($title);
                $rootnode->appendChild($newchild);
             }
        }
    } else {
        die("illegal class $class");
    }
    $xml->appendChild($rootnode);
    return $xml->saveXML();
}

////////// some utility functions //////////

/*
 * Escapes various characters in the given string using the
 * mysql_escape_string function.
 */
function escape_string($str) {

    /*
       Now every request var should be retrieved from the
       getvar() function, which will strip the slashes.
    if (ini_get("magic_quotes_gpc")) {
        // unstrip quotes
        $str = stripslashes($str);
    }
    */
    $str=str_replace(array("<", ">", "\"", "(", ")"), array("&lt;", "&gt;", "&quot;", "&#40;", "&#41;"), $str);
    return mysql_escape_string($str);

}

/*
 * Simply executes the given query.  If silence is passed, any errors
 * are ignored.
 */
function execute_query($sql, $silent = 0) {
    if (DEBUG > 1) { echo "$sql<br>\n"; }
    if ($silent) {
        mysql_query($sql);
    }
    else {
        mysql_query($sql) or die_with_mysql_error("Something went wrong", $sql);
    }
}

function die_with_mysql_error($msg, $sql = "") {
    $msg =
        "<p><strong>$msg</strong><br>\n" .
        "<code>" . mysql_error() . "</code></p>\n";

    if ($sql) {
        $msg .= "<hr><p><code>$sql</code></p>\n";
    }

    die($msg);
}

function get_zoph_info_array() {
    $album= get_root_album();
    $category = get_root_category();

    $size=get_human(get_photo_sizes_sum());
    return array(
        translate("number of photos") => get_count("photo"),
        translate("size of photos") => "$size",
        translate("number of photos in an album") =>
            $album->get_total_photo_count(),
        translate("number of categorized photos") =>
            $category->get_total_photo_count(),
        translate("number of people") => get_count("person"),
        translate("number of places") => get_count("place")
    );
}

function get_field_lengths($table_name) {

    $db_fields = mysql_list_fields(DB_NAME, $table_name);
    $columns = mysql_num_fields($db_fields);
    for ($i = 0; $i < $columns; $i++) {
        $field_lengths[mysql_field_name($db_fields, $i)] =
            mysql_field_len($db_fields, $i);
    }

    return $field_lengths;
}

?>
