/***************************************************************************
 *
 * COPYRIGHTHERE
 *
 * $Id: attach.c,v 1.7.2.12 2004/02/03 09:16:28 sasa Exp $
 *
 * Author  : Bazsi
 * Auditor :
 * Last audited version:
 * Notes:
 *
 ***************************************************************************/

#include <zorp/conntrack.h>
#include <zorp/attach.h>
#include <zorp/connect.h>
#include <zorp/log.h>
#include <zorp/streamfd.h>

#include <string.h>

struct _ZAttach
{
  GStaticRecMutex lock; /* lock protecting ref_cnt */
  gint ref_cnt;
  gchar session_id[MAX_SESSION_ID];
  guint proto;
  ZSockAddr *bind_addr;
  ZSockAddr *local;
  ZSockAddr *remote;
  ZAttachParams params;
  ZAttachCallback callback;
  gpointer user_data;
  GDestroyNotify user_data_notify;
  
  GMutex *connected_lock;
  GCond *connected_cond;
  ZConnection *conn;
  
  union 
  {
    ZIOConnect *connect;
  } c;
};

#define not_connected_yet_mark ((ZConnection *) &z_attach_new)


static void
z_attach_free(ZAttach *self)
{
  z_session_enter(self->session_id);
  
  if (self->conn && self->conn != not_connected_yet_mark)
    z_connection_destroy(self->conn, FALSE);
  
  z_sockaddr_unref(self->bind_addr);
  z_sockaddr_unref(self->local);
  z_sockaddr_unref(self->remote);
  switch (self->proto)
    {
    case ZD_PROTO_TCP:
      if (self->c.connect)
        z_io_connect_unref(self->c.connect);
      break;
    }
  if (self->connected_lock)
    {
      g_mutex_free(self->connected_lock);
      g_cond_free(self->connected_cond);
    }
  g_free(self);
  z_leave();
}

ZAttach *
z_attach_ref(ZAttach *self)
{
  g_static_rec_mutex_lock(&self->lock);
  g_assert(self->ref_cnt);
  self->ref_cnt++;
  g_static_rec_mutex_unlock(&self->lock);
  return self;
}

void
z_attach_unref(ZAttach *self)
{
  g_static_rec_mutex_lock(&self->lock);
  g_assert(self->ref_cnt);
  if (--self->ref_cnt == 0)
    {
      g_static_rec_mutex_unlock(&self->lock);
      z_attach_free(self);
    }
  else
    g_static_rec_mutex_unlock(&self->lock);
}

static void
z_attach_callback(ZAttach *self, ZConnection *conn)
{
  gchar buf[256];

  z_session_enter(self->session_id);
  z_log(NULL, CORE_DEBUG, 6, "Established connection; %s", z_connection_format(conn, buf, sizeof(buf)));
  if (self->callback)
    {
      self->callback(conn, self->user_data);
      if (self->user_data && self->user_data_notify)
        {
          self->user_data_notify(self->user_data);
        }
      self->user_data = NULL;
    }
  else
    {
      g_mutex_lock(self->connected_lock);
      self->conn = conn;
      g_cond_signal(self->connected_cond);
      g_mutex_unlock(self->connected_lock);
    }
  z_session_leave(self->session_id);
}

static void
z_attach_tcp_callback(gint fd, gpointer user_data)
{
  ZAttach *self = (ZAttach *) user_data;
  ZConnection *conn;
  
  z_session_enter(self->session_id);

  if (fd != -1)
    {
      conn = z_connection_new();
      if (z_getsockname(fd, &conn->local) != G_IO_STATUS_NORMAL ||
          z_getpeername(fd, &conn->remote) != G_IO_STATUS_NORMAL)   
        {
          z_connection_destroy(conn, FALSE);
          close(fd);
          z_session_leave(self->session_id);
          return;   
        }
      conn->protocol = ZD_PROTO_TCP;
      conn->stream = z_stream_new(fd, "");
      conn->dest = z_sockaddr_ref(conn->remote);
      conn->bound = z_sockaddr_ref(self->local);
    }
  else
    conn = NULL;
  
  z_attach_callback(self, conn);
  z_session_leave(self->session_id);
}

gboolean
z_attach_start(ZAttach *self)
{
  gboolean res = FALSE;
  
  z_session_enter(self->session_id);
  switch (self->proto)
    {
    case ZD_PROTO_TCP:
      {
        self->c.connect = z_io_connect_new(self->bind_addr, self->remote, z_attach_tcp_callback, self);
        if (self->c.connect)
          {
            z_io_connect_set_timeout(self->c.connect, self->params.tcp.timeout);
            z_attach_ref(self);
            z_io_connect_set_destroy_notify(self->c.connect, (GDestroyNotify) z_attach_unref);
            if (self->callback)
              {
                self->local = z_io_connect_start(self->c.connect);
              }
            else
              {
                self->local = z_io_connect_start_block(self->c.connect);
              }
            if (!self->local)
              {
                z_io_connect_unref(self->c.connect);
                self->c.connect = NULL;
              }
            else
              res = TRUE;
          }
        break;
      }
#if ENABLE_CONNTRACK
    case ZD_PROTO_UDP:
      {
        ZCTSocket *sock;
        ZStream *proxy_stream;
        ZConnection *conn;
        
        sock = z_conntrack_socket_new(self->session_id, self->params.udp.tracker, self->remote, self->bind_addr, ZCS_TO_SERVER, &proxy_stream);
        
        if (sock)
          {
            conn = z_connection_new();
            conn->protocol = ZD_PROTO_UDP;
            conn->stream = proxy_stream;
            conn->local = z_sockaddr_ref(sock->local_addr);
            conn->bound = z_sockaddr_ref(sock->local_addr);
            self->local = z_sockaddr_ref(sock->local_addr);
            conn->remote = z_sockaddr_ref(sock->remote_addr);
            conn->dest = z_sockaddr_ref(sock->remote_addr);
            z_attach_callback(self, conn);
            res = TRUE;
            z_conntrack_socket_start(sock);
            z_conntrack_socket_unref(sock);
          }
        break;
      }
#endif
    }
  z_session_leave(self->session_id);
  return res;
}

ZConnection *
z_attach_block(ZAttach *self)
{
  ZConnection *conn;
  
  z_session_enter(self->session_id);
  z_attach_ref(self);
  
  g_mutex_lock(self->connected_lock);
  while (self->conn == not_connected_yet_mark)
    g_cond_wait(self->connected_cond, self->connected_lock);
  conn = self->conn;
  self->conn = NULL;
  g_mutex_unlock(self->connected_lock);
  
  z_attach_unref(self);
  z_session_leave(self->session_id);
  return conn;
}

void
z_attach_cancel(ZAttach *self)
{
  z_session_enter(self->session_id);
  switch (self->proto)
    {
    case ZD_PROTO_TCP:
      if (self->c.connect)
        z_io_connect_cancel(self->c.connect);
      break;
    case ZD_PROTO_UDP:
      break;
    }
    
  /* either our callback was called in which case user_data is already
   * NULLed or we _have_ to call user_data_notify here. */
  
  if (self->user_data && self->user_data_notify)
    {
      self->user_data_notify(self->user_data);
     }
  self->user_data = NULL;
  z_session_leave(self->session_id);
}

ZSockAddr *
z_attach_get_local(ZAttach *self)
{
  return z_sockaddr_ref(self->local);
}

ZAttach *
z_attach_new(gchar *session_id, 
             guint proto, ZSockAddr *bind_addr, ZSockAddr *remote, 
             ZAttachParams *params,
             ZAttachCallback callback, gpointer user_data, GDestroyNotify notify)
{
  ZAttach *self = g_new0(ZAttach, 1);
  
  z_session_enter(session_id);
  g_strlcpy(self->session_id, session_id, sizeof(self->session_id));
  self->proto = proto;
  self->bind_addr = z_sockaddr_ref(bind_addr);
  self->remote = z_sockaddr_ref(remote);
  self->callback = callback;
  self->user_data = user_data;
  self->user_data_notify = notify;
  self->ref_cnt = 1;
  memcpy(&self->params, params, sizeof(self->params));
  if (!callback)
    {
      self->connected_lock = g_mutex_new();
      self->connected_cond = g_cond_new();
    }
  self->conn = not_connected_yet_mark;
  z_session_leave(session_id);
  return self;
}
