/***************************************************************************
 *
 * COPYRIGHTHERE
 *
 * $Id: szig.c,v 1.4.2.7 2003/10/07 15:09:12 bazsi Exp $
 *
 * Author  : SaSa
 * Auditor :
 * Last audited version:
 * Notes:
 *
 ***************************************************************************/

#include <zorp/szig.h>

#include <zorp/sockaddr.h>
#include <zorp/listen.h>
#include <zorp/log.h>
#include <zorp/stream.h>
#include <zorp/streamfd.h> 
#include <zorp/streamline.h>
#include <zorp/io.h>
#include <zorp/thread.h>

#include <string.h>

#include <glib.h>

#define UNIX_NAME "/var/run/zorp/zorpctl"
#define NAME_LEN 4096

static GNode *root = NULL;

typedef struct _ZSZIGWriteBuf
{
  gchar value[NAME_LEN];
  guint pos;
} ZSZIGWriteBuf;

typedef struct _ZorpSZIGNode
{
  GMutex *lock;
  gchar name[4096];
  GValue *value;
} ZorpSZIGNode;

gboolean z_szig_accept_callback(int  fd, ZSockAddr *client, gboolean last_connection, gpointer  user_data);
gboolean z_szig_write(ZStream *stream, GIOCondition cond, gpointer user_data);

static void 
z_szig_thread_started(ZThread *self G_GNUC_UNUSED, gpointer user_data G_GNUC_UNUSED)
{
  GValue value;
  
  value.g_type = G_TYPE_LONG;
  value.data[0].v_long = 1;
  value.data[1].v_long = 1;
  z_szig_add("zorp.threads", &value);
}

static void 
z_szig_thread_stopped(ZThread *self G_GNUC_UNUSED, gpointer user_data G_GNUC_UNUSED)
{
  GValue value;
  
  value.g_type = G_TYPE_LONG;
  value.data[0].v_long = 1;
  value.data[1].v_long = 1;
  z_szig_sub("zorp.threads", &value);
}

void
z_szig_init(const gchar *instance_name)
{
  ZorpSZIGNode *root_node = g_new0(ZorpSZIGNode, 1);
  ZSockAddr *sockaddr;
  gchar buf[4096];
  ZIOListen *listen;
  
  z_enter();
  
  g_type_init();
  z_thread_register_start_callback((GFunc) z_szig_thread_started, NULL);
  z_thread_register_stop_callback((GFunc) z_szig_thread_stopped, NULL);
  root_node->lock = g_mutex_new();
  root = g_node_new(root_node);
  
  g_snprintf(buf, sizeof(buf), "%s.%s", UNIX_NAME, instance_name);
  
  sockaddr = z_sockaddr_unix_new(buf);
  
  listen = z_io_listen_new(sockaddr, FALSE, 255, z_szig_accept_callback, NULL);
  if (listen)
    z_io_listen_start(listen);
  z_leave();
}

GNode *
z_szig_search_node(const gchar *name, gboolean create)
{
  gchar **split;
  guint i;
  GNode *ret = NULL;
  ZorpSZIGNode *search = NULL;
  GNode *node;
  GNode *old_node;
  gboolean found;
  
  z_enter();
  if (name == NULL)
    {
      z_leave();
      return NULL;
    }
  
  split = g_strsplit(name, ".", 0);
  if (strcmp(split[0], "zorp") != 0)
    {
      g_strfreev(split);
      z_leave();
      return NULL;
    }
  
  i = 1;
  old_node = root;
  node = root;
  found = TRUE;
  while (split[i] != NULL && found)
    {
      found = FALSE;
      node = g_node_first_child(old_node);
      if (node == NULL && create)
        {
          search = g_new0(ZorpSZIGNode, 1);
          search->lock = g_mutex_new();
          g_strlcpy(search->name, split[i], sizeof(search->name));
          g_node_append_data(old_node, search);
          node = g_node_last_child(old_node);
        }
      while (!found && node)
        {
          search = (ZorpSZIGNode *) node->data;
          if (strncmp(split[i], search->name, sizeof(search->name)) == 0)
            {
              found = TRUE;
              old_node = node;
            }
          else
            {
              node = g_node_next_sibling(node);
            }
          
          if (node == NULL && create)
            {
              search = g_new0(ZorpSZIGNode, 1);
              search->lock = g_mutex_new();
              g_strlcpy(search->name, split[i], sizeof(search->name));
              g_node_append_data(old_node, search);
              node = g_node_last_child(old_node);
            }
        }
      i++;
    }
  
  if (found)
    {
      ret = node;
    }
  g_strfreev(split);
  z_leave();
  return ret;
}

GValue *
z_szig_get_value(const gchar *name)
{
  GNode *g_node;
  ZorpSZIGNode *node;
  
  z_enter();
  g_node = z_szig_search_node(name, FALSE);
  
  if (g_node)
    {
      node = (ZorpSZIGNode *) g_node->data;
      if (node)
        {
          z_leave();
          return node->value;
        }
    }
  
  z_leave();
  return NULL;
}

gchar *
z_szig_get_child(const gchar *name)
{
  GNode *g_node, *g_node2;
  ZorpSZIGNode *node = NULL;
  
  z_enter();
  g_node = z_szig_search_node(name, FALSE);
  
  if (g_node)
    {
      g_node2 = g_node_first_child(g_node);
      
      if (g_node2)
        {
          node = (ZorpSZIGNode *) g_node2->data;
        }
    }
  
  if (node)
    {
      return g_strdup(node->name);
    }
  
  z_leave();
  return NULL;
}

gchar *
z_szig_get_sibling(const gchar *name)
{
  GNode *g_node;
  ZorpSZIGNode *node = NULL;
  
  z_enter();
  g_node = z_szig_search_node(name, FALSE);
  
  if (g_node)
    {
      g_node = g_node_next_sibling(g_node);
      
      if (g_node)
        node = (ZorpSZIGNode *) g_node->data;
    }
  
  if (node)
    {
      return g_strdup(node->name);
    }
  
  z_leave();
  return NULL;
}

gboolean
z_szig_set(const gchar *name, GValue *value)
{
  GNode *g_node;
  ZorpSZIGNode *node;
  GValue *node_value;
  
  z_enter();
  g_node = z_szig_search_node(name, TRUE);
  node = (ZorpSZIGNode *) g_node->data;
  if (node != NULL)
    {
      g_mutex_lock(node->lock);
      node_value = node->value;
      if (node_value == NULL)
        {
          node->value = g_new0(GValue, 1);
          g_value_init(node->value, value->g_type);
          g_value_copy(value, node->value);
        }
      else
        {
          g_value_unset(node->value);
          g_free(node->value);
          node->value = value;
        }
      g_mutex_unlock(node->lock);
      return TRUE;
    }
  
  z_leave();
  return FALSE;
}

gboolean
z_szig_add(const gchar *name, GValue *value)
{
  GNode *g_node;
  ZorpSZIGNode *node;
  glong value_long;
  double value_float;
  
  z_enter();
  g_node = z_szig_search_node(name, TRUE);
  if (g_node)
    {
      node = (ZorpSZIGNode *) g_node->data;
      if (node != NULL)
        {
          g_mutex_lock(node->lock);
          if (node->value == NULL)
            {
              node->value = g_new0(GValue, 1);
              g_value_init(node->value, G_VALUE_TYPE(value));
              g_value_copy(value, node->value);
            }
          else
            {
              if (G_VALUE_TYPE(node->value) == G_VALUE_TYPE(value))
                {
                  switch (G_VALUE_TYPE(value))
                    {
                    case G_TYPE_LONG:
                      value_long = g_value_get_long(node->value);
                      value_long += g_value_get_long(value);
                      g_value_set_long(node->value, value_long);
                      break;
                    case G_TYPE_FLOAT:
                      value_float = g_value_get_float(node->value);
                      value_float += g_value_get_float(value);
                      g_value_set_float(node->value, value_float);
                      break;
                    default:
                      z_leave();
                      return FALSE;
                    }
                }
              else
                {
                  z_leave();
                  return FALSE;
                }
            }
          g_mutex_unlock(node->lock);
          return TRUE;
        }
    }
  
  z_leave();
  return FALSE;
}

gboolean
z_szig_sub(const gchar *name, GValue *value G_GNUC_UNUSED)
{
  GNode *g_node;
  ZorpSZIGNode *node;
  glong value_long;
  double value_float;
  
  z_enter();
  g_node = z_szig_search_node(name, TRUE);
  node = (ZorpSZIGNode *) g_node->data;
  if (node != NULL)
    {
      g_mutex_lock(node->lock);
      if (node->value == NULL)
        {
          node->value = g_new0(GValue, 1);
          g_value_init(node->value, value->g_type);
          g_value_copy(value, node->value);
        }
      else
        {
          if (G_VALUE_TYPE(node->value) == G_VALUE_TYPE(value))
            {
              switch (G_VALUE_TYPE(value))
                {
                case G_TYPE_LONG:
                  value_long = g_value_get_long(node->value);
                  value_long -= g_value_get_long(value);
                  g_value_set_long(node->value, value_long);
                  break;
                case G_TYPE_FLOAT:
                  value_float = g_value_get_float(node->value);
                  value_float -= g_value_get_float(value);
                  g_value_set_float(node->value, value_float);
                  break;
                default:
                  z_leave();
                  return FALSE;
                }
            }
          else
            {
              z_leave();
              return FALSE;
            }
        }
      g_mutex_unlock(node->lock);
      return TRUE;
    }
  
  z_leave();
  return FALSE;
}

gboolean
z_szig_handle_command(ZStream *stream, gchar *cmd, guint cmdlen)
{
  gchar name[4096];
  GValue *value;
  ZSZIGWriteBuf *buf = NULL;
  gchar *ret;
  guint len;
  
  if (cmdlen < 10 || cmdlen > sizeof(name))
    {
      z_leave();
      return FALSE;
    }
  
  memmove(name, cmd + 9, cmdlen - 9);
  name[cmdlen - 9] = 0;
  
  buf = g_new0(ZSZIGWriteBuf, 1);
  if (strncmp(cmd, "GETVALUE ", 9) == 0)
    {
      value = z_szig_get_value(name);
      if (value)
        {
          switch (G_VALUE_TYPE(value))
            {
              case G_TYPE_LONG:
              g_snprintf(buf->value,
                         sizeof(buf->value),
                         "%s: %ld",
                         name,
                         g_value_get_long(value));
              break;
            case G_TYPE_FLOAT:
              g_snprintf(buf->value,
                         sizeof(buf->value),
                         "%s: %f",
                         name,
                         g_value_get_float(value));
              break;
            case G_TYPE_STRING:
              g_snprintf(buf->value,
                         sizeof(buf->value),
                         "%s: '%s'",
                         name,
                         g_value_get_string(value));
              break;
            case G_TYPE_BOOLEAN:
              g_snprintf(buf->value,
                         sizeof(buf->value),
                         "%s: %s",
                         name,
                         g_value_get_boolean(value) ? "TRUE": "FALSE");
              break;
            default:
              g_free(buf);
              z_leave();
              return FALSE;
            }
        }
      else
        {
          g_snprintf(buf->value,
                     sizeof(buf->value),
                     "%s: %s",
                     name,
                     "None");
        }
    }
  else if (strncmp(cmd, "GETCHILD ", 9) == 0)
    {
      ret = z_szig_get_child(name);
      if (ret)
        {
          g_strlcpy(buf->value, ret, sizeof(buf->value));
          g_free(ret);
        }
      else
        g_strlcpy(buf->value, "None", sizeof(buf->value));
    }
  else if (strncmp(cmd, "GETSBLNG ", 9) == 0)
    {
      ret = z_szig_get_sibling(name);
      if (ret)
        {
          g_strlcpy(buf->value, ret, sizeof(buf->value));
          g_free(ret);
        }
      else
        g_strlcpy(buf->value, "None", sizeof(buf->value));
    }
  else
    {
      g_free(buf);
      z_leave();
      return FALSE;
    }
  len = strlen(buf->value);
  buf->value[len] = '\n';
  buf->value[len + 1] = 0;
  z_stream_set_callback(stream, Z_STREAM_FLAG_WRITE, z_szig_write, buf, NULL);
  z_stream_set_cond(stream, Z_STREAM_FLAG_READ, FALSE);
  z_stream_set_cond(stream, Z_STREAM_FLAG_WRITE, TRUE);
  z_leave();
  return TRUE;
}

gboolean
z_szig_read(ZStream *stream,
       GIOCondition  cond G_GNUC_UNUSED,
           gpointer  user_data G_GNUC_UNUSED)
{
  gchar *buf;
  guint buflen = NAME_LEN;
  GIOStatus rc;
  
  z_enter();
  rc = z_stream_line_get(stream, &buf, &buflen, NULL);
  if (rc == G_IO_STATUS_NORMAL)
    {
      if (z_szig_handle_command(stream, buf, buflen))
        {
          z_leave();
          return TRUE;
        }
    }
  else if (rc == G_IO_STATUS_AGAIN)
    {
      z_leave();
      return TRUE;
    }
  
  z_stream_detach_source(stream);
  z_stream_close(stream, NULL);
  z_stream_unref(stream);
  z_leave();
  return TRUE;
}

gboolean
z_szig_write(ZStream *stream,
             GIOCondition  cond G_GNUC_UNUSED,
             gpointer  user_data)
{
  ZSZIGWriteBuf *buf = (ZSZIGWriteBuf *)user_data;
  GIOStatus ret;
  guint length;
  
  z_enter();
  ret = z_stream_write(stream,
                       buf->value + buf->pos,
                       strlen(buf->value + buf->pos),
                       &length,
                       NULL);
  if (ret == G_IO_STATUS_NORMAL)
    {
      if (strlen(buf->value + buf->pos) != length)
        {
          buf->pos += length;
          z_leave();
          return TRUE;
        }
    }
  else if (ret == G_IO_STATUS_AGAIN)
    {
      z_leave();
      return TRUE;
    }
  z_stream_set_cond(stream, Z_STREAM_FLAG_READ, TRUE);
  z_stream_set_cond(stream, Z_STREAM_FLAG_WRITE, FALSE);
  g_free(buf);
  z_leave();
  return TRUE;
}


gboolean
z_szig_accept_callback(int fd,
                       ZSockAddr *client,
                       gboolean  last_connection G_GNUC_UNUSED,
                       gpointer  user_data G_GNUC_UNUSED)
{
  ZStream *stream;
  ZStream *streamline;
  
  z_fd_set_nonblock(fd, TRUE);
  
  stream = z_stream_new(fd, "szig");
  
  streamline = z_stream_line_new(stream, NAME_LEN, ZRL_EOL_NL);
  z_stream_unref(stream);
  
  z_stream_set_cond(streamline, Z_STREAM_FLAG_READ, TRUE);
  z_stream_set_callback(streamline, Z_STREAM_FLAG_READ, z_szig_read, NULL, NULL);
  
  z_stream_attach_source(streamline, g_main_context_default());
  z_sockaddr_unref(client);
  return TRUE;
}
