/***************************************************************************
 *
 * COPYRIGHTHERE
 *
 * $Id: tpsocket.c,v 1.9.2.11 2003/09/17 13:16:01 sasa Exp $
 *
 * Author  : Bazsi
 * Auditor :
 * Last audited version:
 * Notes:
 *
 ***************************************************************************/

#include <zorp/zorp.h>
#include <zorp/tpsocket.h>
#include <zorp/log.h>
#include <zorp/cap.h>
#include <zorp/sysdep.h>
#include <zorp/nfiptproxy-kernel.h>

#include <stdlib.h>

const gchar *auto_bind_ip = "1.2.3.4";

gint
z_tp_autobind(gint fd)
{
  struct sockaddr_in sa;
  socklen_t salen = sizeof(sa);
  struct in_addr ab_addr;
  
  memset(&sa, 0, sizeof(sa));
  
  z_inet_aton(auto_bind_ip, &ab_addr);
  if (getsockname(fd, (struct sockaddr *) &sa, &salen) != -1)
    {
      /* already bound */
      if (sa.sin_family == AF_INET && sa.sin_addr.s_addr == ab_addr.s_addr)
        {
          return 0;
        }
    }
  
  sa.sin_family = AF_INET;
  sa.sin_addr = ab_addr;
  z_inet_aton(auto_bind_ip, &sa.sin_addr);
  /* */
  sa.sin_port = htons(0);
  if (bind(fd, (struct sockaddr *) &sa, sizeof(sa)) == -1)
    {
      /*LOG
        This message attempt when bind syscall is failed for the
        given reason.
       */
      z_log(NULL, CORE_ERROR, 3, "Error autobinding transparent socket; fd='%d', ip='%s', error='%m'", fd, auto_bind_ip);
      errno = EADDRNOTAVAIL;
      return -1;
    }
  return 0;
}

#include <string.h>

#if ENABLE_NETFILTER_TPROXY
static gint
z_do_tp_bind(gint fd, struct sockaddr *sa, socklen_t salen)
{
  struct in_tproxy itp;
  socklen_t itplen = sizeof(itp);
  
  if (sa->sa_family != AF_INET)
    return z_do_ll_bind(fd, sa, salen);
  
  if (bind(fd, sa, salen) != -1)
    {
      /* we could bind successfully, it was a local address */

      if (((struct sockaddr_in *) sa)->sin_addr.s_addr != 0)
        {
          /* we don't want NAT, only that -m tproxy matches our session */
          itp.itp_fport = 0;
          itp.itp_faddr.s_addr = 0;
          if (setsockopt(fd, SOL_IP, IP_TPROXY_ASSIGN, &itp, itplen) == -1)
    	    {
    	      /*LOG
    	        This message attempt when setsockopt() syscall failed
    	        for the given reason.
    	       */
              z_log(NULL, CORE_ERROR, 3, "Error in setsockopt(SOL_IP, IP_TPROXY_ASSIGN), netfilter-tproxy support required; fd='%d', error='%m'", fd);
              return 0;
            }
        }
      return 0;
    }
  else if (errno != EADDRNOTAVAIL)
    return -1;
    
  /* we couldn't bind because the address was not available, try transparent proxy tricks */
  if (z_tp_autobind(fd) == -1)
    {
      /* autobind was not successful either */
      return -1;
    }
  itp.itp_fport = ((struct sockaddr_in *) sa)->sin_port;
  itp.itp_faddr = ((struct sockaddr_in *) sa)->sin_addr;
  if (setsockopt(fd, SOL_IP, IP_TPROXY_ASSIGN, &itp, itplen) == -1)
    {
      /*LOG
        This message attempt when setsockopt() syscall failed
        for the given reason.
       */
      z_log(NULL, CORE_ERROR, 3, "Error in setsockopt(SOL_IP, IP_TPROXY_ASSIGN), netfilter-tproxy support required; fd='%d', error='%m'", fd);
      return -1;
    }
  return 0;
}

static gint
z_do_tp_listen(gint fd, gint backlog, gboolean accept_one)
{
  unsigned int flags;
  socklen_t flagslen = sizeof(flags);
  
  z_enter();
  if (z_do_ll_listen(fd, backlog, accept_one) == -1)
    {
      z_leave();
      return -1;
    }
  if (getsockopt(fd, SOL_IP, IP_TPROXY_FLAGS, &flags, &flagslen) == -1)
    {
      /* this is not a real problem, as it might mean that this socket is
         a unix domain socket */
      z_leave();
      return 0;
    }
  if (flags & ITP_MARK)
    {
      z_leave();
      return 0;
    }
    
  flags = ITP_LISTEN | (accept_one ? ITP_ONCE : 0);
  if (setsockopt(fd, SOL_IP, IP_TPROXY_FLAGS, &flags, sizeof(flags)) == -1)
    {
      /*LOG
        This message attempt when setsockopt() syscall failed
        for the given reason.
       */
      z_log(NULL, CORE_ERROR, 3, "Error in setsockopt(SOL_IP, IP_TPROXY_FLAGS), netfilter-tproxy support required; fd='%d', error='%m'", fd);
      z_leave();
      return -1;
    }
  z_leave();
  return 0;
}

static gint
z_do_tp_connect(gint fd, struct sockaddr *sa, socklen_t salen)
{
  unsigned int flags;
  struct in_tproxy itp;
  socklen_t itplen = sizeof(itp);
  
  if (getsockopt(fd, SOL_IP, IP_TPROXY_QUERY, &itp, &itplen) == -1)
    {
      return z_do_ll_connect(fd, sa, salen);
    }
  flags = ITP_CONNECT | ITP_ONCE;
  if (setsockopt(fd, SOL_IP, IP_TPROXY_FLAGS, &flags, sizeof(flags)) == -1)
    {
      /*LOG
        This message attempt when setsockopt() syscall failed
        for the given reason.
       */
      z_log(NULL, CORE_ERROR, 3, "Error in setsockopt(SOL_IP, IP_TPROXY_FLAGS), netfilter-tproxy support required; fd='%d', error='%m'", fd);
      return -1;
    }
  return z_do_ll_connect(fd, sa, salen);
}

#ifndef SO_ORIGINAL_DST
#define SO_ORIGINAL_DST 80
#endif

static gint
z_do_tp_getdestname(gint fd, struct sockaddr *sa, socklen_t *salen)
{
  struct sockaddr_in *sin = (struct sockaddr_in *) sa;
  socklen_t sinlen;
  
  if (*salen < sizeof(*sin))
    {
      errno = -EINVAL;
      return -1;
    }
  
  sinlen = sizeof(*sin);
  if (getsockopt(fd, SOL_IP, SO_ORIGINAL_DST, sin, &sinlen) >= 0)
    {
      *salen = sinlen;
      return 0;
    }
  /* Linux 2.2 fallback */
  if (getsockname(fd, sa, salen) >= 0)
    {
      return 0;
    }
  return -1;
}
  
static gint
z_do_tp_getsockname(gint fd, struct sockaddr *sa, socklen_t *salen)
{
  struct in_tproxy itp;
  socklen_t itplen = sizeof(itp);
  struct sockaddr_in *sin = (struct sockaddr_in *) sa;
  
  if (getsockopt(fd, SOL_IP, IP_TPROXY_QUERY, &itp, &itplen) == -1)
    {
      return z_do_ll_getsockname(fd, sa, salen);
    }
  if (*salen < sizeof(*sin))
    {
      errno = EFAULT;
      return -1;
    }
  sin->sin_family = AF_INET;
  sin->sin_addr = itp.itp_faddr;
  sin->sin_port = itp.itp_fport;
  *salen = sizeof(*sin);  
  return 0;
}

ZSocketFuncs z_tp_socket_funcs = 
{
  z_do_tp_bind,
  z_do_ll_accept,
  z_do_tp_connect,
  z_do_tp_listen,
  z_do_tp_getsockname,
  z_do_ll_getpeername,
  z_do_tp_getdestname
};


#endif

gboolean
z_tp_socket_init(gint sysdep_tproxy)
{
  /* Linux22 doesn't need any magic */
  gboolean res = TRUE;

#if ENABLE_NETFILTER_TPROXY
  if (sysdep_tproxy == Z_SD_TPROXY_NETFILTER)
    {
      gint fd;
      
      fd = socket(AF_INET, SOCK_STREAM, 0);
      if (z_tp_autobind(fd) == -1)
        {
          z_log(NULL, CORE_ERROR, 3, "Binding to dummy interface failed, please create one and pass --autobind-ip parameter; autobind='%s'", auto_bind_ip);
        }
      else
        {
          socket_funcs = &z_tp_socket_funcs;
        }
      close(fd);
    }
  else
#endif
  if (sysdep_tproxy != Z_SD_TPROXY_LINUX22)
    {
      z_log(NULL, CORE_ERROR, 0, "Required transparency support not compiled in (TCP); sysdep_tproxy='%d'", sysdep_tproxy);
      res = FALSE;
    }
  return res;
}
