/***************************************************************************
 *
 * COPYRIGHTHERE
 *
 * $Id: ftp.c,v 1.210.2.37 2004/03/11 16:20:09 bazsi Exp $
 *
 * Author:  Andras Kis-Szabo <kisza@sch.bme.hu>
 * Author:  Attila SZALAY <sasa@balabit.hu>
 * Auditor:
 * Last audited version:
 * Notes:
 *
 ***************************************************************************/

#include "ftp.h"

#include <zorp/zorp.h>
#include <zorp/registry.h>
#include <zorp/sockaddr.h>
#include <zorp/pysockaddr.h>
#include <zorp/policy.h>
#include <zorp/log.h>
#include <zorp/thread.h>
#include <zorp/io.h>
#include <zorp/streamfd.h>
#include <zorp/pystream.h>
#include <zorp/proxy.h>

#include <ctype.h>
#include <assert.h>

GHashTable *ftp_command_hash = NULL;
GHashTable *ftp_answer_hash  = NULL;

#define SIDE_TO_STRING(side) side == EP_CLIENT ? "client" : side == EP_SERVER ? "server" : "unknown"

void ftp_data_reset(FtpProxy *self);
gboolean ftp_data_abort(FtpProxy *self);
gboolean ftp_stream_write(FtpProxy *self,
                              char  side,
                            guchar *line,
                             guint  length);
void ftp_proxy_free(ZProxy * s);

static gboolean
ftp_connect_server_event(FtpProxy *self, gchar *hostname, guint port)
{
  ZSockAddr *client_local, *server_local;
  gchar tmpip[16];
  
  z_proxy_enter(self);
  if (!z_proxy_connect_server_event(&self->super, hostname, port))
    {
      z_proxy_leave(self);
      return FALSE;
    }

  /*
    This must be after connect server event because we want to
    wait for all router, and NATs to change addresses.
    We set up the address pools, where we link to connect and accept
    sockaddrs.
   */

  if (!z_proxy_get_addresses(&self->super, NULL, NULL, &client_local, NULL, &server_local, NULL))
    {
      z_proxy_leave(self);
      return FALSE;
    }

  z_inet_ntoa(tmpip, sizeof(tmpip), ((struct sockaddr_in *) &client_local->sa)->sin_addr);

  if (self->data_port_min && self->data_port_max)
    self->data_local_buf[EP_CLIENT] = z_sockaddr_inet_range_new(tmpip,
                                                                self->data_port_min,
                                                                self->data_port_max);
  else
    self->data_local_buf[EP_CLIENT] = z_sockaddr_inet_new(tmpip, 0);
      
  z_inet_ntoa(tmpip, sizeof(tmpip), ((struct sockaddr_in *) &server_local->sa)->sin_addr);

  if (self->data_port_min != 0 && self->data_port_max != 0)
    self->data_local_buf[EP_SERVER] = z_sockaddr_inet_range_new(tmpip, self->data_port_min, self->data_port_max);
  else
    self->data_local_buf[EP_SERVER] = z_sockaddr_inet_new(tmpip, 0);
  
  z_sockaddr_unref(client_local);
  z_sockaddr_unref(server_local);
  
  z_proxy_leave(self);
  return TRUE;
}

static void
ftp_state_set(FtpProxy *self, guint order)
{
  z_proxy_enter(self);
  z_stream_set_cond(self->super.endpoints[order],
                    Z_STREAM_FLAG_READ,
                    TRUE);

  z_stream_set_cond(self->super.endpoints[1 - order],
                    Z_STREAM_FLAG_READ,
                    FALSE);
  z_proxy_leave(self);
}

void
ftp_state_both(FtpProxy *self)
{
  z_proxy_enter(self);
  z_stream_set_cond(self->super.endpoints[EP_CLIENT],
                    Z_STREAM_FLAG_READ,
                    TRUE);

  z_stream_set_cond(self->super.endpoints[EP_SERVER],
                    Z_STREAM_FLAG_READ,
                    TRUE);
  z_proxy_leave(self);
}

static ZPolicyObj *
ftp_data_end(ZProxy *s, ZPolicyObj *args G_GNUC_UNUSED)
{
  FtpProxy *self = (FtpProxy *) s;
  
  z_proxy_enter(self);
  
  g_mutex_lock(self->lock);
  self->data_state = FTP_DATA_CANCEL;
  g_mutex_unlock(self->lock);

  z_poll_wakeup(self->poll);

  z_policy_var_ref(z_policy_none);
  z_proxy_leave(self);
  return z_policy_none;
}  

static gboolean
ftp_data_client_accepted(ZConnection *conn, gpointer user_data)
{
  FtpProxy *self = (FtpProxy *) user_data;
  
  z_proxy_enter(self);
  
  if (self->data_stream[EP_CLIENT] || self->data_state == FTP_DATA_CANCEL)
    {
      return FALSE;
    }
  
  g_mutex_lock(self->lock);
  
  if (conn && conn->stream)
    {
      z_stream_ref(conn->stream);
      self->data_stream[EP_CLIENT] = conn->stream;
      self->data_remote[EP_CLIENT] = z_sockaddr_ref(conn->remote);
      self->data_state |= FTP_DATA_CLIENT_READY;
    }
  else
    {
      self->data_stream[EP_CLIENT] = NULL;
      self->data_remote[EP_CLIENT] = NULL;
      self->data_state = FTP_DATA_CANCEL;
    }

  if (conn)
    z_connection_destroy(conn, FALSE);

  g_mutex_unlock(self->lock);

  z_poll_wakeup(self->poll);
  
  z_proxy_leave(self);
  return TRUE;
}

static gboolean
ftp_data_server_accepted(ZConnection *conn, gpointer user_data)
{
  FtpProxy *self = (FtpProxy *) user_data;
  
  z_proxy_enter(self);
  
  if (self->data_stream[EP_SERVER] || self->data_state == FTP_DATA_CANCEL)
    {
      return FALSE;
    }
  
  g_mutex_lock(self->lock);
  if (conn && conn->stream)
    {
      z_stream_ref(conn->stream);
      self->data_stream[EP_SERVER] = conn->stream;
      self->data_remote[EP_SERVER] = z_sockaddr_ref(conn->remote);
      self->data_state |= FTP_DATA_SERVER_READY;
    }
  else
    {
      self->data_stream[EP_SERVER] = NULL;
      self->data_remote[EP_SERVER] = NULL;
      self->data_state = FTP_DATA_CANCEL;
    }

  if (conn)
    z_connection_destroy(conn, FALSE);

  g_mutex_unlock(self->lock);

  z_poll_wakeup(self->poll);
  
  z_proxy_leave(self);
  return TRUE;
}

void
ftp_data_client_connected(ZConnection *conn, gpointer user_data)
{
  FtpProxy *self = (FtpProxy *) user_data;
  
  z_proxy_enter(self);
  
  g_mutex_lock(self->lock);
  if (!(self->data_state & FTP_DATA_CLIENT_READY) &&
      self->data_state != FTP_DATA_CANCEL)
    {
      if (conn && conn->stream)
        {
          z_stream_ref(conn->stream);
          self->data_stream[EP_CLIENT] = conn->stream;

          z_sockaddr_unref(self->data_remote[EP_CLIENT]);
          self->data_remote[EP_CLIENT] = z_sockaddr_ref(conn->remote);
  
          self->data_state |= FTP_DATA_CLIENT_READY;
        }
      else
        {
          self->data_state = FTP_DATA_CANCEL;
        }
      
      if (conn)
        {
          z_connection_destroy(conn, FALSE);
          conn = NULL;
        }
  
      z_poll_wakeup(self->poll);
    }
  g_mutex_unlock(self->lock);    

  if (conn)
    z_connection_destroy(conn, TRUE);

  z_proxy_leave(self);
}

void
ftp_data_server_connected(ZConnection *conn, gpointer user_data)
{
  FtpProxy *self = (FtpProxy *) user_data;
  
  z_proxy_enter(self);
  
  g_mutex_lock(self->lock);
  if (!(self->data_state & FTP_DATA_SERVER_READY) &&
      self->data_state != FTP_DATA_CANCEL)
    {
  
      if (conn && conn->stream)
        {
          z_stream_ref(conn->stream);
          self->data_stream[EP_SERVER] = conn->stream;

          z_sockaddr_unref(self->data_remote[EP_SERVER]);
          self->data_remote[EP_SERVER] = z_sockaddr_ref(conn->remote);
  
          self->data_state |= FTP_DATA_SERVER_READY;
        }
      else
        {
          self->data_state = FTP_DATA_CANCEL;
        }
  
      z_poll_wakeup(self->poll);
      
      if (conn)
        {
          z_connection_destroy(conn, FALSE);
          conn = NULL;
        }
    }
  g_mutex_unlock(self->lock);

  if (conn)
    z_connection_destroy(conn, TRUE);
  
  z_proxy_leave(self);
}

ZAttachCallback data_attach_callbacks[] =
{
  ftp_data_client_connected, 
  ftp_data_server_connected
};

ZDispatchCallback data_accept_callbacks[] =
{
  ftp_data_client_accepted, 
  ftp_data_server_accepted
};

gboolean
ftp_data_prepare(FtpProxy *self, gint side, gchar mode)
{
  ZDispatchParams dpparam;
  ZAttachParams aparam;
  ZSockAddr *tmpaddr;
  gchar tmpip[16];

  z_proxy_enter(self);
  /* FIXMEE Correct handling of not freed streams! */
  self->data_stream[side] = NULL;
//  self->data_stream[EP_SERVER] = NULL;

  if (mode == 'L')
    {
      memset(&dpparam, 0, sizeof(dpparam));
      dpparam.tcp.accept_one = FALSE;
      dpparam.tcp.backlog = 1;
      z_proxy_ref(&self->super);
      if (self->data_listen[side] != NULL)
        {
          z_proxy_log(self, FTP_ERROR, 4, "Internal error. Previous Listener not unregistered; side='%s'", SIDE_TO_STRING(side));
          z_dispatch_unregister(self->data_listen[side]);
        }

      self->data_listen[side] = z_dispatch_register(self->super.session_id,
                                                    ZD_PROTO_TCP,
                                                    self->data_local_buf[side],
                                                    &tmpaddr,
                                                    ZD_PRI_RELATED,
                                                    &dpparam,
                                                    data_accept_callbacks[side],
                                                    self,
                                                    (GDestroyNotify)z_proxy_unref);
      if (!self->data_listen[side])
        {
          z_proxy_unref(&self->super);
          z_proxy_leave(self);
          return FALSE;
        }
      self->data_local[side] = tmpaddr;
      
      if (self->data_connect[side])
        {
          /*LOG
            Placeholder ftp.c.2
           */
          z_proxy_log(self, FTP_ERROR, 4, "Internal error. Previous Attach not unregistered; side='%s'", SIDE_TO_STRING(side));
          z_attach_cancel(self->data_connect[side]);
          z_attach_unref(self->data_connect[side]);
          self->data_connect[side] = NULL;
          z_proxy_leave(self);
          return FALSE;
        }
    }
  else if (mode == 'C')
    {
      if (side == EP_CLIENT)
        {
          tmpaddr = self->data_local_buf[side];
          z_inet_ntoa(tmpip, sizeof(tmpip), ((struct sockaddr_in *) &tmpaddr->sa)->sin_addr);
          self->data_local[side] = z_sockaddr_inet_new(tmpip, 20);
        }
      else
        self->data_local[side] = z_sockaddr_ref(self->data_local_buf[side]);
      memset(&aparam, 0, sizeof(aparam));
      aparam.tcp.timeout = -1;
      if (self->data_connect[side] != NULL)
        {
          z_proxy_log(self, FTP_ERROR, 4, "Internal error. Previous Attach not unregistered; side='%s'", SIDE_TO_STRING(side));
          z_attach_cancel(self->data_connect[side]);
          z_attach_unref(self->data_connect[side]);
        }
      z_proxy_ref(&self->super);
      self->data_connect[side] = z_attach_new(self->super.session_id,
                                              ZD_PROTO_TCP,
                                              self->data_local[side],
                                              self->data_remote[side],
                                              &aparam,
                                              data_attach_callbacks[side],
                                              self,
                                              (GDestroyNotify)z_proxy_unref);

      z_sockaddr_unref(self->data_local[side]);
      self->data_local[side] = NULL;
      if (!self->data_connect[side])
        {
          z_proxy_unref(&self->super);
          z_proxy_leave(self);
          return FALSE;
        }

      if (self->data_listen[side])
        {
          /*LOG
            Placeholder ftp.c.1
           */
          z_proxy_log(self, FTP_ERROR, 4, "Internal error. Previous Listener not unregistered; side='%s'", SIDE_TO_STRING(side));
          z_dispatch_unregister(self->data_listen[side]);
          self->data_listen[side] = NULL;
        }
    }

  z_proxy_leave(self);
  return TRUE;
}

void
ftp_data_reset(FtpProxy *self G_GNUC_UNUSED)
{
  z_proxy_enter(self);
  
  /*LOG
    Placeholder ftp.c.3
   */
  z_proxy_log(self, FTP_DEBUG, 6, "Resetting data connection;");
  if (self->data_connect[EP_CLIENT])
    {
      z_attach_cancel(self->data_connect[EP_CLIENT]);
      z_attach_unref(self->data_connect[EP_CLIENT]);
      self->data_connect[EP_CLIENT] = NULL;
    }
  
  if (self->data_connect[EP_SERVER])
    {
      z_attach_cancel(self->data_connect[EP_SERVER]);
      z_attach_unref(self->data_connect[EP_SERVER]);
      self->data_connect[EP_SERVER] = NULL;
    }
  
  if (self->data_listen[EP_CLIENT])
    {
      z_dispatch_unregister(self->data_listen[EP_CLIENT]);
      self->data_listen[EP_CLIENT] = NULL;
    }
  
  if (self->data_listen[EP_SERVER])
    {
      z_dispatch_unregister(self->data_listen[EP_SERVER]);
      self->data_listen[EP_SERVER] = NULL;
    }
  
  if (self->data_stream[EP_CLIENT])
    {
      z_stream_shutdown(self->data_stream[EP_CLIENT], SHUT_RDWR, NULL);
      z_stream_close(self->data_stream[EP_CLIENT], NULL);
      z_stream_unref(self->data_stream[EP_CLIENT]);
      self->data_stream[EP_CLIENT] = NULL;
    }
  
  if (self->data_stream[EP_SERVER])
    {
      z_stream_shutdown(self->data_stream[EP_SERVER], SHUT_RDWR, NULL);
      z_stream_close(self->data_stream[EP_SERVER], NULL);
      z_stream_unref(self->data_stream[EP_SERVER]);
      self->data_stream[EP_SERVER] = NULL;
    }
  
  g_mutex_lock(self->lock);
  
  if (self->data_remote[EP_CLIENT])
    {
      z_sockaddr_unref(self->data_remote[EP_CLIENT]);
      self->data_remote[EP_CLIENT] = NULL;
    }
  
  if (self->data_remote[EP_SERVER])
    {
      z_sockaddr_unref(self->data_remote[EP_SERVER]);
      self->data_remote[EP_SERVER] = NULL;
    }
  
  if (self->data_local[EP_CLIENT])
    {
      z_sockaddr_unref(self->data_local[EP_CLIENT]);
      self->data_local[EP_CLIENT] = NULL;
    }
  
  if (self->data_local[EP_SERVER])
    {
      z_sockaddr_unref(self->data_local[EP_SERVER]);
      self->data_local[EP_SERVER] = NULL;
    }

  self->data_state = 0;
  g_mutex_unlock(self->lock);

  z_stream_set_cond(self->super.endpoints[EP_CLIENT],
                   Z_STREAM_FLAG_PRI,
                   FALSE);

  switch(self->oldstate)
    {
      case FTP_SERVER_TO_CLIENT:
        ftp_state_set(self, EP_SERVER);
        self->state = self->oldstate;
        break;
      case FTP_CLIENT_TO_SERVER:
        ftp_state_set(self, EP_CLIENT);
        self->state = self->oldstate;
        break;
      default:
        break;
    }

  self->oldstate = 0;
  z_proxy_leave(self);
  return;
}

void
ftp_data_start(FtpProxy *self)
{
  z_enter();
  if (self->data_state & FTP_DATA_COMMAND_START)
    {
      /*LOG
        Placeholder ftp.c.4
       */
      z_proxy_log(self, FTP_ERROR, 4, "Internal error; error='Previous data connection isn't closed properly'");
      ftp_data_reset(self);
    }
  
  self->data_state |= FTP_DATA_COMMAND_START;

  z_stream_set_cond(self->super.endpoints[EP_CLIENT],
                   Z_STREAM_FLAG_READ,
                   FALSE);

  z_stream_set_cond(self->super.endpoints[EP_CLIENT],
                   Z_STREAM_FLAG_PRI,
                   TRUE);

  z_leave();
}

static gboolean
ftp_data_start_proxy(FtpProxy *self)
{
  ZPolicyObj *res;
  gboolean called;
  gboolean rc = TRUE;
  ZPolicyObj *streams[EP_MAX], *data_proxy;

  z_proxy_enter(self);
  
  z_policy_lock(self->super.thread);
  if (self->data_stream[EP_CLIENT] == 0 ||
      self->data_stream[EP_SERVER] == 0)
    {
      /*LOG
        Placeholder ftp.c.5
       */  
      z_policy_unlock(self->super.thread);
      z_proxy_log(self, FTP_ERROR, 2, "Internal error; error='ftp_data_start called but peers doesn`t connected'");
      z_proxy_leave(self);
      return FALSE;
    }
  if (self->command_desc && self->command_desc->need_data == 2)
    {
      streams[EP_CLIENT] = z_py_stream_new(self->data_stream[EP_CLIENT]);
      streams[EP_SERVER] = z_py_stream_new(self->data_stream[EP_SERVER]);
    }
  else if (self->command_desc && self->command_desc->need_data == 1)
    {
      streams[EP_CLIENT] = z_py_stream_new(self->data_stream[EP_SERVER]);
      streams[EP_SERVER] = z_py_stream_new(self->data_stream[EP_CLIENT]);
    }
  else
    {
      /*LOG
        Placeholder ftp.c.5.1
       */  
      z_policy_unlock(self->super.thread);
      z_proxy_log(self, FTP_ERROR, 2, "Internal error; error='ftp_data_start called but not needed data'");
      z_proxy_leave(self);
      return FALSE;
    }
  data_proxy = z_policy_getattr(self->super.handler, "data_proxy");
  res = z_policy_call(self->super.handler,
                      "stackProxy",
                      z_policy_var_build("(OOO)",
                                         streams[EP_CLIENT],
                                         streams[EP_SERVER],
                                         data_proxy),
                      &called,
                      self->super.session_id);
  Py_XDECREF(streams[EP_CLIENT]);
  Py_XDECREF(streams[EP_SERVER]);
  Py_XDECREF(data_proxy);
  if (!res)
    {
      z_stream_close(self->data_stream[EP_CLIENT], NULL);
      z_stream_close(self->data_stream[EP_SERVER], NULL);
      rc = FALSE;
    }
  else if (res == z_policy_none)
    rc = FALSE;

  z_policy_var_unref(res);
  z_policy_unlock(self->super.thread);

  z_stream_unref(self->data_stream[EP_SERVER]);
  self->data_stream[EP_SERVER] = NULL;

  z_stream_unref(self->data_stream[EP_CLIENT]);
  self->data_stream[EP_CLIENT] = NULL;

  z_proxy_leave(self);
  return rc;
}

static void
ftp_data_next_step(FtpProxy *self)
{
  gchar buf[4096];

  z_proxy_enter(self);
  g_mutex_lock(self->lock);

  z_proxy_trace(self, "A data_state = '%lx'", self->data_state);

  if ((self->data_state & FTP_DATA_COMMAND_START) &&
      !(self->data_state & FTP_DATA_SERVER_START))
    {
      z_proxy_cp(self);
      if (self->data_connect[EP_SERVER])
        {
          if (ftp_policy_bounce_check(self, EP_SERVER, self->data_remote[EP_SERVER], TRUE))
            {
              if (z_attach_start(self->data_connect[EP_SERVER]))
                self->data_local[EP_SERVER] = z_attach_get_local(self->data_connect[EP_SERVER]);
              else
                {
                  g_mutex_unlock(self->lock);
                  ftp_data_reset(self);
                  z_proxy_leave(self);
                  return ;
                }
            }
          else
            {
              /*LOG
                Placeholder ftp.c.6
               */
              z_proxy_log(self, FTP_POLICY, 3, "Possibly bounce attack; connect='TRUE', side='server', remote='%s'", z_sockaddr_format(self->data_remote[EP_SERVER], buf, sizeof(buf)));
              g_mutex_unlock(self->lock);
              ftp_data_reset(self);
              z_proxy_leave(self);
              return ;
            }
        }
      self->data_state |= FTP_DATA_SERVER_START;
    }
  
  if ((self->data_state & FTP_SERVER_FIRST_READY) == FTP_SERVER_FIRST_READY &&
       !(self->data_state & FTP_DATA_CLIENT_START))
    {
      z_proxy_cp(self);
      if (!self->data_connect[EP_SERVER])
        {
          if (!ftp_policy_bounce_check(self, EP_SERVER, self->data_remote[EP_SERVER], FALSE))
            {
              /*LOG
                Placeholder ftp.c.7
               */
              z_proxy_log(self, FTP_POLICY, 3, "Possibly bounce attack; connect='FALSE', side='server', remote='%s'", z_sockaddr_format(self->data_remote[EP_SERVER], buf, sizeof(buf)));
              g_mutex_unlock(self->lock);
              ftp_data_reset(self);
              z_proxy_leave(self);
              return;
            }
        }

      z_stream_set_cond(self->super.endpoints[EP_SERVER],
                       Z_STREAM_FLAG_READ,
                       FALSE);
                       
      self->data_state |= FTP_DATA_CLIENT_START;
      if (self->data_connect[EP_CLIENT])
        {
          if (ftp_policy_bounce_check(self, EP_CLIENT, self->data_remote[EP_CLIENT], TRUE))
            {
              if (z_attach_start(self->data_connect[EP_CLIENT]))
                self->data_local[EP_CLIENT] = z_attach_get_local(self->data_connect[EP_CLIENT]);
              else
                {
                  g_mutex_unlock(self->lock);
                  ftp_data_reset(self);
                  z_proxy_leave(self);
                  return;
                }
            }
          else
            {
              /*LOG
                Placeholder ftp.c.8
               */
              z_proxy_log(self, FTP_POLICY, 3, "Possibly bounce attack; connect='TRUE', side='client', remote='%s'", z_sockaddr_format(self->data_remote[EP_CLIENT], buf, sizeof(buf)));
              g_mutex_unlock(self->lock);
              ftp_data_reset(self);
              z_proxy_leave(self);
              return;
            }
        }
    }
  
  if (self->data_state == FTP_SERVER_CONNECT_READY)
    {
      z_proxy_cp(self);
      if (!self->data_connect[EP_CLIENT])
        {
          if (!ftp_policy_bounce_check(self, EP_CLIENT, self->data_remote[EP_CLIENT], FALSE))
            {
              /*LOG
                Placeholder ftp.c.9
               */
              z_proxy_log(self, FTP_POLICY, 3, "Possibly bounce attack; connect='FALSE', side='client', remote='%s'", z_sockaddr_format(self->data_remote[EP_CLIENT], buf, sizeof(buf)));
              g_mutex_unlock(self->lock);
              ftp_data_reset(self);
              z_proxy_leave(self);
              return;
            }
        }
      
      g_mutex_unlock(self->lock);
      
      if (ftp_data_start_proxy(self))
        {
          g_mutex_lock(self->lock);
          if (self->data_state != FTP_DATA_CANCEL)
            self->data_state = FTP_DATA_CONVERSATION;
          g_mutex_unlock(self->lock);
        }
      else
        ftp_data_reset(self);

      z_proxy_leave(self);
      return ;
    }
  else if (self->data_state == FTP_DATA_CANCEL)
    {
      g_mutex_unlock(self->lock);
      ftp_data_reset(self);
      return;
    }
  else
    {
      z_proxy_cp(self);
    }

  g_mutex_unlock(self->lock);
  z_proxy_leave(self);
  return;
}

gboolean
ftp_data_abort(FtpProxy *self)
{
  char buf[3];
  guint len;
  GIOStatus rc;

  z_proxy_enter(self);
  
  ftp_data_reset(self);
  
  buf[0]=0xff;
  buf[1]=0xf4;
  buf[2]=0xff;
  rc = z_stream_write_pri(self->super.endpoints[EP_SERVER], buf, 3, &len, NULL);

  if (rc == G_IO_STATUS_NORMAL)
    {
      buf[0]=0xf2;
      rc = z_stream_write(self->super.endpoints[EP_SERVER], buf, 1, &len, NULL);
  
      ftp_stream_write(self, 'S', "ABOR", 4);
    }
  z_proxy_leave(self);
  return (rc == G_IO_STATUS_NORMAL);
}


static gboolean ftp_client_data(ZStream *stream,
                           GIOCondition  cond,
                               gpointer  user_data);

static gboolean ftp_server_data(ZStream *stream,
                           GIOCondition  cond,
                               gpointer  user_data);
        
gboolean
ftp_stream_client_init(FtpProxy *self)
{
  ZStream *tmpstream;
  
  z_proxy_enter(self);
  if (!self->super.endpoints[EP_CLIENT])
    {
      ZPROXY_SET_FLAG(self, PF_QUIT);
      /*LOG
        This messages indicates that proxy started without connection in
        client side.
       */
      z_proxy_log(self, FTP_ERROR, 1, "Client side not connected;");
      z_proxy_leave(self);
      return FALSE;
    }
  self->super.endpoints[EP_CLIENT]->timeout = self->timeout;
  
  tmpstream = self->super.endpoints[EP_CLIENT];
  self->super.endpoints[EP_CLIENT] = z_stream_line_new(tmpstream, self->max_line_length, ZRL_EOL_CRLF);

  z_stream_unref(tmpstream);

  z_stream_set_cond(self->super.endpoints[EP_CLIENT],
                   Z_STREAM_FLAG_READ,
                   FALSE);
                       
  z_stream_set_cond(self->super.endpoints[EP_CLIENT],
                   Z_STREAM_FLAG_WRITE,
                   FALSE);
                       
  z_stream_set_cond(self->super.endpoints[EP_CLIENT],
                   Z_STREAM_FLAG_PRI,
                   FALSE);
                       
  z_stream_set_callback(self->super.endpoints[EP_CLIENT],
                        Z_STREAM_FLAG_READ,
                        ftp_client_data,
                        self, 
                        NULL);
                       
  z_stream_set_callback(self->super.endpoints[EP_CLIENT],
                        Z_STREAM_FLAG_PRI,
                        ftp_client_data,
                        self, 
                        NULL);

  z_poll_add_stream(self->poll, self->super.endpoints[EP_CLIENT]);

  z_proxy_leave(self);
  return TRUE;
}

gboolean
ftp_stream_server_init(FtpProxy *self)
{
  ZStream *tmpstream;
  
  z_proxy_enter(self);
  if (!self->super.endpoints[EP_SERVER])
    {
      ZPROXY_SET_FLAG(self, PF_QUIT);
      /*LOG
        This message indicates that proxy cannot connect to
        server.
       */
      z_proxy_log(self, FTP_ERROR, 1, "Server side not connected;");
      z_proxy_leave(self);
      return FALSE;
    }
  self->super.endpoints[EP_SERVER]->timeout = self->timeout;
  
  tmpstream = self->super.endpoints[EP_SERVER];
  self->super.endpoints[EP_SERVER] = z_stream_line_new(tmpstream, self->max_line_length, ZRL_EOL_CRLF);
  
  z_stream_unref(tmpstream);

  z_stream_set_cond(self->super.endpoints[EP_SERVER],
                   Z_STREAM_FLAG_READ,
                   FALSE);
                       
  z_stream_set_cond(self->super.endpoints[EP_SERVER],
                   Z_STREAM_FLAG_WRITE,
                   FALSE);
                       
  z_stream_set_cond(self->super.endpoints[EP_SERVER],
                   Z_STREAM_FLAG_PRI,
                   FALSE);
                       
  z_stream_set_callback(self->super.endpoints[EP_SERVER],
                        Z_STREAM_FLAG_READ,
                        ftp_server_data,
                        self,
                        NULL);

  z_poll_add_stream(self->poll, self->super.endpoints[EP_SERVER]);

  z_proxy_leave(self);
  return TRUE;
}

void
ftp_proxy_regvars(FtpProxy *self)
{

  z_proxy_enter(self);
  z_proxy_var_new(&self->super, "transparent_mode",
                  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET_CONFIG,
                  &self->transparent_mode);
                  
  z_proxy_var_new(&self->super, "permit_unknown_command",
                  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET_CONFIG,
                  &self->permit_unknown_command);
                  
  z_proxy_var_new(&self->super, "permit_empty_command",
                  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET_CONFIG,
                  &self->permit_empty_command);

  z_proxy_var_new(&self->super, "max_line_length",
                  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET_CONFIG,
                  &self->max_line_length);
                  
  z_proxy_var_new(&self->super, "max_username_length",
                  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET_CONFIG,
                  &self->max_username_length);
                  
  z_proxy_var_new(&self->super, "max_password_length",
                  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET_CONFIG,
                  &self->max_password_length);
                  
  z_proxy_var_new(&self->super, "max_hostname_length",
                  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET_CONFIG,
                  &self->max_hostname_length);
  
  z_proxy_var_new(&self->super, "masq_address_client",
                  Z_VAR_TYPE_STRING | Z_VAR_GET | Z_VAR_SET_CONFIG,
                  self->masq_address[EP_CLIENT]);
                  
  z_proxy_var_new(&self->super, "masq_address_server",
                  Z_VAR_TYPE_STRING | Z_VAR_GET | Z_VAR_SET_CONFIG,
                  self->masq_address[EP_SERVER]);

  z_proxy_var_new(&self->super, "request",
                  Z_VAR_TYPE_HASH | Z_VAR_GET | Z_VAR_GET_CONFIG,
                  self->policy_command_hash);
  z_proxy_var_new(&self->super, "commands",
                  Z_VAR_TYPE_OBSOLETE | Z_VAR_GET | Z_VAR_GET_CONFIG,
                  "request");
                  
  z_proxy_var_new(&self->super, "response",
                  Z_VAR_TYPE_DIMHASH | Z_VAR_GET | Z_VAR_GET_CONFIG,
                  self->policy_answer_hash);
  z_proxy_var_new(&self->super, "answers",
                  Z_VAR_TYPE_OBSOLETE | Z_VAR_GET | Z_VAR_GET_CONFIG,
                  "response");

  z_proxy_var_new(&self->super, "request_command",
                  Z_VAR_TYPE_STRING | Z_VAR_GET | Z_VAR_SET,
                  self->request_cmd);
                  
  z_proxy_var_new(&self->super, "request_parameter",
                  Z_VAR_TYPE_STRING | Z_VAR_GET | Z_VAR_SET,
                  self->request_param);

  z_proxy_var_new(&self->super, "response_status",
                  Z_VAR_TYPE_STRING | Z_VAR_GET | Z_VAR_SET,
                  self->answer_cmd);
  z_proxy_var_new(&self->super, "answer_code",
                  Z_VAR_TYPE_OBSOLETE | Z_VAR_GET | Z_VAR_SET,
                  "response_status");
                  
  z_proxy_var_new(&self->super, "response_parameter",
                  Z_VAR_TYPE_STRING | Z_VAR_GET | Z_VAR_SET,
                  self->answer_param);
  z_proxy_var_new(&self->super, "answer_parameter",
                  Z_VAR_TYPE_OBSOLETE | Z_VAR_GET | Z_VAR_SET,
                  "response_param");
  
  z_proxy_var_new(&self->super, "data_mode",
                  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET_CONFIG,
                  &self->data_mode);

  z_proxy_var_new(&self->super, "username",
                  Z_VAR_TYPE_STRING | Z_VAR_GET,
                  self->username);
                  
  z_proxy_var_new(&self->super, "password",
                  Z_VAR_TYPE_STRING | Z_VAR_GET,
                  self->password);

  z_proxy_var_new(&self->super, "ftpDataStop",
                  Z_VAR_TYPE_METHOD | Z_VAR_GET,
                  self, ftp_data_end);

  z_proxy_var_new(&self->super, "response_strip_msg",
                  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET_CONFIG,
                  &self->response_strip_msg);

  z_proxy_var_new(&self->super, "timeout",
                  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET_CONFIG,
                  &self->timeout);

  z_proxy_var_new(&self->super, "target_port_range",
                  Z_VAR_TYPE_STRING | Z_VAR_GET | Z_VAR_SET_CONFIG,
                  self->target_port_range);

  z_proxy_var_new(&self->super, "data_port_min",
                  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET_CONFIG,
                  &self->data_port_min);

  z_proxy_var_new(&self->super, "data_port_max",
                  Z_VAR_TYPE_INT | Z_VAR_GET | Z_VAR_SET_CONFIG,
                  &self->data_port_max);

  z_proxy_leave(self);
}

void
ftp_config_set_defaults(FtpProxy * self)
{

  z_proxy_enter(self);
  self->ftp_state = FTP_STATE_CONNECT;
  self->transparent_mode = TRUE;
  self->permit_empty_command = TRUE;
  self->permit_unknown_command = FALSE;
  self->response_strip_msg = FALSE;
  
  self->max_line_length = 255;
  self->max_username_length = 32;
  self->max_password_length = 64;
  self->max_hostname_length = 128;

  self->line = g_new0(char, FTP_LINE_MAX_LEN + 1);
  self->username = g_string_new("");
  self->password = g_string_new("");
  self->hostname = g_new0(char, FTP_LINE_MAX_LEN + 1);
  
  self->data_mode = FTP_DATA_KEEP;

//  self->data_remote[EP_SERVER] = z_sockaddr_inet_new("0.0.0.0", 0);
//  self->data_remote[EP_CLIENT] = z_sockaddr_inet_new("0.0.0.0", 0);
//  self->data_local[EP_SERVER] = z_sockaddr_inet_new("0.0.0.0", 0);
//  self->data_local[EP_CLIENT] = z_sockaddr_inet_new("0.0.0.0", 0);

  self->masq_address[EP_SERVER] = g_string_new("");
  self->masq_address[EP_CLIENT] = g_string_new("");

  self->lock = g_mutex_new();

  self->timeout = 300000;
  
  self->policy_command_hash = ftp_policy_command_hash_create();
  self->policy_answer_hash = ftp_policy_answer_hash_create();
  
  self->request_cmd = g_string_sized_new(4);
  self->request_param = g_string_new("");

  self->answer_cmd = g_string_sized_new(4);
  self->answer_param = g_string_new("");
  
  self->target_port_range = g_string_new("21");
  
  self->hostport = 21;
  
  self->poll = z_poll_new();
  
  self->data_port_min = 40000;
  self->data_port_max = 41000;
  
  z_proxy_leave(self);
}

void
ftp_config_init(FtpProxy *self)
{
  z_proxy_enter(self);
  
  if (self->max_line_length > FTP_LINE_MAX_LEN)
    self->max_line_length = FTP_LINE_MAX_LEN;
    
  if (self->max_username_length > self->max_line_length)
    self->max_username_length = self->max_line_length;
    
  if (self->max_password_length > self->max_line_length)
    self->max_password_length = self->max_line_length;
    
  if (self->max_hostname_length > self->max_line_length)
    self->max_hostname_length = self->max_line_length;

  z_proxy_leave(self);
}

GIOStatus
ftp_read_line_get (FtpProxy * self, gchar side)
{
  gint readback = G_IO_STATUS_ERROR;
  guint i;
  gint state;
  char *tmp;
  guint pos1;
  unsigned char funcs[10] = { 241, 242, 243, 244, 245, 246, 247, 248, 249, '\0' };
  unsigned char negot[5] = { 251, 252, 253, 254, '\0' };
  char *line = self->line;

  z_proxy_enter(self);

  self->line_length = self->max_line_length;

  switch (side)
    {
    case 'S':
      readback = z_stream_line_get_copy(self->super.endpoints[EP_SERVER], self->line, &self->line_length, NULL);
      break;
    case 'C':
      readback = z_stream_line_get_copy(self->super.endpoints[EP_CLIENT], self->line, &self->line_length, NULL);
      break;
    default:
      /*LOG
        This message indicates that streamline called with wrong
        parameter.
       */
      z_proxy_log(self, FTP_ERROR, 1, "Internal error; error='side is wrong'");
      break;
    }
  
  if (readback != G_IO_STATUS_NORMAL)
    {
      self->line_length = 0;
      z_proxy_leave(self);
      return readback;
    }

  tmp = g_new0(char, (self->line_length) + 2);
  state = FTP_TELNET;
  pos1 = 0;

  if (self->line_length)
    for (i = 0; i < self->line_length; i++)
      {
	// TODO: adding SB SE
	switch (state)
	  {
	  case FTP_TELNET:
	    if ((unsigned char) line[i] != 255)
	      {
		tmp[pos1] = line[i];
		pos1++;
	      }
	    else
	      {
		if ((unsigned char) line[i + 1] == 255)
		  {
		    tmp[pos1] = line[i];
		    pos1++;
		    i++;
		  }
		else
		  {
		    state = FTP_TELNET_IAC;
		  }
	      }
	    break;
	  case FTP_TELNET_IAC:
	    if (strchr (funcs, line[i]))
	      {
		// in funcs
		state = FTP_TELNET;
	        if ((unsigned char) line[i+1] == 242)
	          {
	            i++;
	          }
	      }
	    else if (strchr (negot, line[i]))
	      {
		// in negotiation
		state = FTP_TELNET_IAC_DW;
	      }
	    else if ((unsigned char) line[i] == 250)
	      {
		state = FTP_TELNET_IAC_FUNC;
	      }
	    else
	      {
		// Bad seq
	      }
	    break;
	  case FTP_TELNET_IAC_DW:
	    state = FTP_TELNET;
	    break;
	  case FTP_TELNET_IAC_FUNC:
	    if ((unsigned char) line[i] == 240)
	      {
		state = FTP_TELNET;
	      }
	    break;
	  default:
	    break;
	  }
      }

  self->line_length = pos1;
  snprintf (line, self->line_length + 1, "%s", tmp);
  if (tmp)
    g_free (tmp);

  z_proxy_leave(self);
  return readback;
}

gboolean
ftp_stream_write(FtpProxy *self, char side, guchar *line, guint length)
{
  guint bytes_written = 0;
  gchar buf[2 * length + 3];
  guint i, j;
  GIOStatus rc;

  z_proxy_enter(self);
  for (i = 0, j = 0; i < length; i++)
    {
      buf[j++] = line[i];
      if (line[i] == 255)
        {
          buf[j++] = 255;
        }
    }
  buf[j++] = '\r';
  buf[j++] = '\n';

  switch (side)
    {
    case 'S':
      rc = z_stream_write(self->super.endpoints[EP_SERVER], buf, j, &bytes_written, NULL);
      break;
    case 'C':
      rc = z_stream_write(self->super.endpoints[EP_CLIENT], buf, j, &bytes_written, NULL);
      break;
    default:
      /*LOG
        This message indicates that write called with wrong
        parameter.
       */
      z_proxy_log(self, FTP_ERROR, 1, "Internal error; error='side is wrong'");
      break;
    }

  if (bytes_written == j)
    {
      z_proxy_leave(self);
      return TRUE;
    }
  else
    {
      z_proxy_leave(self);
      return FALSE;
    }
  z_proxy_leave(self);
}

gboolean
ftp_answer_fetch(FtpProxy *self, gboolean *continued)
{
  guint res;
  gboolean cont = FALSE;
  int i;

  z_proxy_enter(self);
  res = ftp_read_line_get(self, 'S');
  
  if (res == G_IO_STATUS_EOF)
    {
      /* read EOF */
      z_proxy_leave(self);
      return FALSE;
    }
    
  if (res != G_IO_STATUS_NORMAL)
    {
      /*LOG
        Tis message indicates that some error occured when zorp try
        to read from server side. This may couse by timeout.
       */
      z_proxy_log(self, FTP_ERROR, 1, "Error reading from server; result='%m'");
      z_proxy_leave(self);
      return FALSE;
    }
    
  if (self->line_length > self->max_line_length)
    {
      /*LOG
        Received line is too long. If it's not cracking attempt
        try to increase max_line_length.
       */
      z_proxy_log(self, FTP_POLICY, 2, "Error reading from server; error='line too long' length='%d', max_line_length='%d'", self->line_length, self->max_line_length);
      z_proxy_leave(self);
      return FALSE;
    }
    
  if (continued)
    {
      if(self->line_length <4)
        cont = TRUE;
      else

        for (i = 0; i < 3 ; i++)
          if (!isdigit(self->line[i]))
            cont = TRUE;
    }
  else
    {
      if (self->line_length < 4)
        {
          /*LOG
            This message attemted when reply message is shorten than 4
            character.
           */
          z_proxy_log(self, FTP_VIOLATION, 1, "Line is too short to be a valid answer;");
          z_proxy_leave(self);
          return FALSE;
        }

      if (self->line[3] != ' ' && self->line[3] != '-')
        {
          /*LOG
            This message attempted when answer format is not valid.
           */
          z_proxy_log(self, FTP_VIOLATION, 1, "Server answer has wrong continuation mark;");
          z_proxy_leave(self);
          return FALSE;
        }
    }

  *continued = (cont || self->line[3]=='-');
  
  z_proxy_leave(self);
  return TRUE;
}

gboolean
ftp_answer_parse(FtpProxy *self)
{
  char answer[4];
  int i;

  z_proxy_enter(self);  
  for (i = 0; i < 3; i++)
    if (!isdigit(self->line[i]))
      {
        /*LOG
          This message attempted when server violates FTP protokoll.
         */
	z_proxy_log(self, FTP_VIOLATION, 1, "Server answer doesn't begin with number;");
	z_proxy_leave(self);
	return FALSE;
      }
    else
      answer[i] = self->line[i];

  answer[3] = '\0'; /* zero terminate answer */

  self->answer_cmd = g_string_assign(self->answer_cmd, answer);

  self->line[self->line_length] = 0;
  
  self->answer_param = g_string_assign(self->answer_param, self->line + 4);
  
  /*LOG
    Placeholder ftp.9
   */
  z_proxy_log(self, FTP_REPLY, 6, "Response arrived; rsp='%s', rsp_prm='%s'", self->answer_cmd->str, self->answer_param->str);
  
  z_proxy_leave(self);
  return TRUE;
}

gboolean ftp_answer_write_setup(FtpProxy *self, gchar *answer_c, gchar *answer_p);
gboolean ftp_answer_write(FtpProxy *self, gchar *msg);
gboolean ftp_command_write_setup(FtpProxy *self, gchar *answer_c, gchar *answer_p);
gboolean ftp_command_write(FtpProxy *self, char *msg);

void
ftp_answer_process(FtpProxy *self) 
{
  FtpInternalCommand *command = self->command_desc;
  int res;
  gchar newline[FTP_LINE_MAX_LEN];

  z_proxy_enter(self);
  
  
  res = ftp_policy_answer_hash_do(self);

  self->answer_code = atoi(self->answer_cmd->str);
  
  if (res == FTP_RSP_ACCEPT)
    {
      if (command)
        {
          if(command->answer)
            {
              res = command->answer(self);
            }
        }
    }

  self->answer_handle = res;

  switch (res)
    {
    case FTP_RSP_ACCEPT:
      if(self->answer_cont)
        g_snprintf(newline, sizeof(newline), "%s-%s", self->answer_cmd->str, self->answer_param->str);
      else
        g_snprintf(newline, sizeof(newline), "%s %s", self->answer_cmd->str, self->answer_param->str);
      ftp_answer_write(self, newline);
      break;
    case FTP_RSP_ABORT:
      self->state = FTP_QUIT;  /* no break; we must write answer... */
    case FTP_RSP_REJECT:
      /*LOG
        Zorp change answer line because policy require this.
       */
      z_proxy_log(self, FTP_POLICY, 3, "Rejected answer; from='%s', to='%s %s'", self->line, self->answer_cmd->str, self->answer_param->str);
      ftp_answer_write_setup(self, self->answer_cmd->str, self->answer_param->str);
      break;
    default:
      self->state = FTP_QUIT;
      break;
    }
  z_proxy_leave(self);
  return;
}

gboolean
ftp_answer_write_setup(FtpProxy *self, gchar *answer_c, gchar *answer_p)
{
  gchar *tmp;
  gchar newline[self->max_line_length];
  gboolean res = TRUE;

  z_proxy_enter(self);  
  tmp = strchr(answer_p, '\n');
  if(!tmp)
    {

      g_snprintf(newline, sizeof(newline), "%s %s", answer_c, answer_p);
      res = ftp_answer_write(self,newline);
    }
  else
    {
      while (tmp && res)
        {
          *tmp = 0;

          g_snprintf(newline, sizeof(newline), "%s-%s", answer_c, answer_p);
          res = ftp_answer_write(self, newline);

          *tmp = '\n';

          answer_p = tmp + 1;
          tmp = strchr(answer_p, '\n');
        }
        
      if (res)
        {
          if (*answer_p)
            g_snprintf(newline, sizeof(newline), "%s %s", answer_c, answer_p);
          else
            g_snprintf(newline, sizeof(newline), "%s", answer_c);

          res = ftp_answer_write(self,newline);
        }
    }
  z_proxy_leave(self);
  return res;
}
  
gboolean
ftp_answer_write(FtpProxy *self, gchar *msg)
{
  guint bytes_to_write;
  gboolean back;

  z_proxy_enter(self);

  /* TODO PASV doesn't work!!!! */
  if (self->response_strip_msg)
    {
      bytes_to_write = 4;
    }
  else
    {
      bytes_to_write = strlen(msg);
    }
    
  back = ftp_stream_write(self, 'C', msg, bytes_to_write);
  
  z_proxy_leave(self);  
  return back;
}

void ftp_command_reject(FtpProxy *self);

gboolean
ftp_command_fetch(FtpProxy *self)
{
  guint res;

  z_proxy_enter(self);  
  res = ftp_read_line_get(self, 'C');
  
  if (res == G_IO_STATUS_EOF)
    {
      /* read EOF */
      z_proxy_leave(self);
      return FALSE;
    }
  
  if (res != G_IO_STATUS_NORMAL)
    {
      /*LOG
        Placeholder ftp.c.10
       */
      z_proxy_log(self, FTP_ERROR, 2, "Error reading from server; result='%m'");
      if (errno == ETIMEDOUT)
        SET_ANSWER(MSG_TIMED_OUT)
      else
        SET_ANSWER(MSG_LINE_TERM_CRLF)
      ftp_command_reject(self);

      z_proxy_leave(self);
      return FALSE;
    }
    
  if (self->line_length > self->max_line_length)
    {
      /*LOG
        Received line is too long. If it's not cracking attempt
        try to increase max_line_length.
       */
      z_proxy_log(self, FTP_POLICY, 2, "Error reading from client; error='line too long; length='%d', max_line_length='%d'", self->line_length, self->max_line_length);
      SET_ANSWER(MSG_LINE_TOO_LONG);
      ftp_command_reject(self);
      z_proxy_leave(self);
      return FALSE;
    }
  
  self->line[self->line_length] = 0;
  
  z_proxy_leave(self);  
  return TRUE;
}

gboolean
ftp_command_parse(FtpProxy *self)
{
  gchar *src = self->line;
  guint i = 0;

  self->request_cmd = g_string_assign(self->request_cmd, "");

  z_proxy_enter(self);
  while ((*src != ' ') && (i < self->line_length))
    {
      self->request_cmd = g_string_append_c(self->request_cmd, *src++);
      i++;
    }
  self->request_cmd = g_string_append_c(self->request_cmd, 0);


  src++;
  i++;
  
  if (i < self->line_length)
    self->request_param = g_string_assign(self->request_param, src);
  else
    self->request_param = g_string_assign(self->request_param, "");

  /*LOG
    Placeholder ftp.c.11
   */
  z_proxy_log(self, FTP_REQUEST, 6, "Request fetched; req=`%s' req_prm='%s'", self->request_cmd->str, self->request_param->str);

  self->request_cmd = g_string_up(self->request_cmd);
      
  self->command_desc = ftp_command_hash_get(self->request_cmd->str);

  if (!self->request_cmd->len && !self->permit_empty_command)
    {
      /*LOG
        Placeholder ftp.c.11
       */
      z_proxy_log(self, FTP_VIOLATION, 1, "Empty Command. Aborting;");
      z_proxy_leave(self);
      return FALSE;
    }
  else
    if (!self->command_desc && !self->permit_unknown_command && !ftp_policy_command_hash_search(self, self->request_cmd->str))
      {
      /*LOG
        Placeholder ftp.c.11
       */
        z_proxy_log(self, FTP_VIOLATION, 1, "Unknown Command. Aborting; req='%s'", self->request_cmd->str);
        z_proxy_leave(self);
        return FALSE;
      }
  z_proxy_leave(self);  
  return TRUE;
}

/*+ 
    This function processes the parsed command, and decides what to do. It
    then either forwards or drops the command.
  +*/ 
  
void
ftp_command_process(FtpProxy *self) 
{
  FtpInternalCommand *command = self->command_desc;
  int res;

  z_proxy_enter(self);  

  SET_ANSWER(MSG_ERROR_PARSING_COMMAND);

  res = ftp_policy_command_hash_do(self);

  if (res == FTP_REQ_ACCEPT)
    {
      if (command)
        {
          if (!command->parse)
            {
              /*LOG
                This message attempt in internal error only.
                Maybe memory corruption?
               */
              z_proxy_log(self, FTP_ERROR, 1, "Internal error; error='command->parse is NULL', req='%s'", self->request_cmd->str);
              res = FTP_REQ_ABORT;
            }
          else
            res = command->parse(self);
        }
    }
    
  switch (res)
    {
    case FTP_REQ_ACCEPT:
      if (command && command->need_data)
        {
          ftp_data_start(self);
        }
        
      ftp_command_write_setup(self, self->request_cmd->str, self->request_param->str);
      break;
    case FTP_REQ_REJECT:
      ftp_command_reject(self);
      
      if (self->state == FTP_SERVER_TO_CLIENT)
        {
          ftp_state_set(self, EP_CLIENT);
          self->state = FTP_CLIENT_TO_SERVER;
        }
        
      else if (self->state == FTP_NT_SERVER_TO_PROXY)
        {
          ftp_state_set(self, EP_CLIENT);
          self->state = FTP_NT_CLIENT_TO_PROXY;
        }
      
      /*LOG
        This message appear when zorp is drop a command.
        This done becouse policy request it.
       */
      z_proxy_log(self, FTP_POLICY, 3, "Request Rejected; req='%s'", self->request_cmd->str);
      break;
    case FTP_PROXY_ANS:
      ftp_answer_write_setup(self, self->answer_cmd->str, self->answer_param->str);
      
      if (self->state == FTP_SERVER_TO_CLIENT)
        {
          ftp_state_set(self, EP_CLIENT);
          self->state = FTP_CLIENT_TO_SERVER;
        }
        
      else if (self->state == FTP_NT_SERVER_TO_PROXY)
        {
          ftp_state_set( self, EP_CLIENT);
          self->state = FTP_NT_CLIENT_TO_PROXY;
        }
      
      /*LOG
        This message attempt when zorp is answering a command and
        not send it to the server.
       */
      z_proxy_log(self, FTP_POLICY, 4, "Proxy answer; rsp='%s' rsp_prm='%s'", self->answer_cmd->str, self->answer_param->str);
      break;
    case FTP_REQ_ABORT:
      SET_ANSWER(MSG_CONNECTION_ABORTED);
      ftp_command_reject(self);
      
      /*LOG
        This message appear when zorp is dropping the connection,
        because of a harmfull command.
       */
      z_proxy_log(self, FTP_POLICY, 2, "Dropped command (aborting); line='%s'", self->line);
      self->state = FTP_QUIT;
      break;
    case FTP_NOOP:
      break;
    default:
      /*LOG
        This message attempt when policy return with bad data.
       */
      z_proxy_log(self, FTP_POLICY, 1, "Bad policy type, aborting;");
      self->state = FTP_QUIT;
    }  
  z_proxy_leave(self);  
  return;
}

void
ftp_command_reject(FtpProxy *self)
{
  z_proxy_enter(self);
  ftp_answer_write_setup(self, self->answer_cmd->str, self->answer_param->str);
  z_proxy_leave(self);  
}

gboolean
ftp_command_write(FtpProxy *self, char *msg)
{
  gint bytes_to_write = strlen(msg);
  gboolean back;
  
  z_proxy_enter(self);
  back = ftp_stream_write(self, 'S', msg, bytes_to_write);
  z_proxy_leave(self);  
  return back;
}

gboolean
ftp_command_write_setup(FtpProxy *self, gchar *answer_c, gchar *answer_p)
{
  gchar newline[self->max_line_length];
  gboolean vissza = TRUE;

  z_proxy_enter(self);  
  
  if (strlen(answer_p) > 0)
    g_snprintf(newline, sizeof(newline), "%s %s", answer_c, answer_p);
  else
    g_snprintf(newline, sizeof(newline), "%s", answer_c);
  vissza = ftp_command_write(self,newline);
  
  z_proxy_leave(self);
  return vissza;
}


void
ftp_proto_nt_init(FtpProxy *self)
{
  z_proxy_enter(self);
  self->ftp_state = FTP_STATE_PRECONNECT;
  
  SET_ANSWER(MSG_NON_TRANSPARENT_GREETING);
  ftp_answer_write_setup(self, self->answer_cmd->str, self->answer_param->str);
  
  self->state = FTP_NT_CLIENT_TO_PROXY;
  z_proxy_leave(self);  
}

void
ftp_proto_nt_client_to_proxy(FtpProxy *self)
{

  z_proxy_enter(self);
  if (!ftp_command_fetch(self))
    {
      self->state = FTP_QUIT;
      z_proxy_leave(self);
      return;
    }

  if (!ftp_command_parse(self))
    {
      self->state = FTP_QUIT;
      z_proxy_leave(self);
      return;
    }

  ftp_command_process(self);

  switch (self->ftp_state)
    {
    case FTP_STATE_PRECONNECT_LOGIN_P:
      /* by this time login creditentials, and host name is received,
         try to connect to the remote server */
      
      ftp_connect_server_event(self, self->hostname, self->hostport);
      ftp_stream_server_init(self);
      self->state = FTP_NT_SERVER_TO_PROXY;
      self->ftp_state = FTP_STATE_PRECONNECT;
      self->request_cmd = g_string_assign(self->request_cmd, "");
      break;
    case FTP_STATE_PRECONNECT_QUIT:
      self->state = FTP_QUIT;
      break;
    }
  z_proxy_leave(self);  
}

void
ftp_proto_nt_server_to_proxy(FtpProxy *self)
{
  gboolean first = TRUE;
  gboolean continued = TRUE;

  z_proxy_enter(self);
  self->answer_cmd = g_string_assign(self->answer_cmd, "");
  self->answer_code = 0;
  self->answer_cont = 0;

  while (continued)
    {
      if (!ftp_answer_fetch(self, &self->answer_cont))
        {
          self->state = FTP_QUIT;
          z_proxy_leave(self);
          return;
        }
      
      continued = self->answer_cont;
      
      if (first)
        {
          first = FALSE;

          if (!ftp_answer_parse(self))
            {
              self->state = FTP_QUIT;
              z_proxy_leave(self);
              return;
            }
        }
        else
        {
          if (self->answer_handle == FTP_RSP_ACCEPT && !self->response_strip_msg)
            ftp_answer_write(self, self->line);
        }
                                              
    }

  switch (self->ftp_state)
    {
    case FTP_STATE_PRECONNECT:
      if (strcmp(self->answer_cmd->str, "220") == 0)
	{
	  gchar user_line[self->username->len + 6];

	  g_snprintf(user_line, sizeof(user_line), "USER %s", self->username->str);
	  self->request_cmd = g_string_assign(self->request_cmd, "USER");
	  ftp_command_write(self, user_line);
	  self->ftp_state = FTP_STATE_PRECONNECT_LOGIN_U;
	}
      break;
    case FTP_STATE_PRECONNECT_LOGIN_U:
      if (strcmp(self->answer_cmd->str, "331") == 0)
	{
	  gchar pass_line[self->password->len + 6];

	  /* send password */
	  g_snprintf(pass_line, sizeof(pass_line), "PASS %s", self->password->str);
	  self->request_cmd = g_string_assign(self->request_cmd, "PASS");
	  ftp_command_write(self, pass_line);
	  self->ftp_state = FTP_STATE_LOGIN_P;
	  self->state = FTP_SERVER_TO_CLIENT;
	  ftp_state_set(self, EP_SERVER);
	}
      else if (strcmp(self->answer_cmd->str, "230") == 0)
	{
	  /* no password required */
	  ftp_answer_write(self, self->line);
	  self->ftp_state = FTP_STATE_CONVERSATION;
	  ftp_state_set(self, EP_CLIENT);
	  self->state = FTP_CLIENT_TO_SERVER;
	}
    }
  z_proxy_leave(self);  
}

void
ftp_proto_client_to_server(FtpProxy *self)
{
  z_proxy_enter(self);
  if (!ftp_command_fetch(self))
    {
      self->state = FTP_QUIT;
      z_proxy_leave(self);
      return;
    }
  
  if (!ftp_command_parse(self))
    {
      self->state = FTP_QUIT;
      z_proxy_leave(self);
      return;
    }

  self->state = FTP_SERVER_TO_CLIENT;
  ftp_state_set(self, EP_SERVER);
  ftp_command_process(self);
  z_proxy_leave(self);  
}

void
ftp_proto_server_to_client(FtpProxy *self)
{
  gboolean continued = TRUE;
  gboolean first = TRUE;

  z_proxy_enter(self);
  self->answer_cmd = g_string_assign(self->answer_cmd, "");
  self->answer_code = 0;
  self->answer_cont = 0;

  while (continued)
    {
      if (!ftp_answer_fetch(self, &self->answer_cont))
        {
          self->state = FTP_QUIT;
          z_proxy_leave(self);
          return;
        }
      
      continued = self->answer_cont;
        
      if (first)
        {
          first = FALSE;
          
          if (!ftp_answer_parse(self))
            {
              self->state = FTP_QUIT;
              z_proxy_leave(self);
              return;
            }
          
          self->state = FTP_CLIENT_TO_SERVER;
          ftp_state_set(self, EP_CLIENT);
          ftp_answer_process(self);
        }
      else
        {
          if (self->answer_handle == FTP_RSP_ACCEPT && !self->response_strip_msg)
            ftp_answer_write(self, self->line);
        }
    }
  z_proxy_leave(self);  
}

static gboolean
ftp_server_data(ZStream *stream G_GNUC_UNUSED,
           GIOCondition  cond G_GNUC_UNUSED,
               gpointer  user_data)
{
  FtpProxy *self = (FtpProxy *) user_data;
  z_proxy_enter(self);
  ftp_proto_server_to_client(self);
  z_proxy_leave(self);  
  return TRUE;
}

static gboolean
ftp_client_data(ZStream *stream G_GNUC_UNUSED,
           GIOCondition  cond G_GNUC_UNUSED,
               gpointer  user_data)
{
  FtpProxy *self = (FtpProxy *) user_data;
  z_proxy_enter(self);
  ftp_proto_client_to_server(self);
  z_proxy_leave(self);  
  return TRUE;
}

void
ftp_listen_both_side(FtpProxy *self)
{
  guint rc;
  
  z_proxy_enter(self);
  
  if (!(self->data_state & FTP_DATA_CONVERSATION))
    {
      rc = z_poll_iter_timeout(self->poll, self->timeout);
      if (!rc)
        {
          if (errno == ETIMEDOUT)
            {
              SET_ANSWER(MSG_TIMED_OUT);
              ftp_command_reject(self);
            }
          self->state = FTP_QUIT;
          ZPROXY_SET_FLAG(self, PF_QUIT);
        }
    }
  else
    rc = z_poll_iter_timeout(self->poll, -1);

  if (self->data_state && self->state != FTP_QUIT)
    {
      if (rc)
        ftp_data_next_step(self);

      if(self->state != FTP_QUIT)
        self->state = FTP_BOTH_SIDE;
    }
  z_proxy_leave(self);  
}

gpointer
ftp_thread(gpointer s)
{
  FtpProxy *self = (FtpProxy *) s;

  z_proxy_enter(self);
  ftp_config_set_defaults(self);
  ftp_proxy_regvars(self);

  ZPROXY_SET_STATE(self, PS_CONFIG);
  if (!z_proxy_config_event(&self->super))
    {
      /*LOG
        This message attempt, when some error is occured in config
        event. This may be an error in python level, or
        python - C interoperability weekness
       */
      z_proxy_log(self, FTP_POLICY, 1, "Error in config event;");
      ZPROXY_SET_FLAG(self, PF_QUIT);
    }
  else
    {
      /*LOG
        This debug message attempt when config event is done
        correctly.
       */
      z_proxy_log(self, FTP_DEBUG, 7, "Config event done;");
      ftp_config_init(self);
  
      ZPROXY_SET_STATE (self, PS_STARTING_UP);
      if(!z_proxy_startup_event (&self->super))
        {
          /*LOG
            This message attempt, when some error is occured in startup
            event. This may be an error in python level, or
            python - C interoperability weekness
           */
          z_proxy_log(self, FTP_POLICY, 1, "Startup event error;");
          ZPROXY_SET_FLAG(self, PF_QUIT);
        }
      else
        {
          /*LOG
            This debug message attempt when startup event is done
            correctly.
           */
          z_proxy_log(self, FTP_DEBUG, 7, "Startup event done;");
          ZPROXY_SET_STATE(self, PS_WORKING);

          if (!ftp_stream_client_init(self))
            ZPROXY_SET_FLAG(self, PF_QUIT);
          if (self->transparent_mode)
            self->state = FTP_INIT_TRANSPARENT;
          else
            self->state = FTP_INIT_NONTRANSPARENT;

        }
    }
  while (!ZPROXY_GET_FLAG(self, PF_QUIT))
    {
      switch (self->state)
        {
          case FTP_INIT_TRANSPARENT:
            if (!ftp_connect_server_event(self, NULL, 0) ||
                !ftp_stream_server_init(self))
              {
              	self->state = FTP_QUIT;
              }
            else
              {
                self->state = FTP_SERVER_TO_CLIENT;
                ftp_state_set(self, EP_SERVER);
                self->ftp_state = FTP_STATE_LOGIN;
              }
            break;

          case FTP_INIT_NONTRANSPARENT:
            ftp_proto_nt_init(self);
            break;

          case FTP_NT_CLIENT_TO_PROXY:
            ftp_proto_nt_client_to_proxy(self);
            break;

          case FTP_NT_SERVER_TO_PROXY:
            ftp_proto_nt_server_to_proxy(self);
            break;

          case FTP_SERVER_TO_CLIENT:
          case FTP_CLIENT_TO_SERVER:
          case FTP_BOTH_SIDE:
            z_proxy_log(self, FTP_DEBUG, 8, "Reading from %s side;",
                        self->state == FTP_SERVER_TO_CLIENT ? "server" :
                        self->state == FTP_CLIENT_TO_SERVER ? "client" :
                        self->state == FTP_BOTH_SIDE ? "both" : "unknown");
            ftp_listen_both_side(self);
            break;

          case FTP_QUIT:
            ZPROXY_SET_FLAG(self, PF_QUIT);
            break;
        }
    }

  ZPROXY_SET_STATE(self, PS_SHUTTING_DOWN);

  ftp_data_reset(self);

  z_proxy_shutdown_event(&self->super);

  z_proxy_destroy(&self->super);

  z_leave();
  return NULL; 
}

ZProxy *
ftp_proxy_new(gchar * session_id, ZStream * client, ZPolicyObj * handler)
{
  FtpProxy *self = g_new0(FtpProxy, 1);

  z_enter();
  z_proxy_init(&self->super, session_id, client, handler);
  self->super.free = ftp_proxy_free;

  z_thread_new(session_id, ftp_thread, self);
  z_proxy_leave(self);  
  return &self->super;
}

void
ftp_proxy_free(ZProxy * s)
{
  guint i;
  FtpProxy *self = (FtpProxy *) s;

  z_proxy_enter(self);

  z_poll_quit(self->poll);
  z_poll_unref(self->poll);

  g_free(self->line);
  g_free(self->hostname);
  
  g_mutex_free(self->lock);
  
  for (i = 0; i < EP_MAX; i++)
    {
      if (self->data_local_buf[i])
        {
          z_sockaddr_unref(self->data_local_buf[i]);
          self->data_local_buf[i] = NULL;
        }
    }

  /*LOG
    This debug message attempt, when FTP proxy stopped.
   */
  z_proxy_log(self, FTP_DEBUG, 7, "Ftp Proxy bye;");
  z_proxy_leave(self);
}

/*+ Zorp initialization function +*/
gint
zorp_module_init (void)
{
  ftp_command_hash_create();
  z_registry_add ("ftp", ZR_PROXY, ftp_proxy_new);
  return TRUE;
}
