/***************************************************************************
 *
 * COPYRIGHTHERE
 *
 * $Id: httphdr.c,v 1.8.2.5 2003/09/16 19:41:50 sasa Exp $
 * 
 * Author: Balazs Scheidler <bazsi@balabit.hu>
 * Auditor: 
 * Last audited version: 
 * Notes:
 *   Based on the code by: Viktor Peter Kovacs <vps__@freemail.hu>
 *   
 ***************************************************************************/

#include "http.h"

#include <zorp/log.h>
#include <ctype.h>

gboolean
http_header_equal(gconstpointer k1, gconstpointer k2)
{
  return g_strcasecmp(k1, k2) == 0;
}

guint
http_header_hash(gconstpointer key)
{
  const char *p = key;
  guint h = toupper(*p);
    
  if (h)
    for (p += 1; *p != '\0'; p++)
      h = (h << 5) - h + toupper(*p);
                
  return h;
}

static void
http_header_free(HttpHeader *h)
{
  g_string_free(h->name, TRUE);
  g_string_free(h->value, TRUE);
  g_free(h);
}

void
http_log_headers(HttpProxy *self, gint side, gchar *tag)
{
  HttpHeaders *hdrs = &self->headers[side];
  
  if ((side == EP_CLIENT && z_log_enabled(HTTP_REQUEST, 7)) ||
      (side == EP_SERVER && z_log_enabled(HTTP_RESPONSE, 7)))
    {
      GList *l = g_list_last(hdrs->list);

      while (l)
        {
          if (side == EP_CLIENT)
	    z_proxy_log(self, HTTP_REQUEST, 7, "request %s header; hdr='%s', value='%s'", tag, 
		        ((HttpHeader *) l->data)->name->str, ((HttpHeader *) l->data)->value->str);
          else
	    z_proxy_log(self, HTTP_RESPONSE, 7, "response %s header; hdr='%s', value='%s'", tag, 
		        ((HttpHeader *) l->data)->name->str, ((HttpHeader *) l->data)->value->str);
            
	  l = g_list_previous(l);
        }
    }
}

/* duplicated headers are simply put on the list and not inserted into
   the hash, thus looking up a header by name always results the first
   added header */

HttpHeader *
http_add_header(HttpHeaders *hdrs, gchar *name, gint name_len, gchar *value, gint value_len)
{
  HttpHeader *h = g_new0(HttpHeader, 1);
  GList *l;

  h->name = g_string_sized_new(name_len + 1);

  g_string_assign_len(h->name, name, name_len);

  h->value = g_string_sized_new(value_len + 1);
  g_string_assign_len(h->value, value, value_len);
  h->present = TRUE;

  l = hdrs->list = g_list_prepend(hdrs->list, h);
  if (!g_hash_table_lookup(hdrs->hash, h->name->str))
    {
      g_hash_table_insert(hdrs->hash, h->name->str, l);
    }
  return h;
}

static gboolean
http_clear_header(gpointer key G_GNUC_UNUSED, 
                  gpointer value G_GNUC_UNUSED, 
                  gpointer user_data G_GNUC_UNUSED)
{
  return TRUE;
}

void
http_clear_headers(HttpHeaders *hdrs)
{
  GList *l;

  for (l = hdrs->list; l; l = l->next)
    http_header_free(l->data);
  g_list_free(hdrs->list);
  hdrs->list = NULL;
  g_string_truncate(hdrs->flat, 0);
  g_hash_table_foreach_remove(hdrs->hash, http_clear_header, NULL);
}

gboolean
http_lookup_header(HttpHeaders *headers, gchar *what, HttpHeader **p)
{
  GList *l;

  l = g_hash_table_lookup(headers->hash, what);
  if (l)
    {
      *p = l->data;
      return TRUE;
    }
  *p = NULL;
  return FALSE;
}

static void
http_insert_headers(gchar *key, ZPolicyObj *f, HttpHeaders *hdrs)
{
  guint filter_type;
  gchar *value;

  if (!http_hash_get_type(f, &filter_type))
    {
      /* filter has no type field */
      return;
    }
  switch (filter_type)
    {
    case HTTP_HDR_INSERT:
    case HTTP_HDR_REPLACE:
      if (!z_policy_var_parse(f, "(is)", &filter_type, &value))
	{
	  /* error parsing HTTP_INSERT or HTTP_REPLACE rule */
	  return;
	}

      http_add_header(hdrs, key, strlen(key), value, strlen(value));
      break;
    default:
      break;
    }
}

gboolean
http_filter_headers(HttpProxy *self, guint side, HttpHeaderFilter filter)
{
  HttpHeaders *headers = &self->headers[side];
  GHashTable *hash = (side == EP_CLIENT) ? self->request_header_policy : self->response_header_policy;
  gint action;
  GList *l;

  z_proxy_enter(self);
  l = headers->list;
  while (l)
    {
      HttpHeader *h = (HttpHeader *) l->data;
      ZPolicyObj *f;


      if (filter)
	action = filter(self, h->name, h->value);
      else
	action = HTTP_HDR_ACCEPT;

      g_string_assign_len(self->current_header_name, h->name->str, h->name->len);
      g_string_assign_len(self->current_header_value, h->value->str, h->value->len);

      f = g_hash_table_lookup(hash, self->current_header_name->str);
      if (!f)
	f = g_hash_table_lookup(hash, "*");
      if (f)
	{
	  guint filter_type;
	  ZPolicyObj *handler, *res;
	  gchar *name, *value;
	  
	  z_policy_lock(self->super.thread);
	  if (!http_hash_get_type(f, &filter_type))
	    {
	      /* filter has no type field */
	      z_policy_unlock(self->super.thread);
	      z_proxy_leave(self);
	      return FALSE;
	    }
	  z_policy_unlock(self->super.thread);
	  switch (filter_type)
	    {
	    case HTTP_HDR_POLICY:
	      z_policy_lock(self->super.thread);
	      if (!z_policy_var_parse(f, "(iO)", &filter_type, &handler))
		{
		  /* error parsing HTTP_POLICY_CALL rule */
		  z_proxy_leave(self);
		  return FALSE;
		}
	      res = z_policy_call_object(handler, 
					 z_policy_var_build("(s#s#)", 
							    h->name->str, h->name->len, 
							    h->value->str, h->value->len),
					 self->super.session_id);
	      if (!z_policy_var_parse(res, "i", &action))
		{
		  z_proxy_log(self, HTTP_POLICY, 1, "Policy callback for header returned non-int value; header='%s'", self->current_header_name->str);
		  z_policy_var_unref(res);
		  z_policy_unlock(self->super.thread);
		  z_proxy_leave(self);
		  return FALSE;
		}
	      z_policy_var_unref(res);
	      z_policy_unlock(self->super.thread);
              g_string_assign_len(h->name, self->current_header_name->str, self->current_header_name->len);
              g_string_assign_len(h->value, self->current_header_value->str, self->current_header_value->len);
	      break;
	    case HTTP_HDR_INSERT:
	      /* insert header that already exists */
	      action = HTTP_HDR_ACCEPT;
	      break;
	    case HTTP_HDR_ACCEPT:
	      break;
	    case HTTP_HDR_REPLACE:
	    case HTTP_HDR_DROP:
	      action = HTTP_HDR_DROP;
	      break;
	    case HTTP_HDR_CHANGE_NAME:
	      z_policy_lock(self->super.thread);
	      if (!z_policy_var_parse(f, "(is)", &filter_type, &name))
		{
		  /* invalid CHANGE_NAME rule */
		  z_proxy_log(self, HTTP_POLICY, 1, "Invalid HTTP_HDR_CHANGE_NAME rule in header processing;");
		  z_policy_unlock(self->super.thread);
		  z_proxy_leave(self);
		  return FALSE;
		}
	      g_string_assign(h->name, name);
	      z_policy_unlock(self->super.thread);
	      action = HTTP_HDR_ACCEPT;
	      break;
	    case HTTP_HDR_CHANGE_VALUE:
	      z_policy_lock(self->super.thread);
	      if (!z_policy_var_parse(f, "(is)", &filter_type, &value))
		{
		  /* invalid CHANGE_VALUE rule */
		  z_proxy_log(self, HTTP_POLICY, 1, "Invalid HTTP_HDR_CHANGE_VALUE rule in header processing;");
		  z_policy_unlock(self->super.thread);
		  z_proxy_leave(self);
		  return FALSE;
		}
	      g_string_assign(h->value, value);
	      z_policy_unlock(self->super.thread);
	      action = HTTP_HDR_ACCEPT;
	      break;
	    case HTTP_HDR_CHANGE_BOTH:
	      z_policy_lock(self->super.thread);
	      if (!z_policy_var_parse(f, "(iss)", &filter_type, &name, &value))
		{
		  /* invalid CHANGE_BOTH rule */
		  z_proxy_log(self, HTTP_POLICY, 1, "Invalid HTTP_HDR_CHANGE_BOTH rule in header processing;");
		  z_policy_unlock(self->super.thread);
		  z_proxy_leave(self);
		  return FALSE;
		}
	      g_string_assign(h->name, name);
	      g_string_assign(h->value, value);
	      z_policy_unlock(self->super.thread);
	      action = HTTP_HDR_ACCEPT;
	      break;
	    case HTTP_HDR_ABORT:
	      action = HTTP_HDR_ABORT;
	      break;
	    default:
	      action = HTTP_HDR_ABORT;
	      z_proxy_log(self, HTTP_POLICY, 1, "Invalid value in header action tuple; filter_type='%d'", filter_type);
	      break;
	    }
	}
      switch (action)
	{
	case HTTP_HDR_ACCEPT:
	  break;
	case HTTP_HDR_DROP:
	  h->present = FALSE;
	  break;
	default:
	  self->error_code = HTTP_MSG_POLICY_VIOLATION;
	  z_proxy_leave(self);
	  return FALSE;
	}

      l = g_list_next(l);
    }

  z_policy_lock(self->super.thread);
  g_hash_table_foreach(hash, (GHFunc) http_insert_headers, headers);
  z_policy_unlock(self->super.thread);
  
  z_proxy_leave(self);
  return TRUE;
}

gboolean
http_fetch_headers(HttpProxy *self, int side)
{
  HttpHeaders *headers = &self->headers[side];
  gchar *line;
  GIOStatus res;
  gint line_length;
  guint count = 0;
  HttpHeader *last_hdr = NULL;

  z_proxy_enter(self);
  /* check if we have to fetch request headers */
  if (self->proto_version[side] < 0x0100)
    {
      z_proxy_leave(self);
      return TRUE;
    }
  
  http_clear_headers(headers);
  while (1)
    {
      gchar *colon, *value, c;
      gint name_len;
      
      res = z_stream_line_get(self->super.endpoints[side], &line, &line_length, NULL);
      if (res != G_IO_STATUS_NORMAL)
        {
          if (res == G_IO_STATUS_EOF && side == EP_SERVER && self->permit_null_response)
            break;
          z_proxy_log(self, HTTP_ERROR, 3, "Error reading from peer while fetching headers;");
          z_proxy_leave(self);
	  return FALSE;
	}
      if (line_length == 0)
        {
          break;
        }
      if (*line == ' ' || *line == '\t')
        {
          /* continuation line */
          
          /* skip whitespace */
          while (line_length && (*line == ' ' || *line == '\t'))
            {
              line++;
              line_length--;
            }
          if (last_hdr) 
            {
              g_string_append_len(last_hdr->value, line, line_length);
            }
          else
            {
              /* first line is a continuation line? bad */
              z_proxy_log(self, HTTP_VIOLATION, 2, "First header starts with white space; line='%.*s", line_length, line);
              z_proxy_leave(self);
              return FALSE;
            }
          goto next_header;
        }

      name_len = 0;
      c = line[name_len];
      while (name_len < line_length &&
             !(c == '(' || c == ')' || c == '<' || c == '>' || c == '@' ||
               c == ',' || c == ';' || c == ':' || c == '\\' || c == '"' ||
               c == '/' || c == '[' || c == ']' || c == '?' || c == '=' ||
               c == '{' || c == '}' || c == ' ' || c == '\t'))
        {
          name_len++;
          c = line[name_len];
        }

      for (colon = &line[name_len]; (colon - line) < line_length && *colon == ' ' && *colon != ':'; colon++)
        ;
      if (*colon != ':')
        {
          z_proxy_log(self, HTTP_VIOLATION, 2, "Invalid HTTP header; line='%.*s'", line_length, line);
          if (self->strict_header_checking)
            {
              z_proxy_leave(self);
	      return FALSE;
	    }
	  else
	    goto next_header;
        }
      /* strip trailing white space */
      while (line_length > 0 && line[line_length - 1] == ' ') 
        line_length--;

      for (value = colon + 1; (value - line) < line_length && *value == ' '; value++)
        ;

      last_hdr = http_add_header(headers, line, name_len, value, &line[line_length] - value);
    next_header:

      count++;
      if (count > self->max_header_lines)
        {
          /* too many headers */
          z_proxy_log(self, HTTP_POLICY, 2, "Too many header lines; max_header_lines='%d'", self->max_header_lines);
          z_proxy_leave(self);
          return FALSE;
        }

    }
  /*  g_string_append(headers, "\r\n"); */
  
  http_log_headers(self, side, "prefilter");
  
  z_proxy_leave(self);
  return TRUE;
}

gboolean
http_flat_headers(HttpHeaders *hdrs)
{
  GList *l = g_list_last(hdrs->list);

  g_string_truncate(hdrs->flat, 0);
  while (l)
    {
      if (((HttpHeader *) l->data)->present)
        g_string_sprintfa(hdrs->flat, "%s: %s\r\n", ((HttpHeader *) l->data)->name->str, ((HttpHeader *) l->data)->value->str);
      l = g_list_previous(l);
    }

  return TRUE;
}

void
http_init_headers(HttpHeaders *hdrs)
{
  hdrs->hash = g_hash_table_new(http_header_hash, http_header_equal);
  hdrs->flat = g_string_sized_new(256);
}

void
http_destroy_headers(HttpHeaders *hdrs)
{
  http_clear_headers(hdrs);
  g_hash_table_destroy(hdrs->hash);
  g_string_free(hdrs->flat, TRUE);
}
