/***************************************************************************
 *
 * Copyright (c) 2000, 2001, 2002, 2003, 2004 BalaBit IT Ltd, Budapest, Hungary
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * Note that this permission is granted for only version 2 of the GPL.
 *
 * As an additional exemption you are allowed to compile & link against the
 * OpenSSL libraries as published by the OpenSSL project. See the file
 * COPYING for details.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: packsock.c,v 1.73 2004/05/26 09:06:28 sasa Exp $
 *
 * Author  : yeti
 * Auditor : bazsi
 * Last audited version: 1.39
 * Notes:
 *
 ***************************************************************************/

#include <zorp/packsock.h>
#include <zorp/log.h>

#if ENABLE_CONNTRACK
#include <zorp/sockaddr.h>
#include <zorp/socket.h>
#include <zorp/zorp.h>
#include <zorp/io.h>
#include <zorp/stream.h>
#include <zorp/cap.h>
#include <zorp/sysdep.h>
#include <zorp/tpsocket.h>

#include <time.h>
#include <netinet/ip.h>
#include <netinet/udp.h>
#include <sys/uio.h>
#endif

/*
 * Essentially, ZPacket is a buffer and its length, this module is mainly used
 * in UDP connections as low-level layer. However, the implementation of these
 * functions depend on whether linux-2.2-ipf, tproxy-ipf or tproxy-netfilter
 * support is specified/available.
 * FIXME: what are the differences between them
 */

typedef struct _ZPacksockFuncs
{
  gint (*open)(guint flags, ZSockAddr *remote, ZSockAddr *local, guint32 sock_flags, gint tos, GError **error);
  GIOStatus (*recv)(gint fd, ZPacket **pack, ZSockAddr **from, ZSockAddr **to, gint *tos, GError **error);
  GIOStatus (*read)(gint fd, ZPacket **pack, GError **error);
  GIOStatus (*write)(gint fd, ZPacket *pack, GError **error);
} ZPacksockFuncs;

#if ENABLE_LINUX22_TPROXY || ENABLE_NETFILTER_LINUX22_FALLBACK

/**
 * z_l22_packsock_open:
 * @flags: Additional flags: ZPS_LISTEN for incoming, ZPS_ESTABLISHED for outgoing socket
 * @remote: Address of the remote endpoint
 * @local: Address of the local endpoint
 * @sock_flags: Flags for binding, see 'z_bind' for details
 * @error: not used
 *
 * Create a new UDP socket - Linux 2.2 ipf version.
 *
 * Returns:
 * -1 on error, socket descriptor otherwise
 */
gint
z_l22_packsock_open(guint flags, ZSockAddr *remote, ZSockAddr *local, guint32 sock_flags, gint tos, GError **error G_GNUC_UNUSED)
{
  gint fd;
  gint tmp = 1;
  
  z_enter();
  fd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
  
  if (fd < 0)
    {
      /*LOG
        This message indicate that Zorp failed opening a new socket.
        It is likely that Zorp reached some resource limit.
       */
      z_log(NULL, CORE_ERROR, 3, "Error opening socket; error='%s'", g_strerror(errno));
      close(fd);
      z_leave();
      return -1;
    }
  setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, &tmp, sizeof(tmp));
  tmp = 1;
  setsockopt(fd, SOL_SOCKET, SO_BROADCAST, &tmp, sizeof(tmp));
  
  if (flags & ZPS_LISTEN)
    {
#if ZORPLIB_ENABLE_TOS
      tmp = 1;
      if (setsockopt(fd, SOL_IP, IP_RECVTOS, &tmp, sizeof(tmp)) < 0)
        {
          z_log(NULL, CORE_ERROR, 3, "Error during setsockopt(SOL_IP, IP_RECVTOS); error='%s'", g_strerror(errno));
          close(fd);
          z_leave();
          return -1;
        }
#endif

      if (z_bind(fd, local, sock_flags) != G_IO_STATUS_NORMAL)
        {
          /* z_bind already issued a log message */
          z_leave();
          return -1;
        }
    }
  else if (flags & ZPS_ESTABLISHED)
    {
      if (local && z_bind(fd, local, sock_flags) != G_IO_STATUS_NORMAL)
        {
          close(fd);
          z_leave();
          return -1;
        }
      z_fd_set_our_tos(fd, tos);
      if (connect(fd, &remote->sa, remote->salen) == -1)
        {
          /*LOG
            This message indicates that Zorp was unable to establish a UDP connection.
           */
          z_log(NULL, CORE_ERROR, 3, "Error connecting UDP socket (l22); error='%s'", g_strerror(errno));
          close(fd);
          z_leave();
          return -1;
        }
        
    }
  return fd;
}

/**
 * z_l22_packsock_recv:
 * @fd: Socket descriptor to read from
 * @packet: The received packet
 * @from_addr: Address of the remote endpoint
 * @to_addr: Address of the local endpoint
 * @error: not used
 *
 * Receive data from an UDP socket and encapsulate it in a ZPacket.
 * Provides address information about the source and destination of
 * the packet. - Linux 2.2 ipf version.
 *
 * Returns:
 * The status of the operation
 */
GIOStatus
z_l22_packsock_recv(gint fd, ZPacket **packet, 
                    ZSockAddr **from_addr, ZSockAddr **to_addr, 
                    gint *tos,
                    GError **error G_GNUC_UNUSED)
{
  struct sockaddr_in from;
  gchar buf[65536], ctl_buf[64];
  struct msghdr msg;
  struct iovec iov;
  struct cmsghdr *cmsg;
            
  gint rc;
  
  z_enter();

  memset(&msg, 0, sizeof(msg));
  msg.msg_name = &from;
  msg.msg_namelen = sizeof(from);
  msg.msg_controllen = sizeof(ctl_buf);
  msg.msg_control = ctl_buf;
  msg.msg_iovlen = 1;
  msg.msg_iov = &iov;
  iov.iov_base = buf;
  iov.iov_len = sizeof(buf);

  do
    {                
      rc = recvmsg(fd, &msg, MSG_PROXY);
      
    }
  while (rc < 0 && errno == EINTR);
  
  if (rc < 0)
    {
      z_leave();
      return errno == EAGAIN ? G_IO_STATUS_AGAIN : G_IO_STATUS_ERROR;
    }
  *packet = z_packet_new();
  
  z_packet_set_data(*packet, buf, rc);
  *from_addr = z_sockaddr_inet_new2(&from);
  
  if (to_addr)
    {
      
      if (((struct sockaddr_in *) &from.sin_zero)->sin_family)
        {
          *to_addr = z_sockaddr_inet_new2((struct sockaddr_in *) &from.sin_zero);
        }
      else
        {
          struct sockaddr_in to;
          socklen_t tolen = sizeof(to);
          
          getsockname(fd, (struct sockaddr *) &to, &tolen);
          *to_addr = z_sockaddr_inet_new2(&to);
        }
    }
  if (tos)
    {
      for (cmsg = CMSG_FIRSTHDR(&msg); cmsg; cmsg = CMSG_NXTHDR(&msg, cmsg))
        {
          if (cmsg->cmsg_level == SOL_IP && cmsg->cmsg_type == IP_TOS)
            {
              *tos = *(guchar *) CMSG_DATA(cmsg);
              break;
            }
        }
    }
  z_leave();
  return G_IO_STATUS_NORMAL;
  
}

/**
 * z_l22_packsock_write:
 * @fd: Socket descriptor to write to
 * @packet: The packet to write
 * @error: not used
 *
 * Send a packet to an UDP socket - Linux 2.2 ipf version.
 * Destination address is implicit, specified by socket creation.
 *
 * Returns:
 * The status of the operation
 */
GIOStatus
z_l22_packsock_write(gint fd, ZPacket *packet, GError **error G_GNUC_UNUSED)
{
  int rc;

  z_enter();
  do
    {
      rc = send(fd, packet->data, packet->length, 0);
    }
  while (rc < 0 && errno == EINTR);
  if (rc < 0)
    {
      if (errno == EAGAIN)
        rc = G_IO_STATUS_AGAIN;
      else
        rc = G_IO_STATUS_ERROR;
    }
  else
    rc = G_IO_STATUS_NORMAL;
    
  z_leave();
  return rc;
}

/**
 * z_l22_packsock_read:
 * @fd: Socket descriptor to read from
 * @packet: The received packet
 * @error: not used
 *
 * Receive data from an UDP socket and encapsulate it in a ZPacket.
 * Similar to _recv, but doesn't provide address information.
 * - Linux 2.2 ipf version.
 *
 * Returns:
 * The status of the operation
 */
GIOStatus
z_l22_packsock_read(gint fd, ZPacket **packet, GError **error G_GNUC_UNUSED)
{
  gchar buf[65536];
  GIOStatus res;
  gint rc;

  z_enter();
  
  do
    {
      rc = recv(fd, buf, sizeof(buf), 0);
    }
  while (rc < 0 && errno == EINTR);
  
  if (rc < 0)
    {
      if (errno == EAGAIN)
        res = G_IO_STATUS_AGAIN;
      else
        res = G_IO_STATUS_ERROR;
    }
  else
    {
      res = G_IO_STATUS_NORMAL;
      *packet = z_packet_new();
      z_packet_set_data(*packet, buf, rc);
    }

  z_leave();
  return res;
}

ZPacksockFuncs z_l22_packsock_funcs = 
{
  z_l22_packsock_open,
  z_l22_packsock_recv,
  z_l22_packsock_read,
  z_l22_packsock_write
};

#endif

#if ENABLE_NETFILTER_TPROXY

#include <zorp/nfiptproxy-kernelv2.h>

#ifndef IP_RECVORIGADDRS_V12
#define IP_RECVORIGADDRS_V12 16
#define IP_ORIGADDRS_V12     17
#endif

#endif


#if ENABLE_NETFILTER_TPROXY || ENABLE_IPFILTER_TPROXY
static gint z_nf_recvorigaddrs_opt;

#if ENABLE_NETFILTER_TPROXY
static gint z_nf_origaddrs_opt;
#endif /* ENABLE_NETFILTER_TPROXY */


/**
 * z_nf_packsock_open:
 * @flags: Additional flags: ZPS_LISTEN for incoming, ZPS_ESTABLISHED for outgoing socket
 * @remote: Address of the remote endpoint
 * @local: Address of the local endpoint
 * @sock_flags: Flags for binding, see 'z_bind' for details
 * @error: not used
 *
 * Create a new UDP socket - netfilter tproxy version
 * FIXME: some words about the difference
 *
 * Returns:
 * -1 on error, socket descriptor otherwise
 */
gint
z_nf_packsock_open(guint flags, ZSockAddr *remote, ZSockAddr *local, guint32 sock_flags, gint tos, GError **error G_GNUC_UNUSED)
{
  gint fd;
  gint tmp = 1;
  
  z_enter();
  fd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
  
  if (fd < 0)
    {
      /*LOG
        This message indicate that Zorp failed opening a new socket.
        It is likely that Zorp reached some resource limit.
       */
      z_log(NULL, CORE_ERROR, 3, "Error opening socket; error='%s'", g_strerror(errno));
      close(fd);
      z_leave();
      return -1;
    }
  setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, &tmp, sizeof(tmp));
  tmp = 1;
  setsockopt(fd, SOL_SOCKET, SO_BROADCAST, &tmp, sizeof(tmp));

  if (flags & ZPS_LISTEN)
    {
      tmp = 1;
      if (setsockopt(fd, SOL_IP, z_nf_recvorigaddrs_opt, &tmp, sizeof(tmp)) < 0)
        {
          /*LOG
            This message indicates that the setsockopt failed.
           */
          z_log(NULL, CORE_ERROR, 3, "Error during setsockopt(SOL_IP, IP_RECVORIGADDRS); error='%s'", g_strerror(errno));
          close(fd);
          z_leave();
          return -1;
        }
#if ZORPLIB_ENABLE_TOS
      tmp = 1;
      if (setsockopt(fd, SOL_IP, IP_RECVTOS, &tmp, sizeof(tmp)) < 0)
        {
          z_log(NULL, CORE_ERROR, 3, "Error during setsockopt(SOL_IP, IP_RECVTOS); error='%s'", g_strerror(errno));
          close(fd);
          z_leave();
          return -1;
        }
#endif
      if (z_bind(fd, local, sock_flags) != G_IO_STATUS_NORMAL)
        {
          /* z_bind already issued a log message */
          z_leave();
          return -1;
        }
      if (z_tp_query(fd, NULL, NULL) != -1)
        {
          if (z_tp_set_flags(fd, ITP_LISTEN | ITP_UNIDIR) < 0)
            {
              /*LOG
                This message indicates that the setsockopt failed, and Zorp can not listen on a foreign address.
               */
              z_log(NULL, CORE_ERROR, 3, "Error during setsockopt(SOL_IP, IP_TPROXY_FLAGS, ITP_LISTEN | ITP_UNIDIR); error='%s'", g_strerror(errno));
              close(fd);
              z_leave();
              return -1;
            }
        }
    }
  else if (flags & ZPS_ESTABLISHED)
    {
      if (local && z_bind(fd, local, sock_flags) != G_IO_STATUS_NORMAL)
        {
          close(fd);
          z_leave();
          return -1;
        }
      z_fd_set_our_tos(fd, tos);
      /* NOTE: we use connect instead of z_connect, as we do tproxy calls ourselves */
      if (connect(fd, &remote->sa, remote->salen) < 0)
        {
          /*LOG
            This message indicates that UDP connection failed.
           */
          z_log(NULL, CORE_ERROR, 3, "Error connecting UDP socket (nf); error='%s'", g_strerror(errno));
          close(fd);
          z_leave();
          return -1;
        }
      
      if (!local)
        {
          
          if (z_getsockname(fd, &local, ZSF_MARK_TPROXY) != G_IO_STATUS_NORMAL)
            {
              /*LOG
                This message indicates that Zorp was unable to get a local address.
               */
              z_log(NULL, CORE_ERROR, 3, "Error getting dynamic local address (nf); error='%s'", g_strerror(errno));
              close(fd);
              z_leave();
              return -1;
            }
          /* reopen fd so that we are able to call z_bind() */
          close(fd);
          fd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
          
          if (fd < 0)
            {
              /*LOG
                This message indicate that Zorp failed opening a new socket.
                It is likely that Zorp reached some resource limit.
               */
              z_log(NULL, CORE_ERROR, 3, "Error opening socket; error='%s'", g_strerror(errno));
              close(fd);
              z_leave();
              return -1;
            }
          
          setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, &tmp, sizeof(tmp));
          tmp = 1;
          setsockopt(fd, SOL_SOCKET, SO_BROADCAST, &tmp, sizeof(tmp));

          if (z_bind(fd, local, sock_flags) != G_IO_STATUS_NORMAL)
            {
              z_sockaddr_unref(local);
              close(fd);
              z_leave();
              return -1;
            }
          z_sockaddr_unref(local);
          local = NULL;
          if (connect(fd, &remote->sa, remote->salen) < 0)
            {
              /*LOG
               */
              z_log(NULL, CORE_ERROR, 3, "Error reconnecting UDP socket (nf); error='%s'", g_strerror(errno));
              close(fd);
              z_leave();
              return -1;
            }
        }
      
      if (z_tp_query(fd, NULL, NULL) != -1)
        {
          z_tp_connect(fd, ((struct sockaddr_in *) &remote->sa)->sin_addr.s_addr, ((struct sockaddr_in *) &remote->sa)->sin_port); 
          if (z_tp_set_flags(fd, ITP_ESTABLISHED) < 0) 
            {
              /*LOG
               */
              z_log(NULL, CORE_ERROR, 3, "Error during setsockopt(SOL_IP, IP_TPROXY_FLAGS, ITP_ESTABLISHED); error='%s'", g_strerror(errno));
              close(fd);
              z_leave();
              return -1;
            }
        }
    }
  return fd;
}

#endif /* ENABLE_NETFILTER_TPROXY || ENABLE_IPFILTER_TPROXY */

#if ENABLE_NETFILTER_TPROXY

/**
 * z_nf_packsock_recv:
 * @fd: Socket descriptor to read from
 * @packet: The received packet
 * @from_addr: Address of the remote endpoint
 * @to_addr: Address of the local endpoint
 * @error: not used
 *
 * Receive data from an UDP socket and encapsulate it in a ZPacket.
 * Provides address information about the source and destination of
 * the packet. - netfilter tproxy version.
 * FIXME: some words about the difference
 *
 * Returns:
 * The status of the operation
 */
GIOStatus
z_nf_packsock_recv(gint fd, ZPacket **packet, ZSockAddr **from_addr, ZSockAddr **to_addr, gint *tos, GError **error G_GNUC_UNUSED)
{
  struct sockaddr_in from, to;
  gchar buf[65536], ctl_buf[64];
  struct msghdr msg;
  struct cmsghdr *cmsg;
  struct iovec iov;
            
  gint rc;
  
  z_enter();

  memset(&msg, 0, sizeof(msg));
  msg.msg_name = &from;
  msg.msg_namelen = sizeof(from);
  msg.msg_controllen = sizeof(ctl_buf);
  msg.msg_control = ctl_buf;
  msg.msg_iovlen = 1;
  msg.msg_iov = &iov;
  iov.iov_base = buf;
  iov.iov_len = sizeof(buf);
                
  do
    {
      rc = recvmsg(fd, &msg, 0);
    }
  while (rc < 0 && errno == EINTR);
  
  if (rc < 0)
    {
      z_leave();
      return errno == EAGAIN ? G_IO_STATUS_AGAIN : G_IO_STATUS_ERROR;
    }
  *packet = z_packet_new();
  
  z_packet_set_data(*packet, buf, rc);
  *from_addr = z_sockaddr_inet_new2(&from);
  
  if (to_addr || tos)
    {
      if (to_addr)
        *to_addr = NULL;
      if (tos)
        *tos = -1;
      for (cmsg = CMSG_FIRSTHDR(&msg); cmsg; cmsg = CMSG_NXTHDR(&msg,cmsg))
        {
          if (to_addr && cmsg->cmsg_level == SOL_IP && cmsg->cmsg_type == z_nf_origaddrs_opt)
            {
              struct in_origaddrs *ioa = (struct in_origaddrs *) CMSG_DATA(cmsg);

              if (ioa->ioa_dstaddr.s_addr && ioa->ioa_dstport)
                {
                  to.sin_family = AF_INET;
                  to.sin_addr = ioa->ioa_dstaddr;
                  to.sin_port = ioa->ioa_dstport;
                  *to_addr = z_sockaddr_inet_new2(&to);
                }
            }
          else if (tos && cmsg->cmsg_level == SOL_IP && cmsg->cmsg_type == IP_TOS)
            {
              *tos = *(int *) CMSG_DATA(cmsg);
            }
        }
      if (to_addr && *to_addr == NULL)
        {
          struct sockaddr_in to;
          socklen_t tolen = sizeof(to);
                             
          getsockname(fd, (struct sockaddr *) &to, &tolen);
          *to_addr = z_sockaddr_inet_new2(&to);
        }
    }

  z_leave();
  return G_IO_STATUS_NORMAL;
  
}

#endif /* ENABLE_NETFILTER_TPROXY */

#if ENABLE_NETFILTER_TPROXY || ENABLE_IPFILTER_TPROXY

/**
 * z_nf_packsock_write:
 * @fd: Socket descriptor to write to
 * @packet: The packet to write
 * @error: not used
 *
 * Send a packet to an UDP socket - netfilter tproxy version.
 * Destination address is implicit, specified by socket creation.
 * FIXME: some words about the difference
 *
 * Returns:
 * The status of the operation
 */
GIOStatus
z_nf_packsock_write(gint fd, ZPacket *packet, GError **error G_GNUC_UNUSED)
{
  int rc;

  z_enter();
  do
    {
      rc = send(fd, packet->data, packet->length, 0);
    }
  while (rc < 0 && errno == EINTR);
  if (rc < 0)
    {
      z_leave();
      return errno == EAGAIN ? G_IO_STATUS_AGAIN : G_IO_STATUS_ERROR;
    }
  z_leave();
  return G_IO_STATUS_NORMAL;
}

/**
 * z_nf_packsock_read:
 * @fd: Socket descriptor to read from
 * @packet: The received packet
 * @error: not used
 *
 * Receive data from an UDP socket and encapsulate it in a ZPacket.
 * Similar to _recv, but doesn't provide address information.
 * - netfilter tproxy version.
 * FIXME: some words about the difference
 *
 * Returns:
 * The status of the operation
 */
GIOStatus 
z_nf_packsock_read(gint fd, ZPacket **packet, GError **error G_GNUC_UNUSED)
{
  gchar buf[65536];
  gint rc;

  z_enter();
  
  do
    {
      rc = recv(fd, buf, sizeof(buf), 0);
    }
  while (rc < 0 && errno == EINTR);
  
  if (rc < 0)
    {
      z_leave();
      return errno == EAGAIN ? G_IO_STATUS_AGAIN : G_IO_STATUS_ERROR;
    }
  *packet = z_packet_new();
  z_packet_set_data(*packet, buf, rc);

  z_leave();
  return G_IO_STATUS_NORMAL;
}

#endif

#if ENABLE_NETFILTER_TPROXY
ZPacksockFuncs z_nf_packsock_funcs = 
{
  z_nf_packsock_open,
  z_nf_packsock_recv,
  z_nf_packsock_read,
  z_nf_packsock_write
};

#endif


ZPacksockFuncs *packsock_funcs;

/**
 * z_packsock_open:
 *
 * Generic packsock_open, will use the enabled one of _l22_, _nf_ or _ipf_.
 */
gint 
z_packsock_open(guint flags, ZSockAddr *remote, ZSockAddr *local, guint32 sock_flags, gint tos, GError **error)
{
  return packsock_funcs->open(flags, remote, local, sock_flags, tos, error);
}

/**
 * z_packsock_recv:
 *
 * Generic packsock_recv, will use the enabled one of _l22_, _nf_ or _ipf_.
 */
GIOStatus 
z_packsock_recv(gint fd, ZPacket **pack, ZSockAddr **from, ZSockAddr **to, gint *tos, GError **error)
{
  return packsock_funcs->recv(fd, pack, from, to, tos, error);
}

/**
 * z_packsock_read:
 *
 * Generic packsock_read, will use the enabled one of _l22_, _nf_ or _ipf_.
 */
GIOStatus 
z_packsock_read(gint fd, ZPacket **pack, GError **error)
{
  return packsock_funcs->read(fd, pack, error);
}

/**
 * z_packsock_write:
 *
 * Generic packsock_write, will use the enabled one of _l22_, _nf_ or _ipf_.
 */
GIOStatus 
z_packsock_write(gint fd, ZPacket *pack, GError **error)
{
  return packsock_funcs->write(fd, pack, error);
}

/**
 * z_packsock_init:
 * @sysdep_tproxy: Required functionality to use: Z_SD_TPROXY_[LINUX22|NETFILTER_V12|NETFILTER_V20]
 *
 * Module initialisation, initialises the function table according to the
 * requested transparency method.
 *
 * Returns:
 * TRUE on success
 */
gboolean
z_packsock_init(gint sysdep_tproxy)
{
#if ENABLE_CONNTRACK
  switch (sysdep_tproxy)
    {
#if ENABLE_LINUX22_TPROXY || ENABLE_NETFILTER_LINUX22_FALLBACK
    case Z_SD_TPROXY_LINUX22:
      packsock_funcs = &z_l22_packsock_funcs;
      break;
#endif
#if ENABLE_NETFILTER_TPROXY
    case Z_SD_TPROXY_NETFILTER_V12:
      z_nf_recvorigaddrs_opt = IP_RECVORIGADDRS_V12;
      z_nf_origaddrs_opt = IP_ORIGADDRS_V12;
      packsock_funcs = &z_nf_packsock_funcs;
      break;
    case Z_SD_TPROXY_NETFILTER_V20:
      z_nf_recvorigaddrs_opt = IP_RECVORIGADDRS;
      z_nf_origaddrs_opt = IP_ORIGADDRS;
      packsock_funcs = &z_nf_packsock_funcs;
      break;
#endif
    default:
      /*LOG
        This message indicates that Zorp was compiled without the required transparency support for UDP, or bad 
        transparency support was specified on command line.
        Check your "instances.conf" or your kernel tproxy capabilities.
       */
      z_log(NULL, CORE_ERROR, 0, "Required transparency support not compiled in (UDP); sysdep_tproxy='%d'", sysdep_tproxy);
      return FALSE;
    }
#endif
  return TRUE;
} 
