/***************************************************************************
 *   
 * Copyright (c) 2000, 2001, 2002, 2003, 2004 BalaBit IT Ltd, Budapest, Hungary
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * Note that this permission is granted for only version 2 of the GPL.
 *
 * As an additional exemption you are allowed to compile & link against the
 * OpenSSL libraries as published by the OpenSSL project. See the file
 * COPYING for details.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: plugsession.c,v 1.18 2004/07/01 16:55:25 bazsi Exp $
 *
 * Author  : bazsi
 * Auditor : 
 * Last audited version: 1.1
 * Notes:
 *   
 ***************************************************************************/

#include <zorp/plugsession.h>

#include <zorp/log.h>
#include <zorp/stream.h>
#include <zorp/source.h>

typedef struct _ZPlugIOBuffer
{
  gchar *buf;
  gsize ofs, end;
  gsize packet_count, packet_bytes;
} ZPlugIOBuffer;

struct _ZPlugSession
{
  ZProxy *owner;
  ZSessionVars *vars;
  ZPlugSessionData *session_data;
  ZPoll *poll;
  ZStream *endpoints[EP_MAX];
  ZStackedProxy *stacked;
  ZPlugIOBuffer buffers[EP_MAX];
  ZPlugIOBuffer downbufs[EP_MAX];
  gint eofmask;
  GSource *timeout;
  GSource *stats_timeout;
  GTimeVal started_time;
  guint global_packet_count;
};

/* possible eofmask values */
#define EOF_CLIENT_R         0x0001
#define EOF_SERVER_R         0x0002
#define EOF_CLIENT_W         0x0004
#define EOF_SERVER_W         0x0008
#define EOF_CLIENT_REMOVED   0x0010
#define EOF_SERVER_REMOVED   0x0020
#define EOF_DESTROYED        0x0040

#define EOF_ALL              0x000f

static void
z_plug_update_eof_mask(ZPlugSession *self, guint add_mask)
{
  guint old_mask = self->eofmask;
  
  self->eofmask |= add_mask;
  
  if ((self->eofmask & (EOF_CLIENT_R | EOF_CLIENT_W | EOF_CLIENT_REMOVED)) == (EOF_CLIENT_R | EOF_CLIENT_W))
    {
      z_poll_remove_stream(self->poll, self->endpoints[EP_CLIENT]);
      self->eofmask |= EOF_CLIENT_REMOVED;
    }

  if ((self->eofmask & (EOF_SERVER_R | EOF_SERVER_W | EOF_SERVER_REMOVED)) == (EOF_SERVER_R | EOF_SERVER_W))
    {
      z_poll_remove_stream(self->poll, self->endpoints[EP_SERVER]);
      self->eofmask |= EOF_SERVER_REMOVED;
    }
  
  if ((self->eofmask & (EOF_DESTROYED | EOF_CLIENT_REMOVED | EOF_SERVER_REMOVED)) == (EOF_CLIENT_REMOVED | EOF_SERVER_REMOVED))
    {
      self->session_data->destroy_queue = g_list_prepend(self->session_data->destroy_queue, self);
      self->eofmask |= EOF_DESTROYED;
    }
  /*LOG
    This message reports that the end-of-file status has been updated.
   */
  z_proxy_log(self->owner, CORE_DEBUG, 7, "eofmask updated; old_mask='%04x', eof_mask='%04x'", old_mask, self->eofmask);
}



static guint
z_plug_read_input(ZPlugSession *self, ZStream *input, ZPlugIOBuffer *buf)
{
  GIOStatus rc;

  z_proxy_enter(self->owner);
  rc = z_stream_read(input, buf->buf, self->session_data->buffer_size, &buf->end, NULL);
  if (rc == G_IO_STATUS_NORMAL)
    {
      buf->packet_bytes += buf->end;
      buf->packet_count++;
      self->global_packet_count++;
      if (self->session_data->packet_stats_interval_packet &&
         (self->global_packet_count % self->session_data->packet_stats_interval_packet) == 0)
        {
          if (!self->session_data->packet_stats(self->owner, self->vars, 
                                                self->buffers[EP_CLIENT].packet_bytes, 
                                                self->buffers[EP_CLIENT].packet_count,
                                                self->buffers[EP_SERVER].packet_bytes,
                                                self->buffers[EP_SERVER].packet_count))
            z_plug_update_eof_mask(self, EOF_ALL);
        }
    }
  z_proxy_leave(self->owner);
  return rc;
}

static GIOStatus
z_plug_write_output(ZPlugSession *self G_GNUC_UNUSED, ZPlugIOBuffer *buf, ZStream *output)
{
  GIOStatus rc;
  gsize bytes_written;
  
  z_proxy_enter(self->owner);
  if (buf->ofs != buf->end)
    {
      /* buffer not empty */
      rc = z_stream_write(output, &buf->buf[buf->ofs], buf->end - buf->ofs, &bytes_written, NULL);
      switch (rc) 
        {
        case G_IO_STATUS_NORMAL:
          buf->ofs += bytes_written;
          break;
        case G_IO_STATUS_AGAIN:
          break;
        default:
          z_proxy_leave(self->owner);
          return rc;
        }
      if (buf->ofs != buf->end)
        {
          z_stream_set_cond(output,
                           Z_STREAM_FLAG_WRITE,
                           TRUE);
          z_proxy_leave(self->owner);
          return G_IO_STATUS_AGAIN;
        }
      else
        {
          z_proxy_leave(self->owner);
          return G_IO_STATUS_NORMAL;
        }
    }
  z_proxy_leave(self->owner);
  return G_IO_STATUS_NORMAL;
}

static GIOStatus
z_plug_copy_data(ZPlugSession *self, ZStream *from, ZStream *to, ZPlugIOBuffer *buf)
{
  GIOStatus rc = G_IO_STATUS_ERROR;
  int pkt_count = 0;

  z_proxy_enter(self->owner);
  
  if (self->timeout)
    z_timeout_source_set_timeout(self->timeout, self->session_data->timeout);
  
  z_stream_set_cond(from,
                   Z_STREAM_FLAG_READ,
                   FALSE);

  if (to)
    {
      z_stream_set_cond(to,
                       Z_STREAM_FLAG_WRITE,
                       FALSE);
      rc = z_plug_write_output(self, buf, to);
      if (rc != G_IO_STATUS_NORMAL)
        {
          z_proxy_leave(self->owner);
          return rc;
        }
    }

  while (1) 
    {  
      buf->ofs = buf->end = 0;
      rc = z_plug_read_input(self, from, buf);
      if (rc == G_IO_STATUS_NORMAL)
        {
          if (to)
            {
              rc = z_plug_write_output(self, buf, to);
              if (rc == G_IO_STATUS_AGAIN)
                break;
              else if (rc != G_IO_STATUS_NORMAL)
                {
                  z_proxy_leave(self->owner);
                  return rc;
                }
            }
        }
      else if (rc == G_IO_STATUS_AGAIN)
        break;
      else if (rc == G_IO_STATUS_EOF)
        {
          z_proxy_leave(self->owner);
          return rc;
        }
      else
        {
          z_proxy_leave(self->owner);
          return G_IO_STATUS_ERROR;
        }
      pkt_count++;
    }

  if (buf->ofs == buf->end)
    {
      z_stream_set_cond(from,
                       Z_STREAM_FLAG_READ,
                       TRUE);
    }
  z_proxy_leave(self->owner);
  return rc;
}


/* callbacks when no stacking is made */
static gboolean
z_plug_copy_client_to_server(ZStream *stream G_GNUC_UNUSED,
                      GIOCondition  cond G_GNUC_UNUSED,
                          gpointer  user_data)
{
  ZPlugSession *self = (ZPlugSession *)user_data;
  gboolean ret;

  z_proxy_enter(self->owner);
  if (self->session_data->copy_to_server)
    ret = z_plug_copy_data(self,
                         self->endpoints[EP_CLIENT],
                         self->endpoints[EP_SERVER],
                         &self->buffers[EP_SERVER]);
  else
    ret = z_plug_copy_data(self,
                         self->endpoints[EP_CLIENT],
                         NULL,
                         &self->buffers[EP_SERVER]);

  switch (ret)
    {
    case G_IO_STATUS_NORMAL:
    case G_IO_STATUS_AGAIN:
      break;
    case G_IO_STATUS_EOF:
      if (self->session_data->shutdown_soft)
        {
          z_stream_shutdown(self->endpoints[EP_CLIENT], SHUT_RD, NULL);
          z_stream_shutdown(self->endpoints[EP_SERVER], SHUT_WR, NULL);
          z_plug_update_eof_mask(self, EOF_CLIENT_R | EOF_SERVER_W);
        }
      else
        {
          z_plug_update_eof_mask(self, EOF_ALL);
        }
      break;
    default:
      z_plug_update_eof_mask(self, EOF_ALL);
      z_proxy_leave(self->owner);
      return FALSE;
    }
  z_proxy_leave(self->owner);
  return TRUE;
}

static gboolean
z_plug_copy_server_to_client(ZStream *stream G_GNUC_UNUSED,
                      GIOCondition  cond G_GNUC_UNUSED,
                          gpointer  user_data)
{
  ZPlugSession *self = (ZPlugSession *) user_data;
  GIOStatus ret;

  z_proxy_enter(self->owner);
  if (self->session_data->copy_to_client)
    ret = z_plug_copy_data(self,
                         self->endpoints[EP_SERVER],
                         self->endpoints[EP_CLIENT],
                         &self->buffers[EP_CLIENT]);
  else
    ret = z_plug_copy_data(self,
                         self->endpoints[EP_SERVER],
                         NULL,
                         &self->buffers[EP_CLIENT]);

  switch (ret)
    {
    case G_IO_STATUS_NORMAL:
    case G_IO_STATUS_AGAIN:
      break;
    case G_IO_STATUS_EOF:
      if (self->session_data->shutdown_soft)
        {
          z_stream_shutdown(self->endpoints[EP_SERVER], SHUT_RD, NULL);
          z_stream_shutdown(self->endpoints[EP_CLIENT], SHUT_WR, NULL);
          z_plug_update_eof_mask(self, EOF_SERVER_R | EOF_CLIENT_W);
        }
      else
        {
          z_plug_update_eof_mask(self, EOF_ALL);
        }
      break;
    default:
      z_plug_update_eof_mask(self, EOF_ALL);
      z_proxy_leave(self->owner);
      return FALSE;
    }

  z_proxy_leave(self->owner);
  return TRUE;
}

/* callbacks when a stacked module exists */
static gboolean
z_plug_copy_client_to_down(ZStream       *stream G_GNUC_UNUSED,
                           GIOCondition  cond G_GNUC_UNUSED,
                           gpointer      user_data)
{
  ZPlugSession *self = (ZPlugSession *) user_data;
  GIOStatus ret;

  z_proxy_enter(self->owner);
  if (self->session_data->copy_to_server)
    ret = z_plug_copy_data(self,
                         self->endpoints[EP_CLIENT],
                         self->stacked->downstreams[EP_CLIENT],
                         &self->downbufs[EP_CLIENT]);
  else
    ret = z_plug_copy_data(self,
                         self->endpoints[EP_CLIENT],
                         NULL,
                         &self->downbufs[EP_CLIENT]);

  switch (ret)
    {
    case G_IO_STATUS_NORMAL:
    case G_IO_STATUS_AGAIN:
      break;
    case G_IO_STATUS_EOF:
      if (self->session_data->shutdown_soft)
        {
          z_stream_shutdown(self->endpoints[EP_CLIENT],
                            SHUT_RD,
                            NULL);
          z_stream_shutdown(self->stacked->downstreams[EP_CLIENT],
                            SHUT_WR,
                            NULL);
          z_plug_update_eof_mask(self, EOF_CLIENT_R);
        }
      else
        {
          z_plug_update_eof_mask(self, EOF_ALL);
        }
      break;
    default:
      z_plug_update_eof_mask(self, EOF_ALL);
      z_proxy_leave(self->owner);
      return FALSE;
    }

  z_proxy_leave(self->owner);
  return TRUE;
}

static gboolean
z_plug_copy_down_to_client(ZStream *stream G_GNUC_UNUSED,
                    GIOCondition  cond G_GNUC_UNUSED,
                        gpointer  user_data)
{
  ZPlugSession *self = (ZPlugSession *) user_data;
  GIOStatus ret;

  z_proxy_enter(self->owner);
  ret = z_plug_copy_data(self,
                       self->stacked->downstreams[EP_CLIENT],
                       self->endpoints[EP_CLIENT],
                       &self->buffers[EP_CLIENT]);

  switch (ret)
    {
    case G_IO_STATUS_NORMAL:
    case G_IO_STATUS_AGAIN:
      break;
    case G_IO_STATUS_EOF:
      if (self->session_data->shutdown_soft)
        {
          z_stream_shutdown(self->stacked->downstreams[EP_CLIENT],
                            SHUT_RD,
                            NULL);
          z_stream_shutdown(self->endpoints[EP_CLIENT],
                            SHUT_WR,
                            NULL);
          z_plug_update_eof_mask(self, EOF_CLIENT_W);
        }
      else
        {
          z_plug_update_eof_mask(self, EOF_ALL);
        }
      break;
    default:
      z_plug_update_eof_mask(self, EOF_ALL);
      z_proxy_leave(self->owner);
      return FALSE;
    }
  z_proxy_leave(self->owner);
  return TRUE;
}

static gboolean
z_plug_copy_server_to_down(ZStream *stream G_GNUC_UNUSED,
                    GIOCondition  cond G_GNUC_UNUSED,
                        gpointer  user_data)
{
  ZPlugSession *self = (ZPlugSession *) user_data;
  GIOStatus ret;

  z_proxy_enter(self->owner);
  if (self->session_data->copy_to_client)
    ret = z_plug_copy_data(self,
                         self->endpoints[EP_SERVER],
                         self->stacked->downstreams[EP_SERVER],
                         &self->downbufs[EP_SERVER]);
  else
    ret = z_plug_copy_data(self,
                         self->endpoints[EP_SERVER],
                         NULL,
                         &self->downbufs[EP_SERVER]);

  switch (ret)
    {
    case G_IO_STATUS_NORMAL:
    case G_IO_STATUS_AGAIN:
      break;
    case G_IO_STATUS_EOF:
      if (self->session_data->shutdown_soft)
        {
          z_stream_shutdown(self->endpoints[EP_SERVER],
                            SHUT_RD,
                            NULL);
          z_stream_shutdown(self->stacked->downstreams[EP_SERVER],
                            SHUT_WR,
                            NULL);
          z_plug_update_eof_mask(self, EOF_SERVER_R);
        }
      else
        {
          z_plug_update_eof_mask(self, EOF_ALL);
        }
      break;
    default:
      z_plug_update_eof_mask(self, EOF_ALL);
      z_proxy_leave(self->owner);
      return FALSE;
    }

  z_proxy_leave(self->owner);
  return TRUE;
}

static gboolean
z_plug_copy_down_to_server(ZStream *stream G_GNUC_UNUSED,
                    GIOCondition  cond G_GNUC_UNUSED,
                        gpointer  user_data)
{
  ZPlugSession *self = (ZPlugSession *) user_data;
  GIOStatus ret;

  z_proxy_enter(self->owner);
  ret = z_plug_copy_data(self,
                       self->stacked->downstreams[EP_SERVER],
                       self->endpoints[EP_SERVER],
                       &self->buffers[EP_SERVER]);

  switch (ret)
    {
    case G_IO_STATUS_NORMAL:
    case G_IO_STATUS_AGAIN:
      z_proxy_leave(self->owner);
      return TRUE;
    case G_IO_STATUS_EOF:
      if (self->session_data->shutdown_soft)
        {
          z_stream_shutdown(self->stacked->downstreams[EP_SERVER],
                            SHUT_RD,
                            NULL);
          z_stream_shutdown(self->endpoints[EP_SERVER],
                            SHUT_WR,
                            NULL);
          z_plug_update_eof_mask(self, EOF_SERVER_W);
        }
      else
        {
          z_plug_update_eof_mask(self, EOF_ALL);
        }
      break;
    default:
      z_plug_update_eof_mask(self, EOF_ALL);
      z_proxy_leave(self->owner);
      return FALSE;
    }

  z_proxy_leave(self->owner);
  return TRUE;
}

gboolean
z_plug_timeout(gpointer user_data)
{
  ZPlugSession *self = (ZPlugSession *) user_data;
  
  z_proxy_enter(self->owner);
  z_plug_update_eof_mask(self, EOF_ALL);
  z_proxy_leave(self->owner);
  return FALSE;
}

/* FIXME: merge these two functions */
gboolean
z_plug_session_init_streams(ZPlugSession *self)
{
  z_proxy_enter(self->owner);
  
  self->buffers[EP_CLIENT].buf = g_new0(char, self->session_data->buffer_size);
  self->buffers[EP_SERVER].buf = g_new0(char, self->session_data->buffer_size);
  
  z_stream_set_nonblock(self->endpoints[EP_CLIENT], TRUE);
  
  z_stream_set_callback(self->endpoints[EP_CLIENT],
                        Z_STREAM_FLAG_READ, 
                        z_plug_copy_client_to_server,
                        self,
                        NULL);
  z_stream_set_callback(self->endpoints[EP_CLIENT],
                        Z_STREAM_FLAG_WRITE,
                        z_plug_copy_server_to_client,
                        self,
                        NULL);
  z_stream_set_cond(self->endpoints[EP_CLIENT],
                   Z_STREAM_FLAG_READ,
                   TRUE);

  z_stream_set_timeout(self->endpoints[EP_CLIENT], -2);
  
  z_stream_set_nonblock(self->endpoints[EP_SERVER], TRUE);
  z_stream_set_callback(self->endpoints[EP_SERVER],
                        Z_STREAM_FLAG_READ,
                        z_plug_copy_server_to_client,
                        self,
                        NULL);
  z_stream_set_callback(self->endpoints[EP_SERVER],
                       Z_STREAM_FLAG_WRITE,
                       z_plug_copy_client_to_server,
                       self,
                       NULL);
  z_stream_set_cond(self->endpoints[EP_SERVER],
                   Z_STREAM_FLAG_READ,
                   TRUE);
  z_stream_set_timeout(self->endpoints[EP_SERVER], -2);

  z_poll_add_stream(self->poll, self->endpoints[EP_CLIENT]);
  z_poll_add_stream(self->poll, self->endpoints[EP_SERVER]);

  z_proxy_leave(self->owner);
  return TRUE;
}

static gboolean
z_plug_session_init_stacked_streams(ZPlugSession *self)
{
  z_proxy_enter(self->owner);
  
  if (self->stacked)
    {
      self->downbufs[EP_CLIENT].buf = g_new0(char, self->session_data->buffer_size);
      self->downbufs[EP_SERVER].buf = g_new0(char, self->session_data->buffer_size);

      z_stream_set_callback(self->endpoints[EP_CLIENT],
                            Z_STREAM_FLAG_READ,
                            z_plug_copy_client_to_down,
                            self,
                            NULL);

      z_stream_set_callback(self->endpoints[EP_CLIENT],
                            Z_STREAM_FLAG_WRITE,
                            z_plug_copy_down_to_client,
                            self,
                            NULL);

      z_stream_set_callback(self->endpoints[EP_SERVER],
                            Z_STREAM_FLAG_READ,
                            z_plug_copy_server_to_down,
                            self,
                            NULL);
      z_stream_set_callback(self->endpoints[EP_SERVER],
                            Z_STREAM_FLAG_WRITE,
                            z_plug_copy_down_to_server,
                            self,
                            NULL);

      z_stream_set_callback(self->stacked->downstreams[EP_CLIENT],
                            Z_STREAM_FLAG_READ,
                            z_plug_copy_down_to_client,
                            self, 
                            NULL);
      z_stream_set_callback(self->stacked->downstreams[EP_CLIENT],
                            Z_STREAM_FLAG_WRITE,
                            z_plug_copy_client_to_down,
                            self, 
                            NULL);
      z_stream_set_cond(self->stacked->downstreams[EP_CLIENT],
                       Z_STREAM_FLAG_READ,
                       TRUE);

      z_stream_set_callback(self->stacked->downstreams[EP_SERVER],
                            Z_STREAM_FLAG_READ,
                            z_plug_copy_down_to_server,
                            self,
                            NULL);
      z_stream_set_callback(self->stacked->downstreams[EP_SERVER],
                            Z_STREAM_FLAG_WRITE,
                            z_plug_copy_server_to_down,
                            self,
                            NULL);
      z_stream_set_cond(self->stacked->downstreams[EP_SERVER],
                       Z_STREAM_FLAG_READ,
                       TRUE);

      z_poll_add_stream(self->poll, self->stacked->downstreams[EP_CLIENT]);
      z_poll_add_stream(self->poll, self->stacked->downstreams[EP_SERVER]);
    }
  z_proxy_leave(self->owner);
  return TRUE;
}

static gboolean
z_plug_session_stats_timeout(gpointer user_data)
{
  ZPlugSession *self = (ZPlugSession *) user_data;
  
  if (self->session_data->packet_stats)
    {
      if (!self->session_data->packet_stats(self->owner, self->vars,
              self->buffers[EP_CLIENT].packet_bytes,
              self->buffers[EP_CLIENT].packet_count,
              self->buffers[EP_SERVER].packet_bytes,
              self->buffers[EP_SERVER].packet_count))
        {
          z_plug_update_eof_mask(self, EOF_ALL);
        }
    }
  else
    {
      /*LOG
	This message indicates that packet stats interval was specified, but no action was configured to handle the event.
	Check your policy for packetStats event.
       */
      z_proxy_log(self->owner, CORE_ERROR, 3, "Packet stats timeout elapsed, and no timeout callback specified;");
      return FALSE;
    }
  return TRUE;
}

gboolean
z_plug_session_start(ZPlugSession *self, ZPoll *poll)
{
  z_poll_ref(poll);
  self->poll = poll;
  
  if (z_plug_session_init_streams(self) && z_plug_session_init_stacked_streams(self))
    {
      g_get_current_time(&self->started_time);
      if (self->session_data->packet_stats_interval_time > 0)
        {
          GMainContext *context;
         
          self->stats_timeout = g_timeout_source_new(self->session_data->packet_stats_interval_time);
          g_source_set_callback(self->stats_timeout, z_plug_session_stats_timeout, self, NULL);
          context = z_poll_get_context(self->poll);
          g_source_attach(self->stats_timeout, context);
        }
      if (self->session_data->timeout > 0)
        {
          GMainContext *context;
         
          self->timeout = z_timeout_source_new(self->session_data->timeout);
          g_source_set_callback(self->timeout, z_plug_timeout, self, NULL);
          context = z_poll_get_context(self->poll);
          g_source_attach(self->timeout, context);
        }
      self->session_data->num_sessions++;
      return TRUE;
    }
  return FALSE;
}

static ZPolicyObj *
z_plug_session_query_bandwidth(ZProxy *proxy G_GNUC_UNUSED, gchar *name, gpointer value)
{
  ZPlugSession *self = (ZPlugSession *) value;
  GTimeVal now, spent;
  double bandwidth = 0.0;
  
  g_get_current_time(&now);
  spent.tv_sec = now.tv_sec - self->started_time.tv_sec;
  spent.tv_usec = now.tv_usec - self->started_time.tv_usec;
  if (spent.tv_usec < -500000)
    spent.tv_sec++;
    
  if (strcmp(name, "bandwidth_to_client") == 0)
    {
      bandwidth = (double) self->buffers[EP_CLIENT].packet_bytes / spent.tv_sec;
    }
  else if (strcmp(name, "bandwidth_to_server") == 0)
    {
      bandwidth = (double) self->buffers[EP_SERVER].packet_bytes / spent.tv_sec;
    }
  return z_policy_var_build("d", bandwidth);
}

static void
z_plug_do_nothing(gpointer user_data G_GNUC_UNUSED)
{
}

void
z_plug_session_register_vars(ZPlugSession *self)
{
  self->vars = z_session_vars_new();
  
  z_session_var_new(self->vars, "bandwidth_to_client", 
                    Z_VAR_TYPE_CUSTOM | Z_VAR_GET,
                    self, z_plug_session_query_bandwidth, NULL, z_plug_do_nothing);
  z_session_var_new(self->vars, "bandwidth_to_server",
                    Z_VAR_TYPE_CUSTOM | Z_VAR_GET,
                    self, z_plug_session_query_bandwidth, NULL, z_plug_do_nothing);

}

ZPlugSession *
z_plug_session_new(ZProxy *owner, ZPlugSessionData *session_data, ZStream *client_stream, ZStream *server_stream, ZStackedProxy *stacked)
{
  ZPlugSession *self = g_new0(ZPlugSession, 1);
  
  self->owner = owner;
  z_stream_ref(client_stream);
  z_stream_ref(server_stream);
  
  if (!client_stream->name[0])
    g_snprintf(client_stream->name, sizeof(client_stream->name), "%s/%s", owner->session_id, "client");
  if (!server_stream->name[0])
    g_snprintf(server_stream->name, sizeof(server_stream->name), "%s/%s", owner->session_id, "server");

  self->endpoints[EP_CLIENT] = client_stream;
  self->endpoints[EP_SERVER] = server_stream;
  self->stacked = stacked;
  self->session_data = session_data;
  z_plug_session_register_vars(self);
  return self;
}

void
z_plug_session_free(ZPlugSession *self)
{
  gint i;
  
  for (i = EP_CLIENT; i < EP_MAX; i++)
    {
      if (self->stacked)
        {
          g_free(self->downbufs[i].buf);
          z_poll_remove_stream(self->poll, self->stacked->downstreams[i]);
        }
      g_free(self->buffers[i].buf);
      z_poll_remove_stream(self->poll, self->endpoints[i]);
      z_stream_shutdown(self->endpoints[i], SHUT_RDWR, NULL);
      z_stream_close(self->endpoints[i], NULL);
      z_stream_unref(self->endpoints[i]);
    }
  
  if (self->stacked)
    z_stacked_proxy_destroy(self->stacked);
  z_session_vars_destroy(self->vars);
  if (self->stats_timeout)
    {
      g_source_destroy(self->stats_timeout);
      g_source_unref(self->stats_timeout);
      self->stats_timeout = NULL;
    }
  if (self->timeout)
    {
      g_source_destroy(self->timeout);
      g_source_unref(self->timeout);
      self->timeout = NULL;
    }
  z_poll_unref(self->poll);
  g_free(self);
}

void
z_plug_sessions_purge(ZPlugSessionData *session_data)
{
  while (session_data->destroy_queue)
    {
      ZPlugSession *session;
          
      session = (ZPlugSession *) session_data->destroy_queue->data;
      session_data->num_sessions--;
      z_plug_session_free(session);
      session_data->destroy_queue = g_list_delete_link(session_data->destroy_queue, session_data->destroy_queue);
    }
}

