/***************************************************************************
 *
 * Copyright (c) 2000, 2001, 2002, 2003, 2004 BalaBit IT Ltd, Budapest, Hungary
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * Note that this permission is granted for only version 2 of the GPL.
 *
 * As an additional exemption you are allowed to compile & link against the
 * OpenSSL libraries as published by the OpenSSL project. See the file
 * COPYING for details.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: szig.c,v 1.32 2004/07/02 10:03:33 bazsi Exp $
 *
 * Author  : SaSa
 * Auditor :
 * Last audited version:
 * Notes:
 *
 ***************************************************************************/

#include <zorp/szig.h>
#include <zorp/log.h>
#include <zorp/io.h>
#include <zorp/thread.h>
#include <zorp/stream.h>
#include <zorp/streamfd.h>
#include <zorp/streambuf.h>
#include <zorp/streamline.h>
#include <zorp/sockaddr.h>
#include <zorp/listen.h>

#include <string.h>

/*
 * The SZIG framework serves as a means to publish internal statistics. It
 * consists of three main parts:
 *   - collecting data using events
 *   - aggregate that data and represent 'results' in an N-ary tree
 *   - publish this data through a UNIX domain socket interface
 *
 * Data collection:
 * 
 *   Primitive data is collected using events. An event is generated (using
 *   a simple function call) when something happens which is interesting
 *   from a statistics standpoint. For example an event is generated when a
 *   thread is started, but a similar event can be generated when a new
 *   entry becomes available in the licensed IPs hash.
 *
 * Data aggregation:
 *
 *   Aggregator functions are generic functions to be applied to events. When
 *   an event occurs all registered functions are called in the order of
 *   registration. This way several different representations of the same
 *   incoming event data are possible. For example we can register two
 *   functions to the START_THREAD event, one of them counts the current
 *   number of threads, the other counts the total number of threads started
 *   so far.
 *
 *   Each aggregator function receives a result node as argument which
 *   specifies a location where results can be stored. This result node also
 *   contains a GStaticMutex which makes it easy to synchronize acceseses to
 *   the result data.
 *
 * Data publishing:
 *
 *   A simple UNIX domain socket based interface is provided to publish
 *   the result tree.
 */


#define Z_SZIG_MAX_LINE 4096
#define Z_SZIG_SOCKET_NAME "/var/run/zorp/zorpctl"

/**
 * ZSzigEventCallback:
 *
 * This structure describes a callback associated with a given event. It
 * stores a pointer to the function and an opaque pointer to be passed to
 * this function once it is invoked.
 **/
typedef struct _ZSzigEventCallback
{
  ZSzigNode *node;
  ZSzigEventHandler func;
  gpointer user_data;
} ZSzigEventCallback;

/**
 * ZSzigEventDesc:
 *
 * This function describes an event. Currently it contains a list of
 * callbacks to be called when the event occurs.
 **/
typedef struct _ZSzigEventDesc
{
  GList *callbacks;
} ZSzigEventDesc;

/**
 * ZSzigQueueItem:
 *
 * The asynchron queue between SZIG and the rest of Zorp contains elements
 * of this type.
 **/
typedef struct _ZSzigQueueItem
{
  ZSzigEvent event;
  ZSzigValue *param;
} ZSzigQueueItem;

/**
 * ZSzigNode:
 *
 * A node in the result tree.
 **/
struct _ZSzigNode
{
  gchar *name;
  
  ZSzigValue value;
  gpointer agr_data;
  GDestroyNotify agr_notify;
};

/**
 * ZSzigConnection:
 *
 * Data associated with a connection accepted through the zorpctl socket.
 **/
typedef struct _ZSzigConnection
{
  int ref_cnt;
  ZStream *stream;
} ZSzigConnection;


/* event descriptors */
static ZSzigEventDesc event_desc[Z_SZIG_MAX + 1];

/* result tree root */
static GNode *result_tree_root;
static GStaticRWLock result_tree_lock = G_STATIC_RW_LOCK_INIT;
static GAsyncQueue *szig_queue = NULL;

/* SZIG values */

void
z_szig_value_repr(ZSzigValue *v, gchar *buf, gsize buflen)
{
  switch (v->type)
    {
    case Z_SZIG_TYPE_NOTINIT:
      g_strlcpy(buf, "None", buflen);
      break;
    case Z_SZIG_TYPE_LONG:
      g_snprintf(buf, buflen, "%ld", v->u.long_value);
      break;
    case Z_SZIG_TYPE_TIME:
      g_snprintf(buf, buflen, "%ld:%ld", (glong) v->u.time_value.tv_sec, (glong) v->u.time_value.tv_usec);
      break;
    case Z_SZIG_TYPE_STRING:
      if (v->u.string_value)
        g_strlcpy(buf, v->u.string_value->str, buflen);
      else
        g_strlcpy(buf, "", buflen);
      break;
    default:
      g_assert(0);
      break;
    }
}

static inline glong
z_szig_value_as_long(ZSzigValue *v)
{
  g_assert(v->type == Z_SZIG_TYPE_LONG);
  return v->u.long_value;
}

static inline GTimeVal *
z_szig_value_as_time(ZSzigValue *v)
{
  g_assert(v->type == Z_SZIG_TYPE_TIME);
  return &v->u.time_value;
}

static inline const gchar *
z_szig_value_as_string(ZSzigValue *v)
{
  g_assert(v->type == Z_SZIG_TYPE_STRING);
  return v->u.string_value->str;
}

static inline GString *
z_szig_value_as_gstring(ZSzigValue *v)
{
  g_assert(v->type == Z_SZIG_TYPE_STRING);
  return v->u.string_value;
}

ZSzigValue *
z_szig_value_new_long(glong val)
{
  ZSzigValue *v = g_new0(ZSzigValue, 1);
  v->type = Z_SZIG_TYPE_LONG;
  v->u.long_value = val;
  return v;
}

ZSzigValue *
z_szig_value_new_time(GTimeVal *val)
{
  ZSzigValue *v = g_new0(ZSzigValue, 1);
  v->type = Z_SZIG_TYPE_TIME;
  v->u.time_value = *val;
  return v;
}

ZSzigValue *
z_szig_value_new_string(const gchar *val)
{
  ZSzigValue *v = g_new0(ZSzigValue, 1);
  v->type = Z_SZIG_TYPE_STRING;
  v->u.string_value = g_string_new(val);
  return v;
}

void
z_szig_value_free(ZSzigValue *v)
{
  g_free(v);
}

/**
 * z_szig_node_new:
 * @name: name of the SZIG node to create
 *
 * This function creates a new z_szig_node data structure with data
 * initialized to zero and name initialized to the @name argument.
 *
 * Returns: the newly allocated ZSzigNode structure
 **/
static ZSzigNode *
z_szig_node_new(const gchar *name)
{
  ZSzigNode *n = g_new0(ZSzigNode, 1);
  n->name = g_strdup(name);
  return n;
}

static inline void
z_szig_node_set_data(ZSzigNode *node, gpointer agr_data, GDestroyNotify notify)
{
  node->agr_data = agr_data;
  node->agr_notify = notify;
}

static inline gpointer
z_szig_node_get_data(ZSzigNode *node)
{
  return node->agr_data;
}

#if 0
/* this block is commented out as otherwise it causes warnings */

/**
 * z_szig_node_free:
 * @n: SZIG node to free
 *
 * This function frees a SZIG node from the result tree. It is currently not
 * used as nodes are never removed from the tree.
 **/
static void
z_szig_node_free(ZSzigNode *n)
{
  if (n->agr_notify)
    n->agr_notify(n->agr_data);
  g_free(n->name);
  g_free(n);
}
#endif

/**
 * z_szig_node_lookup:
 * @node_name: the path to the variable to look up
 * @create: specifies whether an empty node should be created if the name is not found
 * @node: the encapsulating GNode
 * 
 * This function looks up or creates a node in the result tree. Names are
 * dot separated paths which specify a location in our N-ary tree.
 *
 * Returns: the SZIG node if found and NULL otherwise
 **/
static ZSzigNode *
z_szig_node_lookup(const gchar *node_name, gboolean create, GNode **gnode)
{
  gchar **split;
  GNode *root, *node = NULL;
  gint i;
  
  z_enter();  
  split = g_strsplit(node_name, ".", 0);
  if (!split)
    {
      z_leave();
      return NULL;
    }
  if (strcmp(split[0], "zorp") != 0)
    {
      g_strfreev(split);
      z_leave();
      return NULL;
    }
  
  g_static_rw_lock_reader_lock(&result_tree_lock);
  node = root = result_tree_root;
  for (i = 1; split[i]; i++)
    {    
      node = g_node_first_child(root);
      while (node)
        {
          if (strcmp(((ZSzigNode *) node->data)->name, split[i]) == 0)
              break;
              
          node = g_node_next_sibling(node);
        }
      if (!node)
        {
          if (create)
            {
              node = g_node_new(z_szig_node_new(split[i]));
              
              /* NOTE: this is a not atomic update of the rwlock from reader to
               * writer lock, it works here as nodes are never removed from
               * the tree */
              
              g_static_rw_lock_reader_unlock(&result_tree_lock);
              g_static_rw_lock_writer_lock(&result_tree_lock);
              g_node_prepend(root, node);
              g_static_rw_lock_writer_unlock(&result_tree_lock);
              g_static_rw_lock_reader_lock(&result_tree_lock);
            }
          else
            {
              break;
            }
        }
        
      root = node;
    }
  g_static_rw_lock_reader_unlock(&result_tree_lock);
  g_strfreev(split);
  if (gnode)
    *gnode = node;
  return node ? (ZSzigNode *) node->data : NULL;
}


/**
 * z_szig_agr_count_inc:
 * @node:
 * @ev:
 * @p:
 * @user_data:
 * 
 * This aggregator function increments the result value in a thread
 * synchronized manner.
 **/ 
void
z_szig_agr_count_inc(ZSzigNode *node, ZSzigEvent ev G_GNUC_UNUSED, ZSzigValue *p G_GNUC_UNUSED, gpointer user_data G_GNUC_UNUSED)
{
  node->value.type = Z_SZIG_TYPE_LONG;
  node->value.u.long_value++;
}

/**
 * z_szig_agr_count_dec:
 * @node: result node
 * @ev: event, not used
 * @p: event parameter, not used
 * @user_data: not used
 * 
 * This aggregator function decrements the result value in a thread
 * synchronized manner.
 **/ 
void
z_szig_agr_count_dec(ZSzigNode *node, ZSzigEvent ev G_GNUC_UNUSED, ZSzigValue *p G_GNUC_UNUSED, gpointer user_data G_GNUC_UNUSED)
{
  node->value.type = Z_SZIG_TYPE_LONG;
  node->value.u.long_value--;
}

typedef struct _ZSzigAvgState
{
  GTimeVal last_time;
  glong last_value;
  ZSzigNode *source_node;
} ZSzigAvgState;

void
z_szig_agr_average(ZSzigNode *target_node, ZSzigEvent ev G_GNUC_UNUSED, ZSzigValue *p, gpointer user_data)
{
  const gchar *source_node_name = (const gchar *) user_data;
  ZSzigAvgState *avg_state;
  GTimeVal *current_time;
  glong time_diff, current_value;
  
  target_node->value.type = Z_SZIG_TYPE_LONG;
    
  avg_state = (ZSzigAvgState *) z_szig_node_get_data(target_node);
  if (!avg_state)
    {
      avg_state = g_new0(ZSzigAvgState, 1);
      z_szig_node_set_data(target_node, avg_state, g_free);
    }
  if (!avg_state->source_node)
    {
      avg_state->source_node = z_szig_node_lookup(source_node_name, FALSE, NULL);
    }
  if (!avg_state->source_node)
    {
      /*LOG
	This message indicates an internal error, please contact your Zorp support for assistance.
       */
      z_log(NULL, CORE_ERROR, 3, "Invalid average aggregator, no source node; source_node='%s'", source_node_name);
      goto exit_nostore;
    }
  current_time = z_szig_value_as_time(p);
  current_value = z_szig_value_as_long(&avg_state->source_node->value);
  if (avg_state->last_time.tv_sec == 0 && avg_state->last_time.tv_usec == 0)
    {
      goto exit_store_current;
    }
  
  target_node->value.type = Z_SZIG_TYPE_LONG;
  time_diff = ((current_time->tv_sec - avg_state->last_time.tv_sec) * 1000000 + 
              (current_time->tv_usec - avg_state->last_time.tv_usec)) / 1000000;
  if (time_diff == 0)
    time_diff = 1;
  target_node->value.u.long_value = (current_value - avg_state->last_value) / time_diff;
  
 exit_store_current:
  avg_state->last_time = *current_time;
  avg_state->last_value = current_value;
  
 exit_nostore:
  return;
}

void
z_szig_agr_append_string(ZSzigNode *target_node, ZSzigEvent ev G_GNUC_UNUSED, ZSzigValue *p, gpointer user_data G_GNUC_UNUSED)
{
  if (target_node->value.type == Z_SZIG_TYPE_NOTINIT)
    {
      target_node->value.type = Z_SZIG_TYPE_STRING;
      target_node->value.u.string_value = g_string_new(z_szig_value_as_string(p));
    }
  else
    {
      g_string_sprintfa(z_szig_value_as_gstring(&target_node->value), ":%s", z_szig_value_as_string(p));
    }
}

/* general framework */

/**
 * z_szig_event_get_desc:
 * @ev: event type
 *
 * This function returns the event descriptor for the event @ev.
 **/
static inline ZSzigEventDesc *
z_szig_event_get_desc(ZSzigEvent ev)
{
  g_assert(ev >= 0 && ev <= Z_SZIG_MAX);
  return &event_desc[ev];
}

/**
 * z_szig_process_event:
 * @ev:
 * @param:
 *
 **/
void
z_szig_event(ZSzigEvent ev, ZSzigValue *param)
{
  ZSzigQueueItem *q = g_new0(ZSzigQueueItem, 1);
  q->event = ev;
  q->param = param;
  if (szig_queue)
    g_async_queue_push(szig_queue, q);
}

/**
 * z_szig_process_event:
 * @ev:
 * @param:
 *
 **/
void
z_szig_process_event(ZSzigEvent ev, ZSzigValue *param)
{
  ZSzigEventDesc *d;
  ZSzigEventCallback *cb;
  GList *p;
  
  d = z_szig_event_get_desc(ev);
  
  for (p = d->callbacks; p; p = g_list_next(p))
    {
      cb = (ZSzigEventCallback *) p->data;
      
      cb->func(cb->node, ev, param, cb->user_data);
    }
    
  z_szig_value_free(param);
}

/**
 * z_szig_register_handler:
 * @ev:
 * @func:
 * @node_name:
 * @user_data:
 *
 * This function registers an aggregator for the specified event using the
 * specified target node.
 **/
void
z_szig_register_handler(ZSzigEvent ev, ZSzigEventHandler func, const gchar *node_name, gpointer user_data)
{
  ZSzigEventCallback *cb;
  ZSzigEventDesc *d;
  
  d = z_szig_event_get_desc(ev);
  
  cb = g_new0(ZSzigEventCallback, 1);
  cb->node = z_szig_node_lookup(node_name, TRUE, NULL);
  cb->user_data = user_data;
  cb->func = func;
  d->callbacks = g_list_append(d->callbacks, cb);
}

/* basic SZIG events */

/**
 * z_szig_thread_started:
 * @self:
 * @user_data:
 *
 * This function is registered as a thread startup function. It simply generates a
 * Z_SZIG_THREAD_START event.
 **/
static void 
z_szig_thread_started(ZThread *self G_GNUC_UNUSED, gpointer user_data G_GNUC_UNUSED)
{
  z_szig_event(Z_SZIG_THREAD_START, NULL);
}

/**
 * z_szig_thread_stopped:
 * @self:
 * @user_data:
 *
 * This function is registered as a thread stop function. It simply generates a
 * Z_SZIG_THREAD_STOP event.
 **/
static void 
z_szig_thread_stopped(ZThread *self G_GNUC_UNUSED, gpointer user_data G_GNUC_UNUSED)
{
  z_szig_event(Z_SZIG_THREAD_STOP, NULL);
}

/**
 *
 **/
static gboolean
z_szig_tick_callback(GSource *source)
{
  GTimeVal current_time;
  static gint ticks = 0;
  
  g_source_get_current_time(source, &current_time);
  z_szig_event(Z_SZIG_MIN1_TICK, z_szig_value_new_time(&current_time));
  if ((ticks % 5) == 0)
      z_szig_event(Z_SZIG_MIN5_TICK, z_szig_value_new_time(&current_time));
  if ((ticks % 15) == 0)
      z_szig_event(Z_SZIG_MIN15_TICK, z_szig_value_new_time(&current_time));    

  ticks++;
  return TRUE;
}

static gpointer
z_szig_thread(gpointer st G_GNUC_UNUSED)
{
  if (!szig_queue)
    return NULL;
  while (1)
    {
      ZSzigQueueItem *q = (ZSzigQueueItem *) g_async_queue_pop(szig_queue);
      
      z_szig_process_event(q->event, q->param);
      
      g_free(q);
    }
  return NULL;
}

/* SZIG I/O */

#if 0
/* this is not currently used, it is implemented here for the sake of correctness */
static void
z_szig_connection_ref(ZSzigConnection *self)
{
  g_assert(self->ref_cnt > 0);
  self->ref_cnt++;
}
#endif

static void
z_szig_connection_unref(ZSzigConnection *self)
{
  g_assert(self->ref_cnt > 0);
  if (--self->ref_cnt == 0)
    {
      /* detatch from main context & close */
      z_stream_detach_source(self->stream);
      z_stream_close(self->stream, NULL);
      z_stream_unref(self->stream);
      g_free(self);
    }
}

static gboolean
z_szig_handle_command(ZSzigConnection *conn, gchar *cmd_line)
{
  gchar response[16384], *cmd, *name;
  GNode *gnode;
  ZSzigNode *node;
  gchar **argv;
  const gchar *logspec;
  gint direction, value;
  
  z_enter();
  argv = g_strsplit(cmd_line, " ", 0);
  
  if (!argv || !argv[0])
    {
      if (argv)
        g_strfreev(argv);
      z_leave();
      return FALSE;
    }
  
  cmd = argv[0];
  
  g_strlcpy(response, "None\n", sizeof(response));
  if (strcmp(cmd, "GETVALUE") == 0 ||
      strcmp(cmd, "GETCHILD") == 0 ||
      strcmp(cmd, "GETSBLNG") == 0)
    {
      name = argv[1];
      node = z_szig_node_lookup(name, FALSE, &gnode);
      
      if (strcmp(cmd, "GETVALUE") == 0)
        {
          if (node)
            {
              z_szig_value_repr(&node->value, response, sizeof(response)-1);
              strncat(response, "\n", sizeof(response));
            }
        }
      else if (strcmp(cmd, "GETCHILD") == 0)
        {
          if (node)
            {
              gnode = g_node_first_child(gnode);
              if (gnode)
                {
                  node = (ZSzigNode *) gnode->data;
                  g_snprintf(response, sizeof(response), "%s.%s\n", name, node->name);
                }
            }
        }
      else if (strcmp(cmd, "GETSBLNG") == 0)
        {
          if (node)
            {
              gnode = g_node_next_sibling(gnode);
              if (gnode)
                {
                  gchar *e = name + strlen(name) - 1;
                  while (e > name && *e != '.')
                    e--;
                  *e = 0;
                  node = (ZSzigNode *) gnode->data;
                  g_snprintf(response, sizeof(response), "%s.%s\n", name, node->name);
                }
            }
        }
    }
  else if (strcmp(cmd, "LOGGING") == 0)
    {
      g_strlcpy(response, "FAIL\n", sizeof(response));
      if (!argv[1])
        g_strlcpy(response, "FAIL LOGGING subcommand required", sizeof(response));
      else if (strcmp(argv[1], "VINC") == 0 ||
               strcmp(argv[1], "VDEC") == 0 ||
               strcmp(argv[1], "VSET") == 0)
        {
          
          if (argv[1][1] == 'I')
            {
              direction = 1;
            }
          else if (argv[1][1] == 'D')
            {
              direction = -1;
            }
          else
            {
              direction = 0;
            }
          if (argv[2])
            value = strtol(argv[2], NULL, 10);
          else
            value = 0;
            
          if (z_log_change_verbose_level(direction, value, &value))
            g_snprintf(response, sizeof(response), "OK %d\n", value);
          else
            g_snprintf(response, sizeof(response), "FAIL Error changing verbose level\n");
        }
      else if (strcmp(argv[1], "VGET") == 0)
        {
          if (z_log_change_verbose_level(1, 0, &value))
            g_snprintf(response, sizeof(response), "OK %d\n", value);
          else
            g_snprintf(response, sizeof(response), "FAIL Error querying verbose level\n");
            
        }
      else if (strcmp(argv[1], "GETSPEC") == 0)
        {
          if (z_log_change_logspec(NULL, &logspec))
            g_snprintf(response, sizeof(response), "OK %s\n", logspec ? logspec : "");
          else
            g_snprintf(response, sizeof(response), "FAIL Error querying logspec\n");
        }
      else if (strcmp(argv[1], "SETSPEC") == 0)
        {
          if (argv[2])
            {
              if (z_log_change_logspec(argv[2], &logspec))
                g_snprintf(response, sizeof(response), "OK %s\n", logspec);
              else
                g_snprintf(response, sizeof(response), "FAIL Error setting logspec\n");
            }
          else
            g_snprintf(response, sizeof(response), "FAIL No logspec specified\n");
        }
    }
  else
    {
      g_strlcpy(response, "FAIL No such command", sizeof(response));
    }
    
  g_strfreev(argv);
  
  if (z_stream_write_buf(conn->stream, response, strlen(response), TRUE, FALSE) != G_IO_STATUS_NORMAL)
    {
      z_leave();
      return FALSE;
    }
  z_leave();
  return TRUE;
}

static gboolean
z_szig_read_callback(ZStream *stream, GIOCondition cond G_GNUC_UNUSED, gpointer user_data)
{
  ZSzigConnection *conn = (ZSzigConnection *) user_data;
  gchar buf[Z_SZIG_MAX_LINE];
  gsize buflen = sizeof(buf) - 1;
  GIOStatus res;
  
  z_enter();
  res = z_stream_line_get_copy(stream, buf, &buflen, NULL);
  if (res == G_IO_STATUS_NORMAL)
    {
      buf[buflen] = 0;
      g_static_rw_lock_reader_lock(&result_tree_lock);
      if (z_szig_handle_command(conn, buf))
        {
          z_leave();
          return TRUE;
        }
      g_static_rw_lock_reader_unlock(&result_tree_lock);
    }
  else if (res == G_IO_STATUS_AGAIN)
    {
      z_leave();
      return TRUE;
    }
  z_szig_connection_unref(conn);
  z_leave();
  return TRUE;
}

/**
 * z_szig_accept_callback:
 * @fd: fd that refers to the connection
 * @client: socket address
 * @last_connection: this is the last connection (when accept_one was specified)
 * @user_data: opaque pointer
 *
 * This function is called as soon as a new connection is received.
 **/
static gboolean
z_szig_accept_callback(int fd,
                       ZSockAddr *client,
                       gboolean  last_connection G_GNUC_UNUSED,
                       gpointer  user_data G_GNUC_UNUSED)
{
  ZSzigConnection *conn;
  ZStream *fdstream, *linestream, *bufstream;

  z_fd_set_nonblock(fd, TRUE);

  fdstream = z_stream_fd_new(fd, "szig");
  linestream = z_stream_line_new(fdstream, Z_SZIG_MAX_LINE, ZRL_EOL_NL);
  bufstream = z_stream_buf_new(linestream, 1024, 2048, NULL, NULL);
  
  z_stream_unref(fdstream);
  z_stream_unref(linestream);
  
  conn = g_new0(ZSzigConnection, 1);
  conn->ref_cnt = 1;
  conn->stream = bufstream;
  
  z_stream_set_callback(conn->stream, Z_STREAM_FLAG_READ, z_szig_read_callback, conn, (GDestroyNotify) z_szig_connection_unref);
  z_stream_set_cond(conn->stream, Z_STREAM_FLAG_READ, TRUE);
  
  z_stream_attach_source(conn->stream, g_main_context_default());
  z_sockaddr_unref(client);
  return TRUE;
}

/**
 * z_szig_init:
 * @instance_name: the name of this Zorp instance
 *
 * This function initializes the SZIG subsystem, creates the UNIX domain
 * socket and initializes basic aggregations.
 **/
void
z_szig_init(const gchar *instance_name)
{
  ZSockAddr *sockaddr;
  ZIOListen *listen;
  gchar buf[256];
  GSource *tick_source;
  
  result_tree_root = g_node_new(z_szig_node_new("zorp"));
  memset(event_desc, 0, sizeof(event_desc));
  szig_queue = g_async_queue_new();
  
  if (z_thread_new("szig", z_szig_thread, NULL))
    {
      z_szig_register_handler(Z_SZIG_THREAD_START, z_szig_agr_count_inc, "zorp.stats.threads_running", NULL);
      z_szig_register_handler(Z_SZIG_THREAD_STOP,  z_szig_agr_count_dec, "zorp.stats.threads_running", NULL);
      z_szig_register_handler(Z_SZIG_THREAD_START, z_szig_agr_count_inc, "zorp.stats.thread_number", NULL);

      z_szig_register_handler(Z_SZIG_MIN1_TICK, z_szig_agr_average, "zorp.stats.thread_avg1", "zorp.stats.thread_number");
      z_szig_register_handler(Z_SZIG_MIN5_TICK, z_szig_agr_average, "zorp.stats.thread_avg5", "zorp.stats.thread_number");
      z_szig_register_handler(Z_SZIG_MIN15_TICK, z_szig_agr_average, "zorp.stats.thread_avg15", "zorp.stats.thread_number");
      z_szig_register_handler(Z_SZIG_COUNTED_IP, z_szig_agr_append_string, "zorp.license.ipaddrs", NULL);

      z_thread_register_start_callback((GFunc) z_szig_thread_started, NULL);
      z_thread_register_stop_callback((GFunc) z_szig_thread_stopped, NULL);
      tick_source = g_timeout_source_new(60000);
      g_source_set_callback(tick_source, (GSourceFunc) z_szig_tick_callback, tick_source, NULL);
      g_source_attach(tick_source, g_main_context_default());

      g_snprintf(buf, sizeof(buf), "%s.%s", Z_SZIG_SOCKET_NAME, instance_name);
  
      sockaddr = z_sockaddr_unix_new(buf);
  
      listen = z_io_listen_new("szig", sockaddr, FALSE, 255, z_szig_accept_callback, NULL);
      if (listen)
        {
          z_io_listen_start(listen);
          z_io_listen_unref(listen);
        }
    }
  else
    {
      g_async_queue_unref(szig_queue);
      szig_queue = NULL;
    }
}

void
z_szig_destroy(void)
{
  /* FIXME: free result tree */
}

