/***************************************************************************
 *
 * Copyright (c) 2000, 2001, 2002, 2003, 2004 BalaBit IT Ltd, Budapest, Hungary
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * Note that this permission is granted for only version 2 of the GPL.
 *
 * As an additional exemption you are allowed to compile & link against the
 * OpenSSL libraries as published by the OpenSSL project. See the file
 * COPYING for details.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: proxy.h,v 1.82 2004/06/11 12:57:39 bazsi Exp $
 *
 ***************************************************************************/

#ifndef ZORP_PROXY_H_INCLUDED
#define ZORP_PROXY_H_INCLUDED

#include <zorp/zorp.h>
#include <zorp/zobject.h>
#include <zorp/stream.h>
#include <zorp/zpython.h>
#include <zorp/conntrack.h>
#include <zorp/proxyvars.h>
#include <zorp/dispatch.h>
#include <zorp/fastpath.h>

/* a two-way connection between streams */

/* endpoint indexes */
#define EP_CLIENT      0
#define EP_SERVER      1
#define EP_MAX         2


/* proxy states */
#define ZPS_INITIAL        0
#define ZPS_CONFIG         1
#define ZPS_STARTING_UP    2
#define ZPS_WORKING        3
#define ZPS_SHUTTING_DOWN  4
#define ZPS_DESTROYING     5

/* secondary mask */
#define ZS_MATCH_SADDR 0x0001
#define ZS_MATCH_SPORT 0x0002
#define ZS_MATCH_DADDR 0x0004
#define ZS_MATCH_DPORT 0x0008
#define ZS_MATCH_ALL   0x000F

typedef struct _ZProxyParams ZProxyParams;
typedef struct _ZProxy ZProxy;
typedef struct _ZProxyIface ZProxyIface;
typedef struct _ZProxyFuncs ZProxyFuncs;
typedef struct _ZStackedProxy ZStackedProxy;

/* structure describing a stacked proxy instance */
struct _ZStackedProxy
{
  ZStream *downstreams[EP_MAX];
  ZStream *control_stream;
  ZProxy *proxy;
  ZProxy *child_proxy;
};

ZStackedProxy *z_stacked_proxy_new(ZStream *client_stream, ZStream *server_stream, ZStream *control_stream, ZProxy *proxy, ZProxy *child_proxy);
void z_stacked_proxy_destroy(ZStackedProxy *self);

struct _ZProxyParams
{
  const gchar *session_id;
  ZPolicyObj *pyclient;
  ZStream *client;
  ZPolicyObj *handler;
  ZProxy *parent;
};

struct _ZProxyFuncs
{
  ZObjectFuncs super;
  gboolean (*config)(ZProxy *self);
  gboolean (*startup)(ZProxy *self);
  void (*main)(ZProxy *self);
  void (*shutdown)(ZProxy *self);
  void (*destroy)(ZProxy *self);
  void (*run)(ZProxy *self);
};

struct _ZProxy
{
  ZObject super;
  gchar session_id[MAX_SESSION_ID];
  guint32 status;
  ZPolicyObj *handler;
  ZPolicyThread *thread;
  struct _ZProxyVars *vars;
  
  /* the linked list of child proxies */
  GList *child_proxies;

  /* a pointer to the parent proxy */
  struct _ZProxy *parent_proxy;
  
  GList *interfaces;
  
  ZStream *endpoints[EP_MAX];

  ZProxyFastpath fastpath;
};

extern ZClass ZProxy__class;

/* function prototypes registered in the registry */
typedef ZProxy *(*ZProxyCreateFunc)(ZProxyParams *params);


/* not inline because of the variable number of arguments */

#define z_proxy_var_new(p, n, f, a...) z_proxy_vars_var_new((p)->vars->vars, n, f, ##a)
#define z_session_var_new(s, n, f, a...) z_proxy_vars_var_new((s)->vars, n, f, ##a)

/* log functions */
#define z_proxy_log(self, class, level, format, args...) \
                   /*NOLOG*/ \
                   z_log(((ZProxy *)self)->session_id, class, level, format,  ##args)

#if ENABLE_TRACE
  #define z_proxy_trace(self, args...) z_proxy_log(self , CORE_TRACE, 7, ##args)
  #define z_proxy_enter(self) z_proxy_log(self, CORE_TRACE, 7, "Enter %s (%s:%d)", __FUNCTION__, __FILE__, __LINE__)
  #define z_proxy_leave(self) z_proxy_log(self, CORE_TRACE, 7, "Leave %s (%s:%d)", __FUNCTION__, __FILE__, __LINE__)
  #define z_proxy_cp(self) z_proxy_log(self, CORE_TRACE, 7, "Checkpoint %s (%s:%d)", __FUNCTION__, __FILE__, __LINE__)
#else
  #define z_proxy_trace(self, args...)
  #define z_proxy_enter(self)
  #define z_proxy_leave(self)
  #define z_proxy_cp(self)
#endif

/* interface support */
void z_proxy_add_iface(struct _ZProxy *self, struct _ZProxyIface *iface);
void z_proxy_del_iface(struct _ZProxy *self, struct _ZProxyIface *iface);
struct _ZProxyIface *z_proxy_find_iface(struct _ZProxy *self, ZClass *compat);

/* helper functions for communicating with the policy layer */
gboolean z_proxy_policy_config(ZProxy *);
gboolean z_proxy_policy_startup(ZProxy *);
void z_proxy_policy_shutdown(ZProxy *);
void z_proxy_policy_destroy(ZProxy *self); 

/* misc helper functions */
gint z_proxy_connect_server(ZProxy *self, char *host, gint port);
gint z_proxy_user_authenticated(ZProxy *self, gchar *entity);
ZStackedProxy *z_proxy_stack_proxy(ZProxy *self, ZPolicyObj *proxy_class);
ZStackedProxy *z_proxy_stack_program(ZProxy *self, const gchar *program);
ZStackedProxy *z_proxy_stack_object(ZProxy *self, ZPolicyObj *stack_obj);

gboolean
z_proxy_get_addresses(ZProxy *self, 
                      guint *protocol,
                      ZSockAddr **client_address, ZSockAddr **client_local,
                      ZSockAddr **server_address, ZSockAddr **server_local,
                      ZSockAddr **client_listen);

gboolean
z_proxy_check_secondary(ZProxy *self, guint32 secondary_mask, 
                        ZSockAddr *pri_client, ZSockAddr *pri_server,
                        ZSockAddr *sec_client, ZSockAddr *sec_server);

gboolean z_proxy_start(ZProxy *self);

/* constructor for ZProxy */
ZProxy *
z_proxy_new(ZClass *class, 
            ZProxyParams *params);

/* free method for ZProxy */
void z_proxy_free_method(ZObject *s);

static inline gboolean
z_proxy_config(ZProxy *self)
{ 
  return Z_FUNCS(self, ZProxy)->config(self);
}

static inline gboolean
z_proxy_startup(ZProxy *self)
{ 
  return Z_FUNCS(self, ZProxy)->startup(self);
}

static inline void
z_proxy_main(ZProxy *self)
{ 
  Z_FUNCS(self, ZProxy)->main(self);
}

static inline void
z_proxy_shutdown(ZProxy *self)
{ 
  Z_FUNCS(self, ZProxy)->shutdown(self);
}

static inline void
z_proxy_destroy(ZProxy *self)
{ 
  Z_FUNCS(self, ZProxy)->destroy(self);
}

static inline void
z_proxy_run(ZProxy *self)
{ 
  Z_FUNCS(self, ZProxy)->run(self);
}

static inline ZProxy *
z_proxy_ref(ZProxy *self)
{
  return (ZProxy *) z_object_ref(&self->super);
}

static inline void 
z_proxy_unref(ZProxy *self)
{
  z_object_unref(&self->super);
}

static inline void
z_proxy_set_state(ZProxy *self, guint8 new_state)
{
  self->status = (self->status & ~0xFF) | new_state;
}

static inline guint8
z_proxy_get_state(ZProxy *self)
{
  return self->status & 0xFF;
}

/* Root class for proxy specific interfaces */
struct _ZProxyIface
{
  ZObject super;
  ZProxy *owner;
};

typedef ZObjectFuncs ZProxyIfaceFuncs;
extern ZClass ZProxyIface__class;

ZProxyIface *z_proxy_iface_new(ZClass *class, ZProxy *proxy);

/* ZProxyBasicIface */

typedef ZProxyIface ZProxyBasicIface;
typedef struct _ZProxyBasicIfaceFuncs 
{
  ZObjectFuncs super;
} ZProxyBasicIfaceFuncs;

extern ZClass ZProxyBasicIface__class;

ZProxyBasicIface *z_proxy_basic_iface_new(ZClass *class, ZProxy *proxy);

/* ZProxyResultIface */

typedef ZProxyIface ZProxyResultIface;
typedef struct _ZProxyResultIfaceFuncs 
{
  ZObjectFuncs super;
  void (*set_result)(ZProxyBasicIface *self, guint result, gchar *description);
  void (*set_content_length)(ZProxyBasicIface *self, gsize content_length);
} ZProxyResultIfaceFuncs;

extern ZClass ZProxyResultIface__class;

static inline void
z_proxy_result_set_result(ZProxyBasicIface *self, gint result, gchar *description)
{
  Z_FUNCS(self, ZProxyResultIface)->set_result(self, result, description);
}

static inline void
z_proxy_result_set_content_length(ZProxyBasicIface *self, gsize content_length)
{
  if (Z_FUNCS(self, ZProxyResultIface)->set_content_length)
    Z_FUNCS(self, ZProxyResultIface)->set_content_length(self, content_length);
}


#endif
