/***************************************************************************
 *
 * Copyright (c) 2000, 2001, 2002, 2003, 2004 BalaBit IT Ltd, Budapest, Hungary
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * Note that this permission is granted for only version 2 of the GPL.
 *
 * As an additional exemption you are allowed to compile & link against the
 * OpenSSL libraries as published by the OpenSSL project. See the file
 * COPYING for details.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: base64.c,v 1.2 2004/04/15 11:13:36 bazsi Exp $
 *
 *
 * Author  : SaSa
 * Auditor :
 * Last audited version:
 * Notes:
 *
 ***************************************************************************/

#include <zorp/zorp.h>
#include <zorp/proxy/code.h>
#include <zorp/log.h>

/*
 * Base64 coding
 *
 * Binary to printable encoding, see RFC2045 for details
 */

typedef struct _ZCodeBase64
{
  ZCode super;
  char buf[3];
  gboolean err;
  guint len;
  guint encode_line_pos;
  guint max_line_length;
} ZCodeBase64;

/**
 * z_code_base64_decode_char:
 * @c: character to decode
 *
 * Find the 6-bit value that is represented by @c
 *
 * Returns:
 * The 6-bit value
 */
static gint
z_code_base64_decode_char(guchar c)
{
  static const char base64_alphabet[128] =
    {
      -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,     /*  0 */
      -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,     /* 16 */
      -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, 62, -1, -1, -1, 63,     /* 32 */
      52, 53, 54, 55, 56, 57, 58, 59, 60, 61, -1, -1, -1, -1, -1, -1,     /* 48 */
      -1,  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14,     /* 64 */
      15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, -1, -1, -1, -1, -1,     /* 80 */
      -1, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40,     /* 96 */
      41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, -1, -1, -1, -1, -1      /*112 */
    };

  z_enter();

  if (c == '=')
    {
      z_leave();
      return -2;
    }
  else if (c > 127)
    {
      z_leave();
      return -1;
    }
  else
    {
      z_leave();
      return base64_alphabet[c];
    }
  z_leave();
}
                                                                                                          

/**
 * z_code_base64_decode_quattro:
 * @self: this
 * @to: Output buffer
 * @tolen: Size of @to
 * @from: Input buffer
 *
 * Decode the next 4 input characters to (at most) 3*6 bits=24 bits=3 bytes,
 * and put these (at most) 3 bytes into the output buffer.
 *
 * Returns:
 * The number of decoded bytes produced (0..3)
 */
static guint
z_code_base64_decode_pack(ZCodeBase64 *self,
                             gchar *to, guint tolen,
                             gchar *from)
{
  signed char c[4];
  guint len;
  guint i, j;
  
  z_enter();
  c[0] = z_code_base64_decode_char(from[0]);
  c[1] = z_code_base64_decode_char(from[1]);
  c[2] = z_code_base64_decode_char(from[2]);
  c[3] = z_code_base64_decode_char(from[3]);

  len = 3;

  /* sanity check quantum */

  for (i = 0; i < 4; i++)
    if (c[i] == -2)
      {
        c[i] = 0;
        if (i == 2)
          len = 1;
        else if (i == 3)
          len = 2;
        else
          {
            z_leave();
            return 0;
          }

        for (j = i + 1; j < 4; j++)
          {
            if (c[j] != -2)
              {
                z_leave();
                return 0;
              }
            c[j] = 0;
          }
      }
    else if (c[i] == -1)
      {
        z_leave();
        return 0;
      }

  if (len > tolen)
    {
      self->err = TRUE;
      z_leave();
      return 0;
    }

  *to = (c[0] << 2) | ((c[1] & 0x30) >> 4);
  to++;

  if (len < 2)
    {
      z_leave();
      return len;
    }

  *to = ((c[1] & 0x0f) << 4) | ((c[2] & 0x3e) >> 2);
  to++;

  if (len < 3)
    {
      z_leave();
      return len;
    }

  *to = ((c[2] & 0x03) << 6) | (c[3] & 0x3f);

  z_leave();
  return len;
}

/**
 * z_code_base64_decode:
 * @code: this
 * @from: Input buffer
 * @fromlen: Number of input bytes
 * @to: Output buffer
 * @tolen: Number of bytes written into @to
 *
 * Decode the input buffer into the output.
 *
 * Returns:
 * TRUE on success
 */
static gboolean
z_code_base64_decode(ZCode    *code,
                     const gchar *from, guint fromlen,
                     gchar *to, guint *tolen)
{
  ZCodeBase64 *self = (ZCodeBase64 *)code;
  char tmp[4];
  guint len, i, j, frompos, pack_num, topos;
  gchar *strip_begin, *strip_end;

  z_enter();
  
  if (code->type != Z_CODE_BASE64)
    {
      z_leave();
      return FALSE;
    }

  if (self->err)
    {
      z_leave();
      return FALSE;
    }
  
  for (i = 0; i < self->len; i++)
    {
      tmp[i] = self->buf[i];
    }
  
  frompos = 0;

  strip_begin = (gchar *)from;
  strip_end = (gchar *)from + fromlen - 1;
      
  while(strip_begin <= strip_end && (*strip_begin == '\t' || *strip_begin == '\r' || *strip_begin == '\n' || *strip_begin == ' '))
    strip_begin++;

  while(strip_begin <= strip_end && (*strip_end == '\t' || *strip_end == '\r' || *strip_end == '\n' || *strip_end == ' '))
    strip_end--;

  frompos = 0;
  
  from = strip_begin;
  fromlen = strip_end - strip_begin + 1;

  pack_num = (fromlen + i) / 4;
  len = 1;
 
  topos = 0;
  
  while (pack_num && len && topos < *tolen - 1)
    {
      for (j = i; j < 4; j++)
        {
          tmp[j] = from[frompos];
          frompos++;
        }
      len = z_code_base64_decode_pack(self, &to[topos], *tolen - topos - 1, tmp);
      topos += len;
      i = 0;
      pack_num--;
    }
    
  if (len == 0)
    {
      self->err = TRUE;
      z_leave();
      return FALSE;
    }

  j = fromlen - frompos;
  if (j > 3)
    {
      self->err = TRUE;
      z_leave();
      return FALSE;
    }
    
  for (i = 0; i < j; i++)
    {
      self->buf[i] = from[frompos];
      frompos++;
    }
  self->len = i;

  to[topos] = 0;
  *tolen = topos;
  z_leave();
  return TRUE;
}

/**
 * z_code_base64_dstart:
 * @code: this
 *
 * Initialise before decoding
 *
 * Returns:
 * TRUE
 */
static gboolean
z_code_base64_dstart(ZCode *code)
{
  ZCodeBase64 *self = (ZCodeBase64 *)code;
  self->buf[0] = 0;
  self->buf[1] = 0;
  self->buf[2] = 0;
  self->len = 0;
  self->err = FALSE;
  return TRUE;
}

/**
 * z_code_base64_dfinish:
 * @code: this
 * @to: not used
 * @tolen: Base64 produces no output, always set to 0
 *
 * Base64 produces no digest output, so @tolen is always set to 0.
 *
 * Returns:
 * TRUE if decoding was successful
 */
static gboolean
z_code_base64_dfinish(ZCode *code,
                      gchar *to G_GNUC_UNUSED,
                      guint *tolen)
{
  ZCodeBase64 *self = (ZCodeBase64 *)code;
  if (self->len == 0)
    {
      *tolen = 0;
      return TRUE;
    }
  
  return FALSE;
}

/**
 * z_code_base64_encode_pack:
 * @self: not used
 * @to: Output buffer
 * @tolen: Size of @to
 * @from: Input buffer
 * @fromlen: Number of bytes to encode (0..3)
 *
 * Encodes at most @fromlen bytes from @from to @to
 *
 * Returns:
 * Number of encoded characters produced
 */
static guint
z_code_base64_encode_pack(ZCodeBase64 *self G_GNUC_UNUSED,
                            guchar *to, guint tolen,
                            guchar *from, guint fromlen)
{
  static const char base64_alphabet[128] =
    {
      -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,     /*  0 */
      -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,     /* 16 */
      -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, 62, -1, -1, -1, 63,     /* 32 */
      52, 53, 54, 55, 56, 57, 58, 59, 60, 61, -1, -1, -1, -1, -1, -1,     /* 48 */
      -1,  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14,     /* 64 */
      15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, -1, -1, -1, -1, -1,     /* 80 */
      -1, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40,     /* 96 */
      41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, -1, -1, -1, -1, -1      /*112 */
    };
  gchar c, tmp;;

  if (tolen < 4)
    return 0;

  if (fromlen > 0)
    {
      guint i;
      c = from[0] & 0x03;
      tmp = from[0] >> 2;
      for (i = 42; i < 125 && base64_alphabet[i] != tmp; i++);
      if (i > 125)
        return 0;
      to[0] = i;
      
      tmp = c << 4;
      if (fromlen > 1)
        {
          tmp += (from[1] >> 4);
          c = from[1] & 0x0f;
        }
      for (i = 42; i < 125 && base64_alphabet[i] != tmp; i++);
      if (i > 125)
        return 0;
      to[1] = i;
      
      if (fromlen <= 1)
        {
          to[2] = '=';
          to[3] = '=';
          return 4;
        }
      tmp = c << 2;
      if (fromlen > 2)
        {
          tmp += (from[2] >> 6);
          c = from[2] & 0x3f;
        }
      for (i = 42; i < 125 && base64_alphabet[i] != tmp; i++);
      if (i > 125)
        return 0;
      to[2] = i;
      
      if (fromlen <= 2)
        {
          to[3] = '=';
          return 4;
        }
      for (i = 42; i < 125 && base64_alphabet[i] != c; i++);
      if (i > 125)
        return 0;
      to[3] = i;
    }
  return 4;
}

/**
 * z_code_base64_encode:
 * @code: this
 * @from: Input buffer
 * @fromlen: Length of @from
 * @to: Output buffer
 * @tolen: In: size of @to, out: number of bytes written into @to
 * 
 * Encodes @from to @to
 *
 * Returns:
 * TRUE on success
 */
static gboolean
z_code_base64_encode(ZCode *code,
                     const gchar *from, guint fromlen,
                     gchar *to, guint *tolen)
{
  ZCodeBase64 *self = (ZCodeBase64 *)code;
  char tmp[3];
  guint len, i, j, frompos, pack_num, topos;
  
  z_enter();
  
  if (code->type != Z_CODE_BASE64)
    {
      z_leave();
      return FALSE;
    }

  if (self->err)
    {
      z_leave();
      return FALSE;
    }
  
  for (i = 0; i < self->len; i++)
    {
      tmp[i] = self->buf[i];
    }

  frompos = 0;
  
  pack_num = (fromlen + i) / 3;
  
  len = 1;
 
  topos = 0;
  
  while (pack_num && len && topos < *tolen - 1)
    {
      for (j = i; j < 3; j++)
        {
          tmp[j] = from[frompos];
          frompos++;
        }
      len = z_code_base64_encode_pack(self, &to[topos], *tolen - topos - 1, tmp, 3);
      topos += len;
      i = 0;
      pack_num--;
      self->encode_line_pos += len;

      if (self->encode_line_pos >= self->max_line_length)
        {
          to[topos++]='\r';
          to[topos++]='\n';
          self->encode_line_pos = 0;
        }
    }
    
  if (len == 0)
    {
      self->err = TRUE;
      z_leave();
      return FALSE;
    }

  j = fromlen - frompos;
  if (j > 2)
    {
      self->err = TRUE;
      z_leave();
      return FALSE;
    }
    
  for (i = 0; i < j; i++)
    {
      self->buf[i] = from[frompos];
      frompos++;
    }
  self->len = i;

  to[topos] = 0;
  *tolen = topos;
  z_leave();
  return TRUE;
  z_leave();
}

/**
 * z_code_base64_estart:
 * @code: this
 *
 * Initialise before encoding
 *
 * Returns:
 * 
 */
static gboolean
z_code_base64_estart(ZCode *code)
{
  ZCodeBase64 *self = (ZCodeBase64 *)code;
  self->buf[0] = 0;
  self->buf[1] = 0;
  self->buf[2] = 0;
  self->len = 0;
  self->err = FALSE;
  return TRUE;
}

/**
 * z_code_base64_efinish:
 * @code: this
 * @to: Output buffer
 * @tolen: In: size of @to, out: number of bytes written to @to
 *
 * Encodes the remaining bytes (if there's any) and closes the output line by
 * CRLF.
 *
 * Returns:
 * TRUE on success
 */
static gboolean
z_code_base64_efinish(ZCode *code,
                      gchar *to, guint *tolen)
{
  ZCodeBase64 *self = (ZCodeBase64 *)code;
  guint len;
 
  if (self->len == 0)
    {
      len = 0;
      if (self->encode_line_pos != 0)
        {
          to[len++]='\r';
          to[len++]='\n';
  
          to[len] = 0;
        }
      *tolen = len;
      return TRUE;
    }
  
  len = z_code_base64_encode_pack(self, to, *tolen - 1, self->buf, self->len);
    
  if (len == 0)
    {
      self->err = TRUE;
      z_leave();
      return FALSE;
    }

  to[len++]='\r';
  to[len++]='\n';
  
  to[len] = 0;
  *tolen = len;
  
  return TRUE;
}

/**
 * z_code_base64_new:
 *
 * Constructor
 *
 * Returns:
 * The new instance
 */
ZCode *
z_code_base64_new(void)
{
  ZCodeBase64 *self;
  
  z_enter();
  self = g_new0(ZCodeBase64, 1);
  
  self->super.type = Z_CODE_BASE64;
  self->super.decode_start = z_code_base64_dstart;
  self->super.decode = z_code_base64_decode;
  self->super.decode_finish = z_code_base64_dfinish;
  self->super.encode_start = z_code_base64_estart;
  self->super.encode = z_code_base64_encode;
  self->super.encode_finish = z_code_base64_efinish;
  self->max_line_length = 76;
  z_leave();
  return &self->super;
}
