############################################################################
##
## Copyright (c) 2000, 2001, 2002, 2003, 2004, 2005 BalaBit IT Ltd, Budapest, Hungary
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; if not, write to the Free Software
## Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
##
##
## $Id: Ftp.py,v 1.94 2004/07/19 16:56:01 sasa Exp $
##
## Author  : sasa
## Auditor :  
## Last audited version: 
## Notes:
##
############################################################################

"""Module exporting the Ftp proxy interface

This module defines the Ftp proxy interface as implemented by the
Ftp Zorp module.
"""

from Zorp import *
from Plug import PlugProxy
from Proxy import Proxy, proxyLog
from SockAddr import SockAddrInet, SockAddrInetRange
from Session import StackedSession
from Stream import Stream

FTP_DATA_KEEP    = 0
FTP_DATA_PASSIVE = 1
FTP_DATA_ACTIVE  = 2

FTP_ACCEPT = 1
FTP_DENY   = 3
FTP_ABORT  = 4
FTP_POLICY = 6

FTP_REQ_ACCEPT = 1
FTP_REQ_REJECT = 3
FTP_REQ_ABORT  = 4
FTP_REQ_POLICY = 6

FTP_RSP_ACCEPT = 1
FTP_RSP_REJECT = 3
FTP_RSP_ABORT  = 4
FTP_RSP_POLICY = 6

FTP_DEBUG  = "ftp.debug"
FTP_ERROR  = "ftp.error"
FTP_POLICY = "ftp.policy"

FTP_ACTIVE_MINUSONE = 0
FTP_ACTIVE_TWENTY   = 1
FTP_ACTIVE_RANDOM   = 2

FTP_STK_DATA = 3
FTP_STK_NONE = 1

class FtpDataProxy(PlugProxy):
	"""Class encapsulating an FTP data connection.
	
	This class is used as a top-level proxy for FTP data connections.
	It is a simple plug (e.g. no content verification is done), but
	this class ensures that only one way traffic is transmitted
	by setting the appropriate copy_to_* attributes in Plug.
	
	"""
	def config(self):
		"""Configuration for FtpDataProxy
		
		This function sets the configuration for FtpDataProxy.  Our
		desired configuration is to restrict data transfer to one
		way only according to 'self.session.way'.
		
		Attributes
		
		  self -- this instance
		"""
		self.copy_to_client = 0
		if self.session.ftp_stack != None:
			self.shutdown_soft = TRUE
			self.stack_proxy = self.session.ftp_stack

	def __pre_shutdown__(self):
		"""Function called when the session is terminated.
		
		We call the registered stop() function to indicate the end
		of our session.
		
		Arguments
		
		  self -- this instance
		"""
		self.session.ftp_data_stop()

class AbstractFtpProxy(Proxy):
	"""Class encapsulating the FTP proxy implemented in Zorp.
	
	This proxy implements the FTP protocol. It denies everything by
	default, so uou should dervice your customized Ftp proxy classes in
	order to do something useful, or use 'FtpProxy' or
	'FtpProxyRO', etc.

	Usage
	
	  All requests are denied in the low level proxy implementation by
	  default. To enable different commands and/or answers you'll need to
	  extend the low-level proxy in Python. To see how this is done,
	  see the next two sections.
	  
	  Setting policy for commands
	  
	    Changing the default behaviour of commands can be done using the
	    hash named 'self.request'. This hash is indexed by the command name
	    (e.g: USER or PWD), and each item in this hash is an action
	    tuple, defining proxy behaviour for the given command.
	    
	    The first item in this tuple is an integer value, determining
	    the action to take, and also the interpretation of the remaining
	    items in the tuple.
	  
	    Possible actions (Possible actions for commands in FTP)
	  
	      FTP_REQ_ACCEPT -- [] allow request without modification.

	      FTP_REQ_REJECT -- [QSTRING] reject request. (default) Second parameter
	                        contains a string, which will be sent back to client.

	      FTP_REQ_POLICY -- [METHOD] call a Python function to decide what to do.
	                        this value uses an additional tuple item,
	                        which must be a callable Python function.
	                        The function takes two parameters:
	                        'self', 'command'

	      FTP_REQ_ABORT  -- [] reject request, and drop connection.
	  
	      Example (Sample for anonymous only sessions in FTP)

	        class AnonFtp(FtpProxy):

	          def config(self):
	            self.request["USER"] = (FTP_REQ_POLICY, self.pUser)
	            self.request["*"] = (FTP_REQ_ACCEPT)

	          def pUser(self,command):
	            if self.request_parameter == "ftp" or self.request_parameter == "anonymous":
	              return Ftp.FTP_REQ_ACCEPT
	            return Ftp.FTP_REQ_REJECT

	  Changing answer code or string

	    Like with commands, we have a hash for answers as well.  This
	    hash indexed by command and answer, and contains an 
	    action tuple in each item. A special wildcard character '*' matches
	    every command. Another feature is that you don't need to write the
	    full answer code, it's possible to write only the first, or the
	    first two digits. 
	    
	    For example, all three index below is valid (Valid response codes in FTP)

	      "PWD","200"  -- Match exactly the answer 200 coming in reply to a 
	                 PWD command.

	      "PWD","2"    -- Match every answer starting with '2' in reply to a 
	                 PWD command.

	      "*","20"     -- Match every answer between 200 and 209 in reply to 
	                 any command.

	    All answers are "rejected" by default. It's changed to 
	    "500 Error parsing answer", because dropping an answer is not
	    permitted by RFC959. You may enable all answers with "*","*". 
	    
	    The precedence in processing hashtable entries is the following:

	      1. Exact match. ("PWD","200")
	        
	      2. Exact command match, partial answer matches ("PWD","20", "PWD","2", "PWD","*")
	        
	      3. Wildcard command, with answer codes repeated as above. 
	         ("*","200", "*","20", "*","2", "*","*")

	    Possible values for the first item in the action tuple (Possible action codes for responses in FTP)

	      FTP_RSP_ACCEPT -- [] allow answer without modification.

	      FTP_RSP_REJECT -- [QSTRING] change answer code and message. (default) 
	                        Second parameter contains a string, which 
	                        will be sent back to client.

	      FTP_RSP_POLICY -- [METHOD] call a Python function to decide what to do.
	                        this value uses an additional parameter,
	                        which must be a callable Python function.
	                        The function takes three parameters:
	                        self, command, answer

	      FTP_RSP_ABORT  -- [] deny request, and drop conection.

	Stacking

	  FTP_STK_DATA -- [CLASS_proxy]
	  FTP_STK_NONE -- []

	Attributes 
	 
	  request_stack               -- [HASH;STRING;FTP_STK:empty:RW:RW] this
					 hash contains the stacking policy
					 for various FTP commands. The hash
					 is indexed by the FTP command
					 (RETR, STOR) and contains an
					 (FTP_STK_*, class) tuple
	  
	  data_port_min               -- [INTEGER:40000:RW:R] ports should be
					 allocated above or equal this variable
	  
	  data_port_max               -- [INTEGER:41000:RW:R] ports should be
					 allocated below or equal this variable
	  
	  data_mode                   -- [ENUM;FTP_DATA:FTP_DATA_KEEP:RW:R] FTP_DATA_KEEP,
					 FTP_DATA_PASSIVE or FTP_DATA_ACTIVE
					 can be used to force a connection
					 type in server side.
				     
	  masq_address_client         -- [STRING:"":RW:R]
					 Firewall address in client side.
					 If it's set, zorp will send this
					 IP regardless where it's binded.
					 It's may be used, when there a NAT-ing
					 firewall before Zorp.
				     
	  masq_address_server         -- [STRING:"":RW:R]
					 Firewall address in server side.
					 If it's set, zorp will send this
					 IP regardless where it's binded.
					 It's may be used, when there a NAT-ing
					 firewall before Zorp.
				     
	  max_line_length             -- [INTEGER:255:RW:R] Maximum
					 line length, what a proxy may
					 transfer.
				     
	  max_username_length         -- [INTEGER:32:RW:R] Maximum
					 length of usernames.
				     
	  max_password_length         -- [INTEGER:64:RW:R] Maximum
					 length of passwords.
				     
	  max_hostname_length         -- [INTEGER:128:RW:R] Maximum
					 length of hostname. (Used in
					 non-transparent mode.)

	  password                    -- [STRING:"":-:R] contains the
					 password authenticated to the
					 server.

	  permit_unknown_command      -- [BOOLEAN:FALSE:RW:R] Enable
					 commands not understood (and not
					 handled) by proxy.
				     
	  permit_empty_command        -- [BOOLEAN:TRUE:RW:R] Enable
					 lines without commands.
				     
	  request                     -- [HASH;QSTRING;FTP_REQ:empty:RW:RW] normative
					 policy hash, directing the proxy
					 to do something with requests, without
					 the need to call Python.  indexed by
					 the command (e.g. "USER", "PWD" etc)
				     
	  response                    -- [HASH;QSTRING,QSTRING;FTP_RSP:empty:RW:RW]
					 normative policy hash directing the
					 proxy to do something with answers.
					 Indexed by the command, and the
					 answer (e.g. "USER","331"; "PWD",
					 "200" etc)
				     
	  request_command             -- [STRING:n/a:-:RW] When
					 a command goes up to policy level,
					 this contain it.
				     
	  request_parameter           -- [STRING:"":-:RW] When a
					 command goes up to policy level,
					 this variable contains command
					 parameters.
				     
	  response_status             -- [STRING:"":-:RW]
					 When an answer goes up
					 to policy level, this variable
					 contains answer code.
				     
	  response_parameter          -- [STRING:"":-:RW] When an
					 answer go up to policy level,
					 this variable contain answer
					 parameters.
				     
	  response_strip_msg          -- [BOOLEAN:FALSE:RW:R] Strip the
					 response message and only send the
					 response code.

	  target_port_range           -- [STRING:"21":RW:R] In non-Transparent
					 mode this set where client may
					 connect trough ftp proxy

	  timeout                     -- [INTEGER:300000:RW:R] general I/O
					 timeout in milliseconds. When
					 there is no specific timeout
					 for a given operation, this
					 value is used.
	
	  transparent_mode            -- [INTEGER:TRUE:RW:R]  TRUE
					 for transparent proxy, FALSE 
					 otherwise

	  username                    -- [STRING:"":-:R] contains the
					 username authenticated to the
					 server.

	  valid_chars_username        -- [STRING:"a-zA-Z0-9._@":RW:R]
	                                 contains the accepted characters in
	                                 user names.

	  active_connection_mode      -- [ENUM;FTP_ACTIVE:FTP_ACTIVE_MINUSONE:RW:R] In active
                                         mode server connect to the client. In default this
                                         must be from command port minus one. You may choose
                                         to connect from fix port (20) or from random port.

	  strict_port_checking        -- [BOOLEAN:TRUE:RW:RW] If enabed Zorp
					 will strictly check the foreign
					 port. In active mode server must be
					 connected from port 20. In any
					 other situation the foreign port
					 must be above 1023

	"""
	name = "ftp"
	def __init__(self, session):
		"""Constructor to initialize an FtpProxy instance
		
		This constructor initializes an FtpProxy instance by
		calling the inherited __init__ constructor
		with appropriate parameters, and setting up
		local attributes based on arguments.
		
		Arguments

		  self          --      this instance
		  
		  session	--	The session object
		  
		"""
		self.restrict_client_connect = TRUE
		self.restrict_server_connect = FALSE
		self.data_proxy = FtpDataProxy
		self.data_port_min = 40000
		self.data_port_max = 41000
		self.request_stack = {}
		self.strict_port_checking = TRUE
		Proxy.__init__(self, session)

	def __pre_startup__(self):
		Proxy.__pre_startup__(self)
		self.session.ftp_data_stop = self.ftpDataStop

	def __destroy__(self):
		Proxy.__destroy__(self)
		try:
			del self.session.ftp_data_stop
		except AttributeError:
			pass
	
	def ftpDataStop(self):
		"""Function called by the data proxy to indicate the end of the data transfer

		This callback is registered to the data proxy
		('FtpDataProxy' by default), to be called when
		the data connection is terminated. Its task is
		to signal this condition to the Ftp proxy core.

		This function is implemented in C in reality, this stub
		serves documentation purposes only.

		Arguments

		  self -- this instance
		"""
		pass

	def bounceCheck(self, remote, side, connect):
		"""
		  If want to check bounce attack, you may write the bounceCheck
		  function. It's get the SockAddr of the remote side, the
		  side where the connection go to and a boolean value, that TRUE
		  if Zorp initiate the connection.
		"""
		if side == 0:
			ret = (remote.ip == self.session.client_address.ip)
			if self.strict_port_checking:
				if remote.port < 1024:
					## LOG ##
					# This message indicates that the remote port is bellow 1024 and due to the
					# violation Zorp is closing connection.
					##
					proxyLog(self, FTP_POLICY, 3, "Client foreign port below 1024; port='%d'" % remote.port)
					ret = FALSE
		elif side == 1:
			ret = (remote.ip == self.session.server_address.ip)
			if self.strict_port_checking:
				if connect:
					if remote.port < 1024:
						## LOG ##
						# This message indicates that the remote port is bellow 1024 and due to the
						# violation Zorp is closing connection.
						##
						proxyLog(self, FTP_POLICY, 3, "Server foreign port below 1024 in passive mode; port='%d'" % remote.port)
						ret = FALSE
				else:
					if remote.port != 20 and remote.port != self.session.server_address.port - 1:
						## LOG ##
						# This message indicates that the server's remote port is not control_port-1 or 20 and due to the
						# violation Zorp is closing connection.
						##
						proxyLog(self, FTP_POLICY, 3, "Server foreign port is not good in active mode; port='%d', control_port='%d'" % (remote.port, self.session.server_address.port))
						ret = FALSE
		else:
			## LOG ##
			# This message indicates an internal error, please contact the BalaBit QA team.
			##
			proxyLog(self, FTP_POLICY, 3, "Unknown side when calling bounceCheck; side='%d'" % side)
			ret = FALSE

		return ret

	def stopDataConnection(self):
		"""Function called by the data proxy to indicate the end of the

		This callback is registered to the data proxy
		('FtpDataProxy' by default), to be called when
		the data connection is terminated. It's task is
		to signal this condition to the Ftp proxy core.

		Arguments

		  self -- this instance

		"""
		self.ftpDataStop()

	def loadAnswers(self):
		"""This function can be called by derived classes to initialize internal hashtables.

		This function fills in the self.answers hash so that commonly used
		request/answer combinations are enabled.

		Arguments

		  self -- this instance
		  
		"""
		self.response["*", "421"]    = (FTP_RSP_ABORT, "421 Logoff")
		self.response["*", "500"]    = (FTP_RSP_ACCEPT)

		self.response["Null", "120"] = (FTP_RSP_ACCEPT)
		self.response["Null", "220"] = (FTP_RSP_ACCEPT)

		self.response["ABOR", "225"] = (FTP_RSP_ACCEPT)
		self.response["ABOR", "226"] = (FTP_RSP_ACCEPT)
		self.response["ABOR", "501"] = (FTP_RSP_ACCEPT)
		self.response["ABOR", "502"] = (FTP_RSP_ACCEPT)

		self.response["ACCT", "202"] = (FTP_RSP_ACCEPT)
		self.response["ACCT", "230"] = (FTP_RSP_ACCEPT)
		self.response["ACCT", "501"] = (FTP_RSP_ACCEPT)
		self.response["ACCT", "503"] = (FTP_RSP_ACCEPT)
		self.response["ACCT", "530"] = (FTP_RSP_ACCEPT)

		self.response["ALLO", "200"] = (FTP_RSP_ACCEPT)
		self.response["ALLO", "202"] = (FTP_RSP_ACCEPT)
		self.response["ALLO", "501"] = (FTP_RSP_ACCEPT)
		self.response["ALLO", "504"] = (FTP_RSP_ACCEPT)
		self.response["ALLO", "530"] = (FTP_RSP_ACCEPT)

		self.response["APPE", "110"] = (FTP_RSP_ACCEPT)
		self.response["APPE", "125"] = (FTP_RSP_ACCEPT)
		self.response["APPE", "150"] = (FTP_RSP_ACCEPT)
		self.response["APPE", "226"] = (FTP_RSP_ACCEPT)
		self.response["APPE", "250"] = (FTP_RSP_ACCEPT)
		self.response["APPE", "425"] = (FTP_RSP_ACCEPT)
		self.response["APPE", "426"] = (FTP_RSP_ACCEPT)
		self.response["APPE", "450"] = (FTP_RSP_ACCEPT)
		self.response["APPE", "451"] = (FTP_RSP_ACCEPT)
		self.response["APPE", "452"] = (FTP_RSP_ACCEPT)
		self.response["APPE", "501"] = (FTP_RSP_ACCEPT)
		self.response["APPE", "502"] = (FTP_RSP_ACCEPT)
		self.response["APPE", "530"] = (FTP_RSP_ACCEPT)
		self.response["APPE", "551"] = (FTP_RSP_ACCEPT)
		self.response["APPE", "552"] = (FTP_RSP_ACCEPT)
		self.response["APPE", "532"] = (FTP_RSP_ACCEPT)
		self.response["APPE", "550"] = (FTP_RSP_ACCEPT)
		self.response["APPE", "553"] = (FTP_RSP_ACCEPT)

		self.response["CDUP", "200"] = (FTP_RSP_ACCEPT)
		self.response["CDUP", "501"] = (FTP_RSP_ACCEPT)
		self.response["CDUP", "502"] = (FTP_RSP_ACCEPT)
		self.response["CDUP", "530"] = (FTP_RSP_ACCEPT)
		self.response["CDUP", "550"] = (FTP_RSP_ACCEPT)

		self.response["CWD", "250"]  = (FTP_RSP_ACCEPT)
		self.response["CWD", "501"]  = (FTP_RSP_ACCEPT)
		self.response["CWD", "502"]  = (FTP_RSP_ACCEPT)
		self.response["CWD", "530"]  = (FTP_RSP_ACCEPT)
		self.response["CWD", "550"]  = (FTP_RSP_ACCEPT)

		self.response["DELE", "250"] = (FTP_RSP_ACCEPT)
		self.response["DELE", "450"] = (FTP_RSP_ACCEPT)
		self.response["DELE", "550"] = (FTP_RSP_ACCEPT)
		self.response["DELE", "501"] = (FTP_RSP_ACCEPT)
		self.response["DELE", "502"] = (FTP_RSP_ACCEPT)
		self.response["DELE", "530"] = (FTP_RSP_ACCEPT)

		self.response["EPRT", "200"] = (FTP_RSP_ACCEPT)
		self.response["EPRT", "501"] = (FTP_RSP_ACCEPT)
		self.response["EPRT", "522"] = (FTP_RSP_ACCEPT)

		self.response["EPSV", "229"] = (FTP_RSP_ACCEPT)
		self.response["EPSV", "501"] = (FTP_RSP_ACCEPT)

		self.response["HELP", "211"] = (FTP_RSP_ACCEPT)
		self.response["HELP", "214"] = (FTP_RSP_ACCEPT)
		self.response["HELP", "501"] = (FTP_RSP_ACCEPT)
		self.response["HELP", "502"] = (FTP_RSP_ACCEPT)

		self.response["LIST", "125"] = (FTP_RSP_ACCEPT)
		self.response["LIST", "150"] = (FTP_RSP_ACCEPT)
		self.response["LIST", "226"] = (FTP_RSP_ACCEPT)
		self.response["LIST", "250"] = (FTP_RSP_ACCEPT)
		self.response["LIST", "425"] = (FTP_RSP_ACCEPT)
		self.response["LIST", "426"] = (FTP_RSP_ACCEPT)
		self.response["LIST", "451"] = (FTP_RSP_ACCEPT)
		self.response["LIST", "450"] = (FTP_RSP_ACCEPT)
		self.response["LIST", "501"] = (FTP_RSP_ACCEPT)
		self.response["LIST", "502"] = (FTP_RSP_ACCEPT)
		self.response["LIST", "530"] = (FTP_RSP_ACCEPT)

		self.response["MDTM", "213"] = (FTP_RSP_ACCEPT)
		self.response["MDTM", "501"] = (FTP_RSP_ACCEPT) #Hmmm.
		self.response["MDTM", "550"] = (FTP_RSP_ACCEPT)

		self.response["MKD", "257"]  = (FTP_RSP_ACCEPT)
		self.response["MKD", "501"]  = (FTP_RSP_ACCEPT)
		self.response["MKD", "502"]  = (FTP_RSP_ACCEPT)
		self.response["MKD", "530"]  = (FTP_RSP_ACCEPT)
		self.response["MKD", "550"]  = (FTP_RSP_ACCEPT)

		self.response["MODE", "200"] = (FTP_RSP_ACCEPT)
		self.response["MODE", "501"] = (FTP_RSP_ACCEPT)
		self.response["MODE", "504"] = (FTP_RSP_ACCEPT)
		self.response["MODE", "530"] = (FTP_RSP_ACCEPT)

		self.response["NLST", "125"] = (FTP_RSP_ACCEPT)
		self.response["NLST", "150"] = (FTP_RSP_ACCEPT)
		self.response["NLST", "226"] = (FTP_RSP_ACCEPT)
		self.response["NLST", "250"] = (FTP_RSP_ACCEPT)
		self.response["NLST", "425"] = (FTP_RSP_ACCEPT)
		self.response["NLST", "426"] = (FTP_RSP_ACCEPT)
		self.response["NLST", "450"] = (FTP_RSP_ACCEPT)
		self.response["NLST", "451"] = (FTP_RSP_ACCEPT)
		self.response["NLST", "501"] = (FTP_RSP_ACCEPT)
		self.response["NLST", "502"] = (FTP_RSP_ACCEPT)
		self.response["NLST", "530"] = (FTP_RSP_ACCEPT)
		self.response["NLST", "550"] = (FTP_RSP_ACCEPT)

		self.response["NOOP", "200"] = (FTP_RSP_ACCEPT)

		self.response["PASS", "202"] = (FTP_RSP_ACCEPT)
		self.response["PASS", "230"] = (FTP_RSP_ACCEPT)
		self.response["PASS", "332"] = (FTP_RSP_ACCEPT)
		self.response["PASS", "501"] = (FTP_RSP_ACCEPT)
		self.response["PASS", "503"] = (FTP_RSP_ACCEPT)
		self.response["PASS", "530"] = (FTP_RSP_ACCEPT)

		self.response["PASV", "227"] = (FTP_RSP_ACCEPT)
		self.response["PASV", "501"] = (FTP_RSP_ACCEPT)
		self.response["PASV", "502"] = (FTP_RSP_ACCEPT)
		self.response["PASV", "530"] = (FTP_RSP_ACCEPT)

		self.response["PORT", "200"] = (FTP_RSP_ACCEPT)
		self.response["PORT", "501"] = (FTP_RSP_ACCEPT)
		self.response["PORT", "530"] = (FTP_RSP_ACCEPT)

		self.response["PWD", "257"]  = (FTP_RSP_ACCEPT)
		self.response["PWD", "501"]  = (FTP_RSP_ACCEPT)
		self.response["PWD", "502"]  = (FTP_RSP_ACCEPT)
		self.response["PWD", "550"]  = (FTP_RSP_ACCEPT)

		self.response["QUIT", "221"] = (FTP_RSP_ACCEPT)

		self.response["REIN", "120"] = (FTP_RSP_ACCEPT)
		self.response["REIN", "220"] = (FTP_RSP_ACCEPT)
		self.response["REIN", "502"] = (FTP_RSP_ACCEPT)

		self.response["REST", "350"] = (FTP_RSP_ACCEPT)
		self.response["REST", "501"] = (FTP_RSP_ACCEPT)
		self.response["REST", "502"] = (FTP_RSP_ACCEPT)
		self.response["REST", "530"] = (FTP_RSP_ACCEPT)

		self.response["RETR", "110"] = (FTP_RSP_ACCEPT)
		self.response["RETR", "125"] = (FTP_RSP_ACCEPT)
		self.response["RETR", "150"] = (FTP_RSP_ACCEPT)
		self.response["RETR", "226"] = (FTP_RSP_ACCEPT)
		self.response["RETR", "250"] = (FTP_RSP_ACCEPT)
		self.response["RETR", "425"] = (FTP_RSP_ACCEPT)
		self.response["RETR", "426"] = (FTP_RSP_ACCEPT)
		self.response["RETR", "450"] = (FTP_RSP_ACCEPT)
		self.response["RETR", "451"] = (FTP_RSP_ACCEPT)
		self.response["RETR", "452"] = (FTP_RSP_ACCEPT)
		self.response["RETR", "501"] = (FTP_RSP_ACCEPT)
		self.response["RETR", "530"] = (FTP_RSP_ACCEPT)
		self.response["RETR", "532"] = (FTP_RSP_ACCEPT)
		self.response["RETR", "550"] = (FTP_RSP_ACCEPT)
		self.response["RETR", "553"] = (FTP_RSP_ACCEPT)

		self.response["RMD", "250"]  = (FTP_RSP_ACCEPT)
		self.response["RMD", "501"]  = (FTP_RSP_ACCEPT)
		self.response["RMD", "502"]  = (FTP_RSP_ACCEPT)
		self.response["RMD", "530"]  = (FTP_RSP_ACCEPT)
		self.response["RMD", "550"]  = (FTP_RSP_ACCEPT)

		self.response["RNFR", "350"] = (FTP_RSP_ACCEPT)
		self.response["RNFR", "450"] = (FTP_RSP_ACCEPT)
		self.response["RNFR", "501"] = (FTP_RSP_ACCEPT)
		self.response["RNFR", "502"] = (FTP_RSP_ACCEPT)
		self.response["RNFR", "530"] = (FTP_RSP_ACCEPT)
		self.response["RNFR", "550"] = (FTP_RSP_ACCEPT)

		self.response["RNTO", "250"] = (FTP_RSP_ACCEPT)
		self.response["RNTO", "501"] = (FTP_RSP_ACCEPT)
		self.response["RNTO", "502"] = (FTP_RSP_ACCEPT)
		self.response["RNTO", "530"] = (FTP_RSP_ACCEPT)
		self.response["RNTO", "532"] = (FTP_RSP_ACCEPT)
		self.response["RNTO", "553"] = (FTP_RSP_ACCEPT)

		self.response["SITE", "200"] = (FTP_RSP_ACCEPT)
		self.response["SITE", "202"] = (FTP_RSP_ACCEPT)
		self.response["SITE", "501"] = (FTP_RSP_ACCEPT)
		self.response["SITE", "530"] = (FTP_RSP_ACCEPT)

		self.response["SIZE", "213"] = (FTP_RSP_ACCEPT)
		self.response["SIZE", "550"] = (FTP_RSP_ACCEPT)

		self.response["SMNT", "202"] = (FTP_RSP_ACCEPT)
		self.response["SMNT", "250"] = (FTP_RSP_ACCEPT)
		self.response["SMNT", "501"] = (FTP_RSP_ACCEPT)
		self.response["SMNT", "502"] = (FTP_RSP_ACCEPT)
		self.response["SMNT", "530"] = (FTP_RSP_ACCEPT)
		self.response["SMNT", "550"] = (FTP_RSP_ACCEPT)

		self.response["STAT", "211"] = (FTP_RSP_ACCEPT)
		self.response["STAT", "212"] = (FTP_RSP_ACCEPT)
		self.response["STAT", "213"] = (FTP_RSP_ACCEPT)
		self.response["STAT", "450"] = (FTP_RSP_ACCEPT)
		self.response["STAT", "501"] = (FTP_RSP_ACCEPT)
		self.response["STAT", "502"] = (FTP_RSP_ACCEPT)
		self.response["STAT", "530"] = (FTP_RSP_ACCEPT)

		self.response["STOR", "110"] = (FTP_RSP_ACCEPT)
		self.response["STOR", "125"] = (FTP_RSP_ACCEPT)
		self.response["STOR", "150"] = (FTP_RSP_ACCEPT)
		self.response["STOR", "226"] = (FTP_RSP_ACCEPT)
		self.response["STOR", "250"] = (FTP_RSP_ACCEPT)
		self.response["STOR", "425"] = (FTP_RSP_ACCEPT)
		self.response["STOR", "426"] = (FTP_RSP_ACCEPT)
		self.response["STOR", "450"] = (FTP_RSP_ACCEPT)
		self.response["STOR", "451"] = (FTP_RSP_ACCEPT)
		self.response["STOR", "452"] = (FTP_RSP_ACCEPT)
		self.response["STOR", "501"] = (FTP_RSP_ACCEPT)
		self.response["STOR", "530"] = (FTP_RSP_ACCEPT)
		self.response["STOR", "532"] = (FTP_RSP_ACCEPT)
		self.response["STOR", "550"] = (FTP_RSP_ACCEPT)
		self.response["STOR", "551"] = (FTP_RSP_ACCEPT)
		self.response["STOR", "552"] = (FTP_RSP_ACCEPT)
		self.response["STOR", "553"] = (FTP_RSP_ACCEPT)

		self.response["STOU", "110"] = (FTP_RSP_ACCEPT)
		self.response["STOU", "125"] = (FTP_RSP_ACCEPT)
		self.response["STOU", "150"] = (FTP_RSP_ACCEPT)
		self.response["STOU", "226"] = (FTP_RSP_ACCEPT)
		self.response["STOU", "250"] = (FTP_RSP_ACCEPT)
		self.response["STOU", "425"] = (FTP_RSP_ACCEPT)
		self.response["STOU", "426"] = (FTP_RSP_ACCEPT)
		self.response["STOU", "450"] = (FTP_RSP_ACCEPT)
		self.response["STOU", "451"] = (FTP_RSP_ACCEPT)
		self.response["STOU", "452"] = (FTP_RSP_ACCEPT)
		self.response["STOU", "501"] = (FTP_RSP_ACCEPT)
		self.response["STOU", "530"] = (FTP_RSP_ACCEPT)
		self.response["STOU", "532"] = (FTP_RSP_ACCEPT)
		self.response["STOU", "551"] = (FTP_RSP_ACCEPT)
		self.response["STOU", "552"] = (FTP_RSP_ACCEPT)
		self.response["STOU", "553"] = (FTP_RSP_ACCEPT)

		self.response["STRU", "200"] = (FTP_RSP_ACCEPT)
		self.response["STRU", "501"] = (FTP_RSP_ACCEPT)
		self.response["STRU", "504"] = (FTP_RSP_ACCEPT)
		self.response["STRU", "530"] = (FTP_RSP_ACCEPT)

		self.response["SYST", "215"] = (FTP_RSP_ACCEPT)
		self.response["SYST", "501"] = (FTP_RSP_ACCEPT)
		self.response["SYST", "502"] = (FTP_RSP_ACCEPT)

		self.response["TYPE", "200"] = (FTP_RSP_ACCEPT)
		self.response["TYPE", "501"] = (FTP_RSP_ACCEPT)
		self.response["TYPE", "504"] = (FTP_RSP_ACCEPT)
		self.response["TYPE", "530"] = (FTP_RSP_ACCEPT)

		self.response["USER", "230"] = (FTP_RSP_ACCEPT)
		self.response["USER", "331"] = (FTP_RSP_ACCEPT)
		self.response["USER", "332"] = (FTP_RSP_ACCEPT)
		self.response["USER", "501"] = (FTP_RSP_ACCEPT)
		self.response["USER", "530"] = (FTP_RSP_ACCEPT)

	def loadMinimalCommands(self):
		"""This function enable some minimal command set

		This function enable a minimal set of commands, for various
		subclass
		"""
		self.request["ABOR"] = (FTP_REQ_ACCEPT)
		self.request["ACCT"] = (FTP_REQ_ACCEPT)
		self.request["CDUP"] = (FTP_REQ_ACCEPT)
		self.request["CWD"]  = (FTP_REQ_ACCEPT)
		self.request["EPRT"] = (FTP_REQ_ACCEPT)
		self.request["EPSV"] = (FTP_REQ_ACCEPT)
		self.request["LIST"] = (FTP_REQ_ACCEPT)
		self.request["MODE"] = (FTP_REQ_ACCEPT)
		self.request["MDTM"] = (FTP_REQ_ACCEPT)
		self.request["NLST"] = (FTP_REQ_ACCEPT)
		self.request["NOOP"] = (FTP_REQ_ACCEPT)
		self.request["PASV"] = (FTP_REQ_ACCEPT)
		self.request["PASS"] = (FTP_REQ_ACCEPT)
		self.request["PORT"] = (FTP_REQ_ACCEPT)
		self.request["PWD"]  = (FTP_REQ_ACCEPT)
		self.request["QUIT"] = (FTP_REQ_ACCEPT)
		self.request["REST"] = (FTP_REQ_ACCEPT)
		self.request["RETR"] = (FTP_REQ_ACCEPT)
		self.request["SIZE"] = (FTP_REQ_ACCEPT)
		self.request["STAT"] = (FTP_REQ_ACCEPT)
		self.request["STRU"] = (FTP_REQ_ACCEPT)
		self.request["SYST"] = (FTP_REQ_ACCEPT)
		self.request["TYPE"] = (FTP_REQ_ACCEPT)

		self.request["AUTH"] = (FTP_REQ_REJECT)
		self.request["FEAT"] = (FTP_REQ_REJECT)
		self.request["CLNT"] = (FTP_REQ_REJECT)
		self.request["XPWD"] = (FTP_REQ_REJECT)
		self.request["MACB"] = (FTP_REQ_REJECT)
		self.request["OPTS"] = (FTP_REQ_REJECT)

	def requestStack(self):

		self.session.ftp_stack = None
		try:
			stack_proxy = self.request_stack[self.request_command]
                except:
                        try:
                                stack_proxy = self.request_stack["*"]
                        except:
                                return FtpDataProxy
                
                if type(stack_proxy) == type(()) and stack_proxy[0] != FTP_STK_NONE:
			self.session.ftp_stack = stack_proxy[1]
                        return FtpDataProxy

                return None


class FtpProxy(AbstractFtpProxy):
	"""Class encapsulating a permitting ftp proxy which allows everything, even unknown commands.
	    
	This class encapsulates a permitting Ftp proxy allowing unknown commands
	and unknown answers.
	"""

	def config(self):
		"""Configuration for FtpProxy.
		
		Enables all commands by setting permit_unknown_commands to
		TRUE and adding two wildcard entries to the commands
		and answers hash.

		Arguments
		
		  self -- this instance
		
		"""
		self.request["*"] = (FTP_REQ_ACCEPT)
		self.response["*","421"]    = (FTP_RSP_ABORT, "421 Logoff")
		self.response["*", "*"] = (FTP_RSP_ACCEPT)
		self.permit_unknown_command = TRUE

class FtpProxyAnonRO(AbstractFtpProxy):
	"""Class encapsulating an Anonymous DO FTP proxy which allows minimal set of commands.

	This proxy enables only Anonymous login, with only downloading,
	and strictly checks commands and return codes. For unknown
	commands or replies the session is terminated.
	"""
	
	def pUser(self,command):
		if self.request_parameter == "ftp" or self.request_parameter == "anonymous":
			return FTP_REQ_ACCEPT
		proxyLog(self, FTP_POLICY, 3, "Non-anonymous user rejected; user='%s'", (self.request_parameter,))
		return FTP_REQ_REJECT

	def config(self):
		"""Configuration for FtpProxyAnonRO

		It enables a minimal set of commands for a working Anonymous
		Download-Only FTP proxy, and sets permit_unknown_commands
		to FALSE.
		
		Arguments
		
		  self -- this instance
		
		"""

		AbstractFtpProxy.loadMinimalCommands(self)
		self.request["USER"] = (FTP_REQ_POLICY, self.pUser)

		self.request["*"]    = (FTP_REQ_REJECT)

		AbstractFtpProxy.loadAnswers(self)
		self.response["*","*"] = (FTP_RSP_REJECT)
		self.permit_unknown_command = FALSE

class FtpProxyRO(AbstractFtpProxy):
	"""Class encapsulating a Download-Only FTP proxy which allows minimal set of commands.

	This proxy enables the minimal required protocol elements, and strictly
	checks commands and return codes. For unknown commands or replies
	the session is terminated.
	"""
	
	def config(self):
		"""Configuration for FtpProxyRO

		It enables a minimal set of commands for a working
		Download-Only FTP proxy, and sets permit_unknown_commands
		to FALSE.
		
		Arguments
		
		  self -- this instance
		
		"""

		AbstractFtpProxy.loadMinimalCommands(self)

		self.request["USER"] = (FTP_REQ_ACCEPT)
		self.request["*"]    = (FTP_REQ_REJECT)

		AbstractFtpProxy.loadAnswers(self)
		self.response["*","*"] = (FTP_RSP_REJECT)
		self.permit_unknown_command = FALSE

class FtpProxyAnonRW(AbstractFtpProxy):
	"""Class encapsulating an Anonymous RW FTP proxy which allows minimal set of commands.

	This proxy enables only Anonymous login
	and strictly checks commands and return codes. For unknown
	commands or replies the session is terminated.
	"""
	
	def pUser(self,command):
		if self.request_parameter == "ftp" or self.request_parameter == "anonymous":
			return FTP_REQ_ACCEPT
		proxyLog(self, FTP_POLICY, 3, "Non-anonymous user rejected; user='%s'", (self.request_parameter,))
		return FTP_REQ_REJECT

	def config(self):
		"""Configuration for FtpProxyAnonRO

		It enables a minimal set of commands for a working Anonymous
		FTP proxy, and sets permit_unknown_commands to FALSE.
		
		Arguments
		
		  self -- this instance
		
		"""
		AbstractFtpProxy.loadMinimalCommands(self)

		self.request["APPE"] = (FTP_REQ_ACCEPT)
		self.request["DELE"] = (FTP_REQ_ACCEPT)
		self.request["MKD"]  = (FTP_REQ_ACCEPT)
		self.request["RMD"]  = (FTP_REQ_ACCEPT)
		self.request["RNFR"] = (FTP_REQ_ACCEPT)
		self.request["RNTO"] = (FTP_REQ_ACCEPT)
		self.request["STOR"] = (FTP_REQ_ACCEPT)
		self.request["STOU"] = (FTP_REQ_ACCEPT)

		self.request["USER"] = (FTP_REQ_POLICY, self.pUser)

		self.request["*"]    = (FTP_REQ_REJECT)

		AbstractFtpProxy.loadAnswers(self)
		self.response["*","*"] = (FTP_RSP_REJECT)
		self.permit_unknown_command = FALSE

class FtpProxyRW(AbstractFtpProxy):
	"""Class encapsulating a FTP proxy which allows minimal set of commands.

	This proxy enables the minimal required protocol elements, and strictly
	checks commands and return codes. For unknown commands or replies
	the session is terminated.
	"""
	
	def config(self):
		"""Configuration for FtpProxyRW

		It enables a minimal set of commands for a working FTP
		proxy, and sets permit_unknown_commands to FALSE.
		
		Arguments
		
		  self -- this instance
		
		"""
		AbstractFtpProxy.loadMinimalCommands(self)

		self.request["APPE"] = (FTP_REQ_ACCEPT)
		self.request["DELE"] = (FTP_REQ_ACCEPT)
		self.request["MKD"]  = (FTP_REQ_ACCEPT)
		self.request["RMD"]  = (FTP_REQ_ACCEPT)
		self.request["RNFR"] = (FTP_REQ_ACCEPT)
		self.request["RNTO"] = (FTP_REQ_ACCEPT)
		self.request["STOR"] = (FTP_REQ_ACCEPT)
		self.request["STOU"] = (FTP_REQ_ACCEPT)
		self.request["USER"] = (FTP_REQ_ACCEPT)
		self.request["ALLO"] = (FTP_REQ_ACCEPT)

		self.request["*"]    = (FTP_REQ_REJECT)

		AbstractFtpProxy.loadAnswers(self)
		self.response["*","*"] = (FTP_RSP_REJECT)
		self.permit_unknown_command = FALSE

class FtpProxyMinimal(FtpProxyRO):
	"""Alias FtpProxyRO
	"""
	pass
