/**********************************************************************
zyGrib: meteorological GRIB file viewer
Copyright (C) 2008 - Jacques Zaninetti - http://www.zygrib.org

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
***********************************************************************/

#include <cmath>

#include <QApplication>
#include <QPushButton>
#include <QGridLayout>
#include <QStatusBar>
#include <QToolBar>
#include <QMenu>
#include <QFileDialog>
#include <QMessageBox>
#include <QComboBox>

#include "MainWindow.h"
#include "MeteoTable.h"
#include "POI_Editor.h"
#include "Util.h"
#include "Orthodromie.h"
#include "Settings.h"
#include "Version.h"
#include "GUI/AngleConverterDialog.h"


//-----------------------------------------------------------
void MainWindow::InitActionsStatus()
{
    menuBar->acMap_Rivers->setChecked(Util::getSetting("showRivers", false).toBool());
    menuBar->acMap_LonLatGrid->setChecked(Util::getSetting("showLonLatGrid", true).toBool());
    menuBar->acMap_CountriesBorders->setChecked(Util::getSetting("showCountriesBorders", true).toBool());
    menuBar->acMap_Orthodromie->setChecked(Util::getSetting("showOrthodromie", false).toBool());
    
    int mode = Util::getSetting("colorMapMode", MapDrawer::drawWind).toInt();
    terre->setColorMapMode( mode );
    setMenubarColorMapMode( mode );
    
    menuBar->acView_ColorMapSmooth->setChecked(Util::getSetting("colorMapSmooth", true).toBool());
    menuBar->acView_WindArrow->setChecked(Util::getSetting("showWindArrows", true).toBool());
    menuBar->acView_Barbules->setChecked(Util::getSetting("showBarbules", true).toBool());
    
    menuBar->setIsobarsStep(Util::getSetting("isobarsStep", 2).toInt());
    terre->setIsobarsStep(Util::getSetting("isobarsStep", 2).toInt());
    menuBar->acView_Isobars->setChecked(Util::getSetting("showIsobars", true).toBool());
    menuBar->acView_IsobarsLabels->setChecked(Util::getSetting("showIsobarsLabels", false).toBool());
    menuBar->acView_PressureMinMax->setChecked(Util::getSetting("showPressureMinMax", false).toBool());
    
    menuBar->setIsotherms0Step(Util::getSetting("isotherms0Step", 100).toInt());
    terre->setIsotherms0Step(Util::getSetting("isotherms0Step", 100).toInt());
    menuBar->acView_Isotherms0->setChecked(Util::getSetting("showIsotherms0", false).toBool());
    menuBar->acView_Isotherms0Labels->setChecked(Util::getSetting("showIsotherms0Labels", false).toBool());
    
    menuBar->acView_GribGrid->setChecked(Util::getSetting("showGribGrid", false).toBool());
    menuBar->acView_TemperatureLabels->setChecked(Util::getSetting("showTemperatureLabels", false).toBool());
    
    menuBar->acMap_CountriesNames->setChecked(Util::getSetting("showCountriesNames", false).toBool());
    menuBar->setCitiesNamesLevel(Util::getSetting("showCitiesNamesLevel", 0).toInt());
    terre->setCitiesNamesLevel(Util::getSetting("showCitiesNamesLevel", 0).toInt());
    
    menuBar->acMap_ShowPOIs->setChecked(Util::getSetting("showPOIs", true).toBool());
    terre->setShowPOIs(Util::getSetting("showPOIs", true).toBool());

    menuBar->acView_DuplicateFirstCumulativeRecord->setChecked(Util::getSetting("duplicateFirstCumulativeRecord", true).toBool());
    menuBar->acView_InterpolateValues->setChecked(Util::getSetting("interpolateValues", true).toBool());
    menuBar->acView_WindArrowsOnGribGrid->setChecked(Util::getSetting("windArrowsOnGribGrid", false).toBool());

    //----------------------------------------------------------------------
    QString lang = Util::getSetting("appLanguage", "none").toString();
    if (lang == "fr")
        menuBar->acOptions_Lang_fr->setChecked(true);
    else if (lang == "en")
        menuBar->acOptions_Lang_en->setChecked(true);
    else if (lang == "de" && menuBar->acOptions_Lang_de != NULL)
        menuBar->acOptions_Lang_de->setChecked(true);
    else if (lang == "ru" && menuBar->acOptions_Lang_ru != NULL)
        menuBar->acOptions_Lang_ru->setChecked(true);
    else if (lang == "pl" && menuBar->acOptions_Lang_pl != NULL)
        menuBar->acOptions_Lang_pl->setChecked(true);
    else if (lang == "pt" && menuBar->acOptions_Lang_pt != NULL)
        menuBar->acOptions_Lang_pt->setChecked(true);
    else if (lang == "es" && menuBar->acOptions_Lang_es != NULL)
        menuBar->acOptions_Lang_es->setChecked(true);
    
    //----------------------------------------------------------------------
	// Set map quality
	int quality = Util::getSetting("gshhsMapQuality", 1).toInt();
    for (int qual=4; qual>=0; qual--) {
        if (! gshhsReader->gshhsFilesExists(qual)) {
            switch (qual) {
                case 0: menuBar->acMap_Quality1->setEnabled(false); break;
                case 1: menuBar->acMap_Quality2->setEnabled(false); break;
                case 2: menuBar->acMap_Quality3->setEnabled(false); break;
                case 3: menuBar->acMap_Quality4->setEnabled(false); break;
                case 4: menuBar->acMap_Quality5->setEnabled(false); break;
            }
            if (quality > qual)
            	quality = qual-1;
        }
    }
    if (quality < 0) {
        QMessageBox::information (this,
            QString(tr("Erreur")),
            QString(tr("Cartes non trouvées.\n\n")
            		+tr("Vérifiez l'installation du programme."))
        );
    	quality = 0;
	}
	menuBar->setQuality(quality);
	terre->setMapQuality(quality);
}


//-----------------------------------------------------------
// Connexions des signaux
//-----------------------------------------------------------
void MainWindow::connectSignals()
{
    MenuBar  *mb = menuBar;
    
    //-------------------------------------
    // Actions
    //-------------------------------------
    connect(mb->ac_OpenMeteotable, SIGNAL(triggered()), this, SLOT(slotOpenMeteotable()));
    connect(mb->ac_CreatePOI, SIGNAL(triggered()), this, SLOT(slotCreatePOI()));
    connect(mb->ac_CreateAnimation, SIGNAL(triggered()), this, SLOT(slotCreateAnimation()));

    connect(mb->acFile_Open, SIGNAL(triggered()), this, SLOT(slotFile_Open()));
    connect(mb->acFile_Close, SIGNAL(triggered()), this, SLOT(slotFile_Close()));
    connect(mb->acFile_Load_GRIB, SIGNAL(triggered()), this, SLOT(slotFile_Load_GRIB()));
    connect(mb->acFile_Load_IAC, SIGNAL(triggered()), this, SLOT(slotFile_Load_IAC()));
    
    connect(mb->acFile_GribServerStatus, SIGNAL(triggered()), this, SLOT(slotFile_GribServerStatus()));
    connect(mb->acFile_Info_GRIB, SIGNAL(triggered()), this, SLOT(slotFile_Info_GRIB()));
    connect(mb->acFile_Quit, SIGNAL(triggered()), this, SLOT(slotFile_Quit()));

    //-------------------------------------------------------
    connect(mb->acMap_GroupQuality, SIGNAL(triggered(QAction *)),
            this, SLOT(slotMap_Quality()));
            
    connect(mb->acMap_GroupProjection, SIGNAL(triggered(QAction *)),
            this, SLOT(slotMap_Projection(QAction *)));
            
    connect(mb->acMap_Rivers, SIGNAL(triggered(bool)),
            terre,  SLOT(setDrawRivers(bool)));
    connect(mb->acMap_LonLatGrid, SIGNAL(triggered(bool)),
            terre,  SLOT(setDrawLonLatGrid(bool)));
    connect(mb->acMap_CountriesBorders, SIGNAL(triggered(bool)),
            terre,  SLOT(setDrawCountriesBorders(bool)));
    connect(mb->acMap_Orthodromie, SIGNAL(triggered(bool)),
            terre,  SLOT(setDrawOrthodromie(bool)));
    connect(mb->acMap_CountriesNames, SIGNAL(triggered(bool)),
            terre,  SLOT(setCountriesNames(bool)));
    connect(mb->acMap_GroupCitiesNames, SIGNAL(triggered(QAction *)),
            this, SLOT(slotMap_CitiesNames()));

    connect(mb->acMap_Zoom_In, SIGNAL(triggered()),
            terre,  SLOT(slot_Zoom_In()));
    connect(mb->acMap_Zoom_Out, SIGNAL(triggered()),
            terre,  SLOT(slot_Zoom_Out()));
    connect(mb->acMap_Zoom_Sel, SIGNAL(triggered()),
            terre,  SLOT(slot_Zoom_Sel()));
    connect(mb->acMap_Zoom_All, SIGNAL(triggered()),
            terre,  SLOT(slot_Zoom_All()));

    connect(mb->acMap_Go_Left, SIGNAL(triggered()),
            terre,  SLOT(slot_Go_Left()));
    connect(mb->acMap_Go_Right, SIGNAL(triggered()),
            terre,  SLOT(slot_Go_Right()));
    connect(mb->acMap_Go_Up, SIGNAL(triggered()),
            terre,  SLOT(slot_Go_Up()));
    connect(mb->acMap_Go_Down, SIGNAL(triggered()),
            terre,  SLOT(slot_Go_Down()));

	connect(mb->acMap_ShowPOIs, SIGNAL(triggered(bool)),
            terre,  SLOT(setShowPOIs(bool)));
            
    //-------------------------------------------------------
	connect(mb->acView_GroupColorMap, SIGNAL(triggered(QAction *)),
		this, SLOT(slot_GroupColorMap(QAction *)));

    connect(mb->acView_ColorMapSmooth, SIGNAL(triggered(bool)),
            terre,  SLOT(setColorMapSmooth(bool)));
            
    connect(mb->acView_DuplicateFirstCumulativeRecord, SIGNAL(triggered(bool)),
            terre,  SLOT(setDuplicateFirstCumulativeRecord(bool)));
    connect(mb->acView_InterpolateValues, SIGNAL(triggered(bool)),
            terre,  SLOT(setInterpolateValues(bool)));
    connect(mb->acView_WindArrowsOnGribGrid, SIGNAL(triggered(bool)),
            terre,  SLOT(setWindArrowsOnGribGrid(bool)));
    
    connect(mb->acView_WindArrow, SIGNAL(triggered(bool)),
            terre,  SLOT(setDrawWindArrows(bool)));
    connect(mb->acView_WindArrow, SIGNAL(triggered(bool)),
            this,  SLOT(slotWindArrows(bool)));    
    connect(mb->acView_Barbules, SIGNAL(triggered(bool)),
            terre,  SLOT(setBarbules(bool)));
    
    connect(mb->acView_TemperatureLabels, SIGNAL(triggered(bool)),
            terre,  SLOT(slotTemperatureLabels(bool)));
    
    connect(mb->acView_Isobars, SIGNAL(triggered(bool)),
            terre,  SLOT(setDrawIsobars(bool)));
    connect(mb->acView_GroupIsobarsStep, SIGNAL(triggered(QAction *)),
            this, SLOT(slotIsobarsStep()));
    connect(mb->acView_IsobarsLabels, SIGNAL(triggered(bool)),
            terre,  SLOT(setDrawIsobarsLabels(bool)));
    connect(mb->acView_PressureMinMax, SIGNAL(triggered(bool)),
            terre,  SLOT(setPressureMinMax(bool)));
    
    connect(mb->acView_Isotherms0, SIGNAL(triggered(bool)),
            terre,  SLOT(setDrawIsotherms0(bool)));
    connect(mb->acView_GroupIsotherms0Step, SIGNAL(triggered(QAction *)),
            this, SLOT(slotIsotherms0Step()));
    connect(mb->acView_Isotherms0Labels, SIGNAL(triggered(bool)),
            terre,  SLOT(setDrawIsotherms0Labels(bool)));
    
    connect(mb->acView_GribGrid, SIGNAL(triggered(bool)),
            terre,  SLOT(setGribGrid(bool)));
    
    //-------------------------------------------------------
    connect(mb->acOptions_Units, SIGNAL(triggered()), &dialogUnits, SLOT(exec()));
    connect(&dialogUnits, SIGNAL(accepted()), terre, SLOT(slotMustRedraw()));
    connect(&dialogUnits, SIGNAL(signalTimeZoneChanged()), this, SLOT(slotTimeZoneChanged()));
    
    connect(mb->acOptions_GraphicsParams, SIGNAL(triggered()), &dialogGraphicsParams, SLOT(exec()));
    connect(&dialogGraphicsParams, SIGNAL(accepted()), terre, SLOT(updateGraphicsParameters()));
    
    connect(mb->acOptions_Proxy, SIGNAL(triggered()), &dialogProxy, SLOT(exec()));
    connect(mb->acOptions_AngleConverter, SIGNAL(triggered()), this, SLOT(slotOpenAngleConverter()));

    connect(mb->acOptions_GroupLanguage, SIGNAL(triggered(QAction *)),
            this, SLOT(slotOptions_Language()));
    
    //-------------------------------------------------------
    connect(mb->acHelp_Help, SIGNAL(triggered()), this, SLOT(slotHelp_Help()));
    connect(mb->acHelp_APropos, SIGNAL(triggered()), this, SLOT(slotHelp_APropos()));
    connect(mb->acHelp_AProposQT, SIGNAL(triggered()), this, SLOT(slotHelp_AProposQT()));
    
    //-------------------------------------
    // Autres objets de l'interface
    //-------------------------------------
    connect(mb->cbDatesGrib, SIGNAL(activated(int)),
            this, SLOT(slotDateGribChanged(int)));
    connect(mb->acDatesGrib_next, SIGNAL(triggered()),
            this, SLOT(slotDateGribChanged_next()));
    connect(mb->acDatesGrib_prev, SIGNAL(triggered()),
            this, SLOT(slotDateGribChanged_prev()));
    
    
    //-------------------------------------
    // Autres signaux
    //-------------------------------------
    connect(this, SIGNAL(signalMapQuality(int)),
            terre,  SLOT(setMapQuality(int)));    
    //-----------------------------------------------------------
    connect(terre, SIGNAL(mouseClicked(QMouseEvent *)),
            this,  SLOT(slotMouseClicked(QMouseEvent *)));
    connect(terre, SIGNAL(mouseMoved(QMouseEvent *)),
            this,  SLOT(slotMouseMoved(QMouseEvent *)));
    //-----------------------------------------------------------
    connect(&dialogLoadGRIB, SIGNAL(signalMeteoFileReceived(QString)),
            this,  SLOT(slotGribFileReceived(QString)));
    connect(&dialogLoadIAC, SIGNAL(signalMeteoFileReceived(QString)),
            this,  SLOT(slotGribFileReceived(QString)));
}

//----------------------------------------------------
void MainWindow::slotGribFileReceived(QString fileName)
{
    openMeteoDataFile(fileName, true);
}

//=============================================================
MainWindow::MainWindow(int w, int h, QString startFileName, QWidget *parent)
    : QMainWindow(parent)
{
    setWindowTitle("zyGrib");
    gribFileName = "";

    //--------------------------------------------------
    resize( Util::getSetting("mainWindowSize", QSize(w,h)).toSize() );
    move  ( Util::getSetting("mainWindowPos", QPoint()).toPoint() );

    menuBar = new MenuBar(this);
    assert(menuBar);
    
    //--------------------------------------------------
    int mapQuality = 0;
    gshhsReader = new GshhsReader("maps/gshhs", mapQuality);
    assert(gshhsReader);
    
    //---------------------------------------------------
	// Projection
	proj = NULL;
    initProjection();
    
    //--------------------------------------------------
    terre = new Terrain(this, proj, gshhsReader);
    assert(terre);
    
    statusBar =new QStatusBar(this);
    QFontInfo finfo = statusBar->fontInfo();
    QFont font("", finfo.pointSize(), QFont::Normal, false);
    font.setStyleHint(QFont::TypeWriter);
    font.setStretch(QFont::SemiCondensed);
    statusBar->setFont(font);
    
    boardToolBar = new BoardToolBar(this);
    this->addToolBar(Qt::LeftToolBarArea, boardToolBar);
    
    //--------------------------------------------------
    toolBar = addToolBar(tr("Outils"));
    toolBar->setFloatable(false);
    toolBar->setMovable(false);
    toolBar->addAction(menuBar->acFile_Quit);
    toolBar->addSeparator();
    toolBar->addAction(menuBar->acFile_Open);
    toolBar->addSeparator();
    toolBar->addWidget(menuBar->cbDatesGrib);
    toolBar->addAction(menuBar->acDatesGrib_prev);
    toolBar->addAction(menuBar->acDatesGrib_next);
    toolBar->addSeparator();
    toolBar->addAction(menuBar->acMap_Zoom_In);
    toolBar->addAction(menuBar->acMap_Zoom_Out);
    toolBar->addAction(menuBar->acMap_Zoom_Sel);
    toolBar->addAction(menuBar->acMap_Zoom_All);
    toolBar->addSeparator();
    toolBar->addAction(menuBar->acMap_Go_Left);
    toolBar->addAction(menuBar->acMap_Go_Right);
    toolBar->addAction(menuBar->acMap_Go_Up);
    toolBar->addAction(menuBar->acMap_Go_Down);
    toolBar->addSeparator();
    toolBar->addAction(menuBar->acFile_Load_GRIB);
    toolBar->addAction(menuBar->acFile_GribServerStatus);
    toolBar->addAction(menuBar->acFile_Info_GRIB);
    toolBar->addSeparator();
    toolBar->addAction(menuBar->ac_CreateAnimation);
    toolBar->addSeparator();
    
    //--------------------------------------------------
    setMenuBar(menuBar);
    setStatusBar(statusBar);
    setCentralWidget(terre);
    
    //--------------------------------------------------
    gribFilePath = Util::getSetting("gribFilePath", "").toString();
    if (startFileName == "")
    {	// load previous grib file at previous scale
		QString fname = Util::getSetting("gribFileName", "").toString();
		if (fname != "" && QFile::exists(fname))
		{
			openMeteoDataFile(fname, false);
		}
	}
	else {
		// load command line grib file			
		// openGribFile(startFileName, true);

		// Not open file here, but after show() call (see main.cpp)
	}
		
		
    //---------------------------------------------------------
	// Menu popup : bouton droit de la souris
    //---------------------------------------------------------
    menuPopupBtRight = menuBar->createPopupBtRight(this);

    //---------------------------------------------------------
	// Active les actions
    //---------------------------------------------------------
    connectSignals();
    InitActionsStatus();

    //---------------------------------------------------------
    // Initialisation des POI's
    //---------------------------------------------------------
	POI *poi;
	bool visible = Util::getSetting("showPOIs", true).toBool();
    QList<uint> lscodes = Settings::getSettingAllCodesPOIs();
	for (int i=0; i < lscodes.size(); ++i)
	{
		uint code = lscodes.at(i);
		//printf("found poi: %d : \n", code);
 		poi = new POI(code, proj, this, terre);
    	if (poi!=NULL) {
    		if (poi->isValid()) {
    			poi->setVisible(visible);
			}
			else {
				Settings::deleteSettingsPOI(code);
				delete poi;
			}
		}
 	}
}

//-----------------------------------------------
MainWindow::~MainWindow()
{
    //--------------------------------------------------
    // Save global settings
    //--------------------------------------------------
    Util::setSetting("mainWindowSize", size());
    Util::setSetting("mainWindowPos", pos());
    
    Util::setSetting("gribFileName",  gribFileName);
    Util::setSetting("gribFilePath",  gribFilePath);
    
    Util::setSetting("projectionCX", proj->getCX());
    Util::setSetting("projectionCY", proj->getCY());
    Util::setSetting("projectionScale",  proj->getScale());
}
    
//---------------------------------------------------------------
void MainWindow::initProjection()
{
	if (proj != NULL) {
		delete proj;
		proj = NULL;
	}
    int idproj = Util::getSetting("projectionId", Projection::PROJ_ZYGRIB).toInt();
    double cx = (double) Util::getSetting("projectionCX", 0.0).toDouble();
    double cy = (double) Util::getSetting("projectionCY", 0.0).toDouble();
    double scale = (double) Util::getSetting("projectionScale", 0.5).toDouble();
    
    switch (idproj)
    {
    	case Projection::PROJ_EQU_CYL :
			proj = new Projection_EQU_CYL (width(), height(), cx, cy, scale);
			break;
    	case Projection::PROJ_CENTRAL_CYL :
			proj = new Projection_CENTRAL_CYL (width(), height(), cx, cy, scale);
			break;
    	case Projection::PROJ_MERCATOR :
			proj = new Projection_MERCATOR (width(), height(), cx, cy, scale);
			break;
    	case Projection::PROJ_MILLER :
			proj = new Projection_MILLER (width(), height(), cx, cy, scale);
			break;
    	
    	case Projection::PROJ_ZYGRIB :
    	default :
			proj = new Projection_ZYGRIB (width(), height(), cx, cy, scale);
	}
	assert(proj);
	menuBar->setProjection(idproj);
}
//-------------------------------------------------
void MainWindow::slotMap_Projection(QAction *act)
{
	int idproj = Projection::PROJ_ZYGRIB;
    MenuBar  *mb = menuBar;
    if (act == mb->acMap_PROJ_ZYGRIB)
        idproj = Projection::PROJ_ZYGRIB;
    else if (act == mb->acMap_PROJ_MERCATOR)
        idproj = Projection::PROJ_MERCATOR;
    else if (act == mb->acMap_PROJ_MILLER)
        idproj = Projection::PROJ_MILLER;
    else if (act == mb->acMap_PROJ_CENTRAL_CYL)
        idproj = Projection::PROJ_CENTRAL_CYL;
    else if (act == mb->acMap_PROJ_EQU_CYL)
        idproj = Projection::PROJ_EQU_CYL;
	
	double x,y;    // current position
	proj->screen2map(proj->getW()/2,proj->getH()/2, &x, &y);
        
	Util::setSetting("projectionCX", proj->getCX());
	Util::setSetting("projectionCY", proj->getCY());
	Util::setSetting("projectionScale",  proj->getScale());
	Util::setSetting("projectionId", idproj);

	initProjection();
	proj->setMapPointInScreen(x, y, proj->getW()/2, proj->getH()/2);
	
	terre->setProjection(proj);
}

//-------------------------------------------------
void MainWindow::slotTimeZoneChanged()
{
    if (terre->getGribPlot()->isGribReaderOk())
    {
        menuBar->updateListeDates(
        			terre->getGribPlot()->getListDates(),
        			terre->getGribPlot()->getCurrentDate());
    }
}

//-------------------------------------------------
void MainWindow::openMeteoDataFile(QString fileName, bool zoomOnGrib)
{
//printf("Opening file: %s\n", qPrintable(fileName));
	if (QFile::exists(fileName))
	{
		terre->loadGribFile(fileName, zoomOnGrib);
		if (terre->getGribPlot()->isGribReaderOk())
		{
			//printf("Found GRIB file \n");
			setWindowTitle(tr("zyGrib - ")+ QFileInfo(fileName).fileName());
			menuBar->updateListeDates(
						terre->getGribPlot()->getListDates(),
						terre->getGribPlot()->getCurrentDate());
			slotDateGribChanged(0);
			gribFileName = fileName;
			
			menuBar->menuGrib->setEnabled(true);
			menuBar->acDatesGrib_prev->setEnabled(true);
			menuBar->acDatesGrib_next->setEnabled(true);
		
			bool ok;
			GribReader *grib = terre->getGribPlot()->getGribReader();
			dataPresentInGrib(grib,GRB_TEMP,LV_ABOV_GND,2,&ok);
			menuBar->acView_TempColors->setEnabled(ok);
			menuBar->acView_TemperatureLabels->setEnabled(ok);
			
			dataPresentInGrib(grib,GRB_TPOT,LV_SIGMA,9950,&ok);
			menuBar->acView_TempPotColors->setEnabled(ok);
			
			dataPresentInGrib(grib,GRB_SNOW_CATEG,LV_GND_SURF,0,&ok);
			menuBar->acView_SnowCateg->setEnabled(ok);
			dataPresentInGrib(grib,GRB_SNOW_DEPTH,LV_GND_SURF,0,&ok);
			menuBar->acView_SnowDepth->setEnabled(ok);
			
			dataPresentInGrib(grib,GRB_FRZRAIN_CATEG,LV_GND_SURF,0,&ok);
			menuBar->acView_FrzRainCateg->setEnabled(ok);
			
			dataPresentInGrib(grib,GRB_CAPE,LV_GND_SURF,0,&ok);
			menuBar->acView_CAPEsfc->setEnabled(ok);
			
			dataPresentInGrib(grib,GRB_WIND_VX,LV_ABOV_GND,10,&ok);
			menuBar->acView_WindColors->setEnabled(ok);
			menuBar->acView_WindArrow->setEnabled(ok);
			menuBar->acView_Barbules->setEnabled(ok);
			
			dataPresentInGrib(grib,GRB_PRECIP_TOT,LV_GND_SURF,0,&ok);
			menuBar->acView_RainColors->setEnabled(ok);
			dataPresentInGrib(grib,GRB_CLOUD_TOT,LV_ATMOS_ALL,0,&ok);
			menuBar->acView_CloudColors->setEnabled(ok);
			dataPresentInGrib(grib,GRB_HUMID_REL,LV_ABOV_GND,2,&ok);
			menuBar->acView_HumidColors->setEnabled(ok);
			dataPresentInGrib(grib,GRB_DEWPOINT,LV_ABOV_GND,2,&ok);
			//menuBar->acView_DewpointColors->setEnabled(ok);
			menuBar->acView_DeltaDewpointColors->setEnabled(ok);

			dataPresentInGrib(grib,GRB_PRESSURE,LV_MSL,0,&ok);
			menuBar->acView_Isobars->setEnabled(ok);
			menuBar->acView_IsobarsLabels->setEnabled(ok);
			menuBar->acView_PressureMinMax->setEnabled(ok);
			menuBar->menuIsobarsStep->setEnabled(ok);
			
			dataPresentInGrib(grib,GRB_GEOPOT_HGT,LV_ISOTHERM0,0,&ok);
			menuBar->acView_Isotherms0->setEnabled(ok);
			menuBar->acView_Isotherms0Labels->setEnabled(ok);
			menuBar->acView_GroupIsotherms0Step->setEnabled(ok);
			menuBar->menuIsotherms0Step->setEnabled(ok);

		}
		else {
			terre->loadIacFile(fileName, zoomOnGrib);
			if (terre->getIacPlot()->isIacReaderOk())
			{
				//printf("Found IAC file \n");
				std::set<time_t> setDatesEmpty;
				setWindowTitle(tr("zyGrib - IAC - ")+ QFileInfo(fileName).fileName());
				gribFileName = fileName;

				menuBar->updateListeDates(&setDatesEmpty, 0);				
				menuBar->menuGrib->setEnabled(false);
				menuBar->acDatesGrib_prev->setEnabled(false);
				menuBar->acDatesGrib_next->setEnabled(false);
			}
			else
			{
				QMessageBox::critical (this,
					tr("Erreur"),
					tr("Fichier : ") + fileName + "\n\n"
						+ tr("Echec lors de l'ouverture.") + "\n\n"
						+ tr("Le fichier ne peut pas être ouvert,") + "\n"
						+ tr("ou ce n'est pas un fichier GRIB,") + "\n"
						+ tr("ou le fichier est corrompu,") + "\n"
						+ tr("ou il contient des données non reconnues,") + "\n"
						+ tr("ou...")
				);
			}
		}	
	}	
}

//-------------------------------------------------
void MainWindow::slotCreateAnimation()
{
	if (terre->getGribPlot()==NULL  ||
			! terre->getGribPlot()->isGribReaderOk())
	{
        QMessageBox::critical (this,
            tr("Erreur"),
			tr("Création de l'animation impossible :") + "\n"
                + tr("pas de fichier GRIB chargé.")
        );
	}
	else
	{		
		new GribAnimator(
				terre->getGribPlot(),
				terre->getDrawer(),
				terre->getProjection(),
				terre->getListPOIs()
			);
	}
}
//-------------------------------------------------
void MainWindow::slotPOImoved(POI* poi)
{
    QString message =
    		poi->getName()+" : "
            + Util::formatPosition(poi->getLongitude(),poi->getLatitude());
    statusBar->showMessage(message);
}
//-------------------------------------------------
void MainWindow::slotCreatePOI()
{
	double lon, lat;
	proj->screen2map(mouseClicX,mouseClicY, &lon, &lat);
	new POI_Editor(Settings::getNewCodePOI(), lon, lat, proj, this, terre);
}
//-------------------------------------------------
void MainWindow::slotOpenMeteotablePOI(POI* poi)
{
	new MeteoTableDialog(terre->getGribPlot(),
				poi->getLongitude(), poi->getLatitude(), poi->getName());
}

//-------------------------------------------------
void MainWindow::slotOpenMeteotable()
{
	double lon, lat;
	proj->screen2map(mouseClicX,mouseClicY, &lon, &lat);
	new MeteoTableDialog(terre->getGribPlot(), lon, lat);
}

//-------------------------------------------------
void MainWindow::slotOpenAngleConverter()
{
	new AngleConverterDialog(this);
}
//-------------------------------------------------
void MainWindow::slotOptions_Language()
{
    QString lang;
    MenuBar  *mb = menuBar;
    QAction *act = mb->acOptions_GroupLanguage->checkedAction();
    if (act == mb->acOptions_Lang_fr) {
        lang = "fr";
        Util::setSetting("appLanguage", lang);
        QMessageBox::information (this,
            QString("Changement de langue"),
            QString("Langue : Français\n\n")
              + QString("Les modifications seront prises en compte\n")
              + QString("au prochain lancement du programme.")
        );
    }
    else if (act == mb->acOptions_Lang_en) {
        lang = "en";
        Util::setSetting("appLanguage", lang);
        QMessageBox::information (this,
            QString("Application Language"),
            QString("Language : English\n\n")
              + QString("Please reload application to activate language.\n")
        );
    }
    else if (act == mb->acOptions_Lang_de) {
        lang = "de";
        Util::setSetting("appLanguage", lang);
        QMessageBox::information (this,
            QString("Application Language"),
            QString("Language : German\n\n")
              + QString("Please reload application to activate language.\n")
        );
    }
    else if (act == mb->acOptions_Lang_ru) {
        lang = "ru";
        Util::setSetting("appLanguage", lang);
        QMessageBox::information (this,
            QString("Application Language"),
            QString("Language : Russian\n\n")
              + QString("Please reload application to activate language.\n")
        );
    }
    else if (act == mb->acOptions_Lang_pl) {
        lang = "pl";
        Util::setSetting("appLanguage", lang);
        QMessageBox::information (this,
            QString("Application Language"),
            QString("Language : Polish\n\n")
              + QString("Please reload application to activate language.\n")
        );
    }
    else if (act == mb->acOptions_Lang_pt) {
        lang = "pt";
        Util::setSetting("appLanguage", lang);
        QMessageBox::information (this,
            QString("Application Language"),
            QString("Language : Portuguese\n\n")
              + QString("Please reload application to activate language.\n")
        );
    }
    else if (act == mb->acOptions_Lang_es) {
        lang = "es";
        Util::setSetting("appLanguage", lang);
        QMessageBox::information (this,
            QString("Application Language"),
            QString("Language : Spanish\n\n")
              + QString("Please reload application to activate language.\n")
        );
    }
    else if (act == mb->acOptions_Lang_it) {
        lang = "it";
        Util::setSetting("appLanguage", lang);
        QMessageBox::information (this,
            QString("Application Language"),
            QString("Language : Italian\n\n")
              + QString("Please reload application to activate language.\n")
        );
    }
}
//-------------------------------------------------
void MainWindow::slotMap_Quality()
{
	int quality = 0;
    MenuBar  *mb = menuBar;
    QAction *act = mb->acMap_GroupQuality->checkedAction();
    if (act == mb->acMap_Quality1)
        quality = 0;
    else if (act == mb->acMap_Quality2)
        quality = 1;
    else if (act == mb->acMap_Quality3)
        quality = 2;
    else if (act == mb->acMap_Quality4)
        quality = 3;
    else if (act == mb->acMap_Quality5)
        quality = 4;

	Util::setSetting("gshhsMapQuality", quality);
	emit signalMapQuality(quality);
}

//-------------------------------------------------
void MainWindow::slotMap_CitiesNames()
{
    MenuBar  *mb = menuBar;
    QAction *act = mb->acMap_GroupCitiesNames->checkedAction();
    if (act == mb->acMap_CitiesNames0)
        terre->setCitiesNamesLevel(0);
    else if (act == mb->acMap_CitiesNames1)
        terre->setCitiesNamesLevel(1);
    else if (act == mb->acMap_CitiesNames2)
        terre->setCitiesNamesLevel(2);
    else if (act == mb->acMap_CitiesNames3)
        terre->setCitiesNamesLevel(3);
    else if (act == mb->acMap_CitiesNames4)
        terre->setCitiesNamesLevel(4);
}
//-------------------------------------------------
void MainWindow::slotIsobarsStep()
{
    int s = 4;
    MenuBar  *mb = menuBar;    
    QAction *act = mb->acView_GroupIsobarsStep->checkedAction();
    if (act == mb->acView_IsobarsStep1)
        s = 1;
    else if (act == mb->acView_IsobarsStep2)
        s = 2;
    else if (act == mb->acView_IsobarsStep3)
        s = 3;
    else if (act == mb->acView_IsobarsStep4)
        s = 4;
    else if (act == mb->acView_IsobarsStep5)
        s = 5;
    else if (act == mb->acView_IsobarsStep6)
        s = 6;
    else if (act == mb->acView_IsobarsStep8)
        s = 8;
    else if (act == mb->acView_IsobarsStep10)
        s = 10;
    terre->setIsobarsStep(s);
}
//-------------------------------------------------
void MainWindow::slotIsotherms0Step()
{
    int s = 100;
    MenuBar  *mb = menuBar;    
    QAction *act = mb->acView_GroupIsotherms0Step->checkedAction();
    if (act == mb->acView_Isotherms0Step10)
        s = 10;
    else if (act == mb->acView_Isotherms0Step20)
        s = 20;
    else if (act == mb->acView_Isotherms0Step50)
        s = 50;
    else if (act == mb->acView_Isotherms0Step100)
        s = 100;
    else if (act == mb->acView_Isotherms0Step200)
        s = 200;
    else if (act == mb->acView_Isotherms0Step500)
        s = 500;
    else if (act == mb->acView_Isotherms0Step1000)
        s = 1000;
    terre->setIsotherms0Step(s);
}

//-------------------------------------------------
void MainWindow::slotHelp_Help() {
    QMessageBox::question (this,
            tr("Aide"),
            tr("Des questions ?"));
            
    QString coderand = QDateTime::currentDateTime().toString("yyyymmsszzz");
    
    QMessageBox::information (this,
            tr("Aide"),
            tr("Félicitations, votre demande a bien été enregistrée.\n")
            +tr("Référence : ") + coderand
            +tr("\n\n")
            +tr("Nos services vous contacteront peut-être quand ils seront en mesure de vous répondre.")
            +tr("\n\n")
            +tr("En attendant, essayez donc d'appuyer au hasard sur des touches du clavier, ")
            +tr("ou bien de bouger la souris en appuyant de temps en temps ")
            +tr("sur l'un de ses boutons, ")
            +tr("ou bien n'importe quoi d'autre, ")
            +tr("et vous verrez bien s'il se passe quelque chose...")
            );
}
//-------------------------------------------------
void MainWindow::slotHelp_APropos()
{
    QMessageBox::information (this,
            tr("A propos"),
            tr("zyGrib : visualisation de fichiers GRIB")
            +"\n"+
            tr("Version : ")+Version::getVersion()
                    +"      "+Version::getDate()
            +"\n"+ tr("Licence : GNU GPL v3")
            +"\n"+ tr("http://www.zygrib.org")
        );
}
//-------------------------------------------------
void MainWindow::slotHelp_AProposQT() {
	QMessageBox::aboutQt (this);
}

//-------------------------------------------------
void MainWindow::slotFile_Quit() {
    QApplication::quit();
}
//-------------------------------------------------
void MainWindow::slotFile_Open()
{
    QString filter;
/*    filter =  tr("Fichiers GRIB (*.grb *.grib *.grb.bz2 *.grib.bz2 *.grb.gz *.grib.gz)")
            + tr(";;Autres fichiers (*)");*/
	filter="";    
    QString fileName = QFileDialog::getOpenFileName(this,
                         tr("Choisir un fichier GRIB"),
                         gribFilePath,
                         filter);
    
    if (fileName != "")
    {
        QFileInfo finfo(fileName);
        gribFilePath = finfo.absolutePath();
    	Util::setSetting("gribFilePath",  gribFilePath);
        openMeteoDataFile(fileName, true);
    }
}

//-------------------------------------------------
void MainWindow::slotFile_Close()
{
    gribFileName = "";
    terre->loadGribFile("", false);
}

//========================================================================
void MainWindow::slotFile_Load_IAC()
{
	dialogLoadIAC.exec();
}

//========================================================================
void MainWindow::slotFile_Load_GRIB()
{
    double x0, y0, x1, y1;
    if ( terre->getSelectedRectangle (&x0,&y0, &x1,&y1) )
    {
        dialogLoadGRIB.setZone(x0, y0, x1, y1);
        dialogLoadGRIB.exec();
    }
    else if ( terre->getGribFileRectangle (&x0,&y0, &x1,&y1) )
    {
        dialogLoadGRIB.setZone(x0, y0, x1, y1);
        dialogLoadGRIB.exec();
    }
    else {
        QMessageBox::warning (this,
            tr("Téléchargement d'un fichier GRIB"),
            tr("Vous devez sélectionner une zone de la carte."));
    }
}

//-----------------------------------------------
void MainWindow::slotFile_GribServerStatus()
{
    DialogServerStatus dial;
    dial.exec();
}

//-----------------------------------------------
QString MainWindow::dataPresentInGrib(GribReader* grib,
				int dataType,int levelType,int levelValue,
				bool *ok)
{
	if (dataType == GRB_DEWPOINT) {
		switch (grib->getDewpointDataStatus(levelType,levelValue)) {
			case GribReader::DATA_IN_FILE :
				if (ok != NULL) *ok = true;
				return tr("oui");
				break;
			case GribReader::NO_DATA_IN_FILE :
				if (ok != NULL) *ok = false;
				return tr("non");
				break;
			case GribReader::COMPUTED_DATA :
			default :
				if (ok != NULL) *ok = true;
				return tr("non (calculé par la formule de Magnus-Tetens)");
				break;
		}
	}
	else {
		if (grib->getNumberOfGribRecords(dataType,levelType,levelValue) > 0) {
			if (ok != NULL) *ok = true;
			return tr("oui");
		}
		else {
			if (ok != NULL) *ok = false;
			return tr("non");
		}
	}
}       

//-----------------------------------------------
void MainWindow::slotFile_Info_GRIB()
{    
    if (! terre->getGribPlot()->isGribReaderOk())    {
        QMessageBox::information (this,
            tr("Informations sur le fichier GRIB"),
            tr("Aucun fichir GRIB n'est chargé."));
		return;
    }
	GribReader * grib = terre->getGribPlot()->getGribReader();
    if (grib == NULL)    {
        QMessageBox::information (this,
            tr("Informations sur le fichier GRIB"),
            tr("Erreur de données."));
		return;
    }	
	GribRecord * gr = grib->getFirstGribRecord();
    if (gr == NULL)    {
        QMessageBox::information (this,
            tr("Informations sur le fichier GRIB"),
            tr("Données non reconnues."));
		return;
    }
    
    QString msg;
	msg += tr("Fichier : %1\n") .arg(grib->getFileName().c_str());
	msg += tr("Taille : %1 octets\n") .arg(grib->getFileSize());
	msg += tr("\n");
	msg += tr("Centre météorologique : %1\n") .arg(gr->getIdCenter());
	msg += tr("Modèle : %1\n") .arg(gr->getIdModel());
	msg += tr("Grille : %1\n") .arg(gr->getIdGrid());
	msg += tr("\n");
	
	msg += tr("%1 enregistrements, ").arg(grib->getTotalNumberOfGribRecords());
	msg += tr("%1 dates :\n").arg(grib->getNumberOfDates());

	std::set<time_t> sdates = grib->getListDates();
	msg += tr("    du %1\n").arg( Util::formatDateTimeLong(*(sdates.begin())) );
	msg += tr("    au %1\n").arg( Util::formatDateTimeLong(*(sdates.rbegin())) );

	msg += tr("\n");
	msg += tr("Données disponibles :\n");
	msg += tr("    Température : %1\n").arg(dataPresentInGrib(grib,GRB_TEMP,LV_ABOV_GND,2));
	msg += tr("    Pression : %1\n").arg(dataPresentInGrib(grib,GRB_PRESSURE,LV_MSL,0));
	msg += tr("    Vent  : %1\n").arg(dataPresentInGrib(grib,GRB_WIND_VX,LV_ABOV_GND,10));
	msg += tr("    Cumul de précipitations : %1\n").arg(dataPresentInGrib(grib,GRB_PRECIP_TOT,LV_GND_SURF,0));
	msg += tr("    Nébulosité : %1\n").arg(dataPresentInGrib(grib,GRB_CLOUD_TOT,LV_ATMOS_ALL,0));
	msg += tr("    Humidité relative : %1\n").arg(dataPresentInGrib(grib,GRB_HUMID_REL,LV_ABOV_GND,2));
	msg += tr("    Isotherme 0°C : %1\n").arg(dataPresentInGrib(grib,GRB_GEOPOT_HGT,LV_ISOTHERM0,0));
	msg += tr("    Point de rosée : %1\n").arg(dataPresentInGrib(grib,GRB_DEWPOINT,LV_ABOV_GND,2));
	msg += tr("    Température (min) : %1\n").arg(dataPresentInGrib(grib,GRB_TMIN,LV_ABOV_GND,2));
	msg += tr("    Température (max) : %1\n").arg(dataPresentInGrib(grib,GRB_TMAX,LV_ABOV_GND,2));
	msg += tr("    Température (pot) : %1\n").arg(dataPresentInGrib(grib,GRB_TPOT,LV_SIGMA,9950));
	msg += tr("    Neige (risque) : %1\n").arg(dataPresentInGrib(grib,GRB_SNOW_CATEG,LV_GND_SURF,0));
	msg += tr("    Neige (épaisseur) : %1\n").arg(dataPresentInGrib(grib,GRB_SNOW_DEPTH,LV_GND_SURF,0));
	msg += tr("    Humidité spécifique : %1\n").arg(dataPresentInGrib(grib,GRB_HUMID_SPEC,LV_ABOV_GND,0));

	msg += tr("\n");
	msg += tr("Grille : %1 points (%2x%3)\n")
					.arg(gr->getNi()*gr->getNj()).arg(gr->getNi()).arg(gr->getNj());
	msg += tr("\n");
	msg += tr("Etendue : ");
	QString pos1, pos2;
	pos1 = Util::formatPosition( gr->getX(0), gr->getY(0) );
	pos2 = Util::formatPosition( gr->getX(gr->getNi()-1), gr->getY(gr->getNj()-1) );
	msg += tr("%1  ->  %2\n").arg( pos1, pos2);
	
	msg += tr("\n");
	msg += tr("Date de référence : %1\n")
					.arg(Util::formatDateTimeLong(gr->getRecordRefDate()));
	//--------------------------------------------------------------
	QMessageBox::information (this,
		tr("Informations sur le fichier GRIB"),
		msg );
}

//========================================================================
void MainWindow::slotDateGribChanged(int id)
{
    time_t tps = menuBar->getDateGribById(id);
    //printf("id= %d : %s\n",id, qPrintable(formatDateTimeLong(tps)));
    terre->setCurrentDate( tps );

    // Ajuste les actions disponibles
    menuBar->acDatesGrib_prev->setEnabled( (id > 0) );
    menuBar->acDatesGrib_next->setEnabled( (id < menuBar->cbDatesGrib->count()-1) );
}
//-------------------------------------------------
void MainWindow::slotDateGribChanged_next()
{
    int id = menuBar->cbDatesGrib->currentIndex();
    if (id < menuBar->cbDatesGrib->count()-1) {
        menuBar->cbDatesGrib->setCurrentIndex(id+1);
        slotDateGribChanged(id+1);
    }
}
//-------------------------------------------------
void MainWindow::slotDateGribChanged_prev()
{
    int id = menuBar->cbDatesGrib->currentIndex();
    if (id > 0) {
        menuBar->cbDatesGrib->setCurrentIndex(id-1);
        slotDateGribChanged(id-1);
    }
}
//-------------------------------------------------
void MainWindow::slotWindArrows(bool b)
{
    // pas de barbules sans flèches
    menuBar->acView_Barbules->setEnabled(b);
}

//-------------------------------------------------
void MainWindow::setMenubarColorMapMode(int colorMapMode)
{
    MenuBar  *mb = menuBar;
    QAction  *act = NULL;
	switch (colorMapMode)
	{
		case MapDrawer::drawWind :
			act = mb->acView_WindColors;
			break;
		case MapDrawer::drawRain :
			act = mb->acView_RainColors;
			break;
		case MapDrawer::drawCloud :
			act = mb->acView_CloudColors;
			break;
		case MapDrawer::drawHumid :
			act = mb->acView_HumidColors;
			break;
		case MapDrawer::drawTemp :
			act = mb->acView_TempColors;
			break;
		case MapDrawer::drawTempPot :
			act = mb->acView_TempPotColors;
			break;
/*		case MapDrawer::drawDewpoint :
			act = mb->acView_DewpointColors;
			break;*/
		case MapDrawer::drawDeltaDewpoint :
			act = mb->acView_DeltaDewpointColors;
			break;
		case MapDrawer::drawSnowCateg :
			act = mb->acView_SnowCateg;
			break;
		case MapDrawer::drawFrzRainCateg :
			act = mb->acView_FrzRainCateg;
			break;
		case MapDrawer::drawSnowDepth :
			act = mb->acView_SnowDepth;
			break;
		case MapDrawer::drawCAPEsfc :
			act = mb->acView_CAPEsfc;
			break;
	}
    mb->acView_GroupColorMap->setCheckedAction(act, true);
}
//-------------------------------------------------
void MainWindow::slot_GroupColorMap(QAction *act)
{
    MenuBar  *mb = menuBar;
    int mode;
    if (act == mb->acView_WindColors)
    	mode = MapDrawer::drawWind;
    else if (act == mb->acView_RainColors)
    	mode = MapDrawer::drawRain;
    else if (act == mb->acView_CloudColors)
    	mode = MapDrawer::drawCloud;
    else if (act == mb->acView_HumidColors)
    	mode = MapDrawer::drawHumid;
    else if (act == mb->acView_TempColors)
    	mode = MapDrawer::drawTemp;
    else if (act == mb->acView_TempPotColors)
    	mode = MapDrawer::drawTempPot;
/*    else if (act == mb->acView_DewpointColors)
    	mode = MapDrawer::drawDewpoint;*/
    else if (act == mb->acView_DeltaDewpointColors)
    	mode = MapDrawer::drawDeltaDewpoint;
    else if (act == mb->acView_SnowCateg)
    	mode = MapDrawer::drawSnowCateg;
    else if (act == mb->acView_FrzRainCateg)
    	mode = MapDrawer::drawFrzRainCateg;
    else if (act == mb->acView_SnowDepth)
    	mode = MapDrawer::drawSnowDepth;
    else if (act == mb->acView_CAPEsfc)
    	mode = MapDrawer::drawCAPEsfc;
    else
    	mode = MapDrawer::drawNone;
    
    terre->setColorMapMode( mode );
}

//-------------------------------------------------
void MainWindow::statusBar_showSelectedZone()
{
    double x0,y0,  x1,y1;
    terre->getSelectedLine(&x0,&y0, &x1,&y1);
    
    QString message =
            tr("Sélection: ")
            + Util::formatPosition(x0,y0)
            + " -> "
            + Util::formatPosition(x1,y1);

    Orthodromie orth(x0,y0, x1,y1);
    
    message = message+ "   "
                + tr("(dist.orthodromique:")
                + Util::formatDistance(orth.getDistance())
                + tr("  init.dir: %1°").arg(qRound(orth.getAzimutDeg()))
                + ")";
    
    statusBar->showMessage(message);
}

//--------------------------------------------------------------
void MainWindow::slotMouseClicked(QMouseEvent * e)
{
statusBar_showSelectedZone();

	mouseClicX = e->x();
 	mouseClicY = e->y();
    switch (e->button()) {
        case Qt::LeftButton : {
            break;
        }
        case Qt::MidButton :   // Centre la carte sur le point
            proj->setCentralPixel(e->x(), e->y());
			terre->setProjection(proj);
            break;
            
        case Qt::RightButton :
            // Affiche un menu popup
            menuPopupBtRight->exec(QCursor::pos());
            break;

        default :
            break;
    }
}
//--------------------------------------------------------------
void MainWindow::slotMouseMoved(QMouseEvent * e)
{
    double xx, yy;
    proj->screen2map(e->x(), e->y(), &xx, &yy);
    if (terre->isSelectingZone())
    {
        statusBar_showSelectedZone();
    }
    else
    {
        GribPointInfo  pf(terre->getGribPlot()->getGribReader(),
							xx, yy, terre->getGribPlot()->getCurrentDate() );
        boardToolBar->showGribPointInfo(pf);
    }
}





