/**********************************************************************
zyGrib: meteorological GRIB file viewer
Copyright (C) 2008 - Jacques Zaninetti - http://www.zygrib.org

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
***********************************************************************/

#include <QMessageBox>
#include <cmath>
#include <cassert>

#include <QPainter>
#include <QColorDialog>
#include <QFontDialog>

#include "MeteotableOptionsDialog.h"
#include "Util.h"
#include "GribRecord.h"

//===========================================================================
// DialogMeteotableOptions
//===========================================================================
DialogMeteotableOptions::DialogMeteotableOptions()
		 : QDialog()
{
	setupUi(this);
	initInterface();
    
    connect(btCancel, SIGNAL(clicked()), this, SLOT(slotBtCancel()));
    connect(btOK, SIGNAL(clicked()), this, SLOT(slotBtOK()));
    
    connect(btAddVisibleItem,    SIGNAL(clicked()), this, SLOT(slotChangeVisibleItems()));
    connect(btRemoveVisibleItem, SIGNAL(clicked()), this, SLOT(slotChangeVisibleItems()));
}

//-------------------------------------------------------------------------------
void DialogMeteotableOptions::slotBtOK()
{
    // Write hidden data in settings
	for (int i=0; i<listHiddenData->count(); i++) {
		uint grbcode = (uint) listHiddenData->item(i)->type();
		Util::setSetting (getSettingName_vis(grbcode), false);
		Util::setSetting (getSettingName_pos(grbcode), i);
	}
    // Write visible data in settings
	for (int i=0; i<listVisibleData->count(); i++) {
		uint grbcode = (uint) listVisibleData->item(i)->type();
		Util::setSetting (getSettingName_vis(grbcode), true);
		Util::setSetting (getSettingName_pos(grbcode), i);
	}

	Util::setSetting("MTABLE_cloudsColorMode", inputCloudsColorMode->itemData(inputCloudsColorMode->currentIndex()).toString());
	Util::setSetting("MTABLE_showWindArrows", cbShowWindArrows->isChecked());
    
    accept();
}
//-------------------------------------------------------------------------------
void DialogMeteotableOptions::slotBtCancel()
{
    reject();
}
//=============================================================================
void DialogMeteotableOptions::slotChangeVisibleItems()
{
	QListWidget *listSource, *listDestin;
	
	if (sender() == btAddVisibleItem) {
		listSource = listHiddenData;
		listDestin = listVisibleData;
	}
	else {
		listSource = listVisibleData;
		listDestin = listHiddenData;
	}
	
	QList<QListWidgetItem *> list = listSource->selectedItems ();
	QList<QListWidgetItem *>::iterator it;
	for (it = list.begin(); it != list.end(); it++)
	{
		QListWidgetItem *item = *it;
		QListWidgetItem *item2 = new QListWidgetItem(item->text(), NULL, item->type());
		listDestin->addItem (item2);
	}
	
	qDeleteAll(listSource->selectedItems());
	listHiddenData->sortItems (Qt::AscendingOrder);
}

//=============================================================================
// GUI
//=============================================================================
void DialogMeteotableOptions::initInterface()
{
	createDataLists();
	
	inputCloudsColorMode->addItem (tr("100 % -> blanc"), "white");
	inputCloudsColorMode->addItem (tr("100 % -> foncé"), "black");
	inputCloudsColorMode->setCurrentIndex (
			inputCloudsColorMode->findData (
					Util::getSetting("MTABLE_cloudsColorMode", "black").toString() ));

	cbShowWindArrows->setChecked (
				Util::getSetting("MTABLE_showWindArrows", true).toBool() );
}


//-----------------------------------------------------------------------
void DialogMeteotableOptions::addData
		( QString title, uchar grbtype, uchar leveltype, uint level,
		  bool defaultvis, int defaultpos)
{
	uint grbcode = (uint) GribCode::makeCode (grbtype, leveltype, level);
	addItemInDataList (title,
			(int) grbcode,
			Util::getSetting (getSettingName_vis(grbcode), defaultvis).toBool(),
			Util::getSetting (getSettingName_pos(grbcode), defaultpos++).toInt()
	);
}
//-----------------------------------------------------------------------
void DialogMeteotableOptions::addItemInDataList
				(QString name, int code, bool isVisible, int position)
{
	QListWidget *listData =
					isVisible ? listVisibleData : listHiddenData;
	QListWidgetItem *item = new QListWidgetItem (name, 0, code);
	assert (item);
	listData->insertItem(position, item);
}

//-------------------------------------------------------------------------
void DialogMeteotableOptions::createDataLists()
{
	int  defaultpos = 1000;
    //----------------------------------
	bool defaultvis = true;
    //----------------------------------
    addData (tr("Vent (10 m)"), GRB_WIND_XY2D,LV_ABOV_GND,10, defaultvis,defaultpos);
    addData (tr("Nébulosité"),  GRB_CLOUD_TOT,LV_ATMOS_ALL,0, defaultvis,defaultpos);
    addData (tr("Cumul de précipitations"), GRB_PRECIP_TOT,LV_GND_SURF,0, defaultvis,defaultpos);
    addData (tr("Température (2 m)"), 		GRB_TEMP,LV_ABOV_GND,2, defaultvis,defaultpos);
    addData (tr("Ecart température-point de rosée (2 m)"), GRB_DIFF_TEMPDEW,LV_ABOV_GND,2, defaultvis,defaultpos);
    addData (tr("Isotherme 0°C"),     GRB_GEOPOT_HGT,LV_ISOTHERM0,0, defaultvis,defaultpos);
    addData (tr("Pression (niveau de la mer)"), GRB_PRESSURE,LV_MSL,0, defaultvis,defaultpos);
    //----------------------------------
    defaultvis = false;
    //----------------------------------
    addData (tr("Point de rosée (2 m)"),    GRB_DEWPOINT,LV_ABOV_GND,2, defaultvis,defaultpos);
    addData (tr("Humidité relative (2 m)"), GRB_HUMID_REL,LV_ABOV_GND,2, defaultvis,defaultpos);
    addData (tr("Température potentielle (sigma 995)"), GRB_TPOT,LV_SIGMA,9950, defaultvis,defaultpos);
    addData (tr("Température min (2 m)"),  GRB_TMIN,LV_ABOV_GND,2, defaultvis,defaultpos);
    addData (tr("Température max (2 m)"),  GRB_TMAX,LV_ABOV_GND,2, defaultvis,defaultpos);
    addData (tr("Neige (risque)"),         GRB_SNOW_CATEG,LV_GND_SURF,0, defaultvis,defaultpos);
    addData (tr("Neige (épaisseur)"),      GRB_SNOW_DEPTH,LV_GND_SURF,0, defaultvis,defaultpos);
    addData (tr("Pluie verglaçante (risque)"), GRB_FRZRAIN_CATEG,LV_GND_SURF,0, defaultvis,defaultpos);
    addData (tr("CAPE (surface)"),         GRB_CAPE,LV_GND_SURF,0, defaultvis,defaultpos);
    //----------------------------------
	listHiddenData->sortItems (Qt::AscendingOrder);
}










