/**********************************************************************
zyGrib: meteorological GRIB file viewer
Copyright (C) 2008-2010 - Jacques Zaninetti - http://www.zygrib.org

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
***********************************************************************/

#include "GriddedPlotter.h"


//--------------------------------------------------------------------
GriddedPlotter::GriddedPlotter ()
{
	initializeColors ();
	windAltitude = Altitude (LV_TYPE_NOT_DEFINED,0);
}

//--------------------------------------------------------------------
void GriddedPlotter::initializeColors ()
{
	setCloudsColorMode("cloudsColorMode");

	isobarsStep = 4;
    isotherms0Step = 50;

    mapColorTransp = 200;
        
    windArrowSpace = 28;      // distance mini entre flèches
    windBarbuleSpace = 34;    // distance mini entre flèches
    
    windArrowSpaceOnGrid = 20;      // distance mini entre flèches
    windBarbuleSpaceOnGrid = 28;    // distance mini entre flèches
    
    windArrowSize = 14;       // longueur des flèches
    windBarbuleSize = 26;     // longueur des flèches avec barbules
    
    // Color scale for wind in beaufort
    windColor[ 0].setRgba(qRgba(   0,  80, 255,  mapColorTransp));
    windColor[ 1].setRgba(qRgba(   0, 150, 255,  mapColorTransp));
    windColor[ 2].setRgba(qRgba(   0, 200, 255,  mapColorTransp));
    windColor[ 3].setRgba(qRgba(   0, 250, 180,  mapColorTransp));
    windColor[ 4].setRgba(qRgba(   0, 230, 150,  mapColorTransp));
    windColor[ 5].setRgba(qRgba( 255, 255,   0,  mapColorTransp));
    windColor[ 6].setRgba(qRgba( 255, 220,   0,  mapColorTransp));
    windColor[ 7].setRgba(qRgba( 255, 180,   0,  mapColorTransp));
    windColor[ 8].setRgba(qRgba( 255, 120,   0,  mapColorTransp));
    windColor[ 9].setRgba(qRgba( 230, 120,   0,  mapColorTransp));
    windColor[10].setRgba(qRgba( 220,  80,   0,  mapColorTransp));
    windColor[11].setRgba(qRgba( 200,  50,  30,  mapColorTransp));
    windColor[12].setRgba(qRgba( 170,   0,  50,  mapColorTransp));
    windColor[13].setRgba(qRgba( 150,   0,  30,  mapColorTransp));
    // Color scale for rain in mm/h
    rainColor[ 0].setRgba(qRgba(255,255,255,  mapColorTransp));
    rainColor[ 1].setRgba(qRgba(200,240,255,  mapColorTransp));
    rainColor[ 2].setRgba(qRgba(150,240,255,  mapColorTransp));
    rainColor[ 3].setRgba(qRgba(100,200,255,  mapColorTransp));
    rainColor[ 4].setRgba(qRgba( 50,200,255,  mapColorTransp));
    rainColor[ 5].setRgba(qRgba(  0,150,255,  mapColorTransp));
    rainColor[ 6].setRgba(qRgba(  0,100,255,  mapColorTransp));
    rainColor[ 7].setRgba(qRgba(  0, 50,255,  mapColorTransp));
    rainColor[ 8].setRgba(qRgba( 50,  0,255,  mapColorTransp));
    rainColor[ 9].setRgba(qRgba(100,  0,255,  mapColorTransp));
    rainColor[10].setRgba(qRgba(150,  0,255,  mapColorTransp));
    rainColor[11].setRgba(qRgba(200,  0,255,  mapColorTransp));
    rainColor[12].setRgba(qRgba(250,  0,255,  mapColorTransp));
    rainColor[13].setRgba(qRgba(200,  0,200,  mapColorTransp));
    rainColor[14].setRgba(qRgba(150,  0,150,  mapColorTransp));
    rainColor[15].setRgba(qRgba(100,  0,100,  mapColorTransp));
    rainColor[16].setRgba(qRgba( 50,  0,50,  mapColorTransp));
}
 
//--------------------------------------------------------------------------
QRgb  GriddedPlotter::getRainColor(double v, bool smooth)
{
    QRgb rgb = 0;
    int  ind;
    
    double indf = cbrt(67.5*v);        // TODO better color map!!!!
	if (v > 0)
		indf += 0.2;    
    
    ind = (int) floor(Util::inRange(indf, 0.0, 15.0));
    
    if (smooth && ind<16) {
        // Interpolation de couleur
        QColor c1 = rainColor[ind];
        QColor c2 = rainColor[ind+1];
        double dcol = indf-ind;
        rgb = qRgba(
            (int)( (double) c1.red()  *(1.0-dcol) + dcol*c2.red()   +0.5),
            (int)( (double) c1.green()*(1.0-dcol) + dcol*c2.green() +0.5),
            (int)( (double) c1.blue() *(1.0-dcol) + dcol*c2.blue()  +0.5),
            mapColorTransp
            );
    }
    else {
		ind = (int) (indf + 0.5);
		ind = Util::inRange(ind, 0, 15);
        rgb = rainColor[ind].rgba();
    }
    return rgb;
}
//--------------------------------------------------------------------------
QRgb  GriddedPlotter::getSnowDepthColor(double mm, bool smooth)
{
    QRgb rgb = 0;
    int  ind;
    double v = mm / 3.0;
    double indf = cbrt(900*v);        // TODO better color map!!!!
	if (v > 0)
		indf += 0.2;    
    ind = (int) floor(Util::inRange(indf, 0.0, 15.0));
    
    if (smooth && ind<16) {
        // Interpolation de couleur
        QColor c1 = rainColor[ind];
        QColor c2 = rainColor[ind+1];
        double dcol = indf-ind;
        rgb = qRgba(
            (int)( (double) c1.red()  *(1.0-dcol) + dcol*c2.red()   +0.5),
            (int)( (double) c1.green()*(1.0-dcol) + dcol*c2.green() +0.5),
            (int)( (double) c1.blue() *(1.0-dcol) + dcol*c2.blue()  +0.5),
            mapColorTransp
            );
    }
    else {
		ind = (int) (indf + 0.5);
		ind = Util::inRange(ind, 0, 15);
        rgb = rainColor[ind].rgba();
    }
    return rgb;
}
//--------------------------------------------------------------------------
QRgb  GriddedPlotter::getCloudColor(double v, bool smooth) {
    QRgb rgb = 0;
    if (!smooth) {
        v = 10.0*floor(v/10.0);
    }
	v = v*v*v/10000;
    int r,g,b;
    int tr;
    if (isCloudsColorModeWhite)
    {
		double k = 2.55;
		r = (int)(k*v);
		g = (int)(k*v);
		b = (int)(k*v);
		tr = (int)(2.5*v);
	}
	else {
		r = 255 - (int)(1.6*v);
		g = 255 - (int)(1.6*v);
		b = 255 - (int)(2.0*v);
		tr = mapColorTransp;
	}
    rgb = qRgba(r,g,b,  tr);
    return rgb;
}
//--------------------------------------------------------------------------
QRgb  GriddedPlotter::getDeltaTemperaturesColor(double v, bool smooth) {
	v = 100.0 - Util::inRange(20.0*v, 0.0, 100.0);
    QRgb rgb = 0;
    if (!smooth) {
        v = 10.0*floor(v/10.0);
    }
    int r,g,b;
    int tr;
		r = 255 - (int)(1.5*v);
		g = 255 - (int)(1.5*v);
		b = 255 - (int)(2.2*v);
		tr = mapColorTransp;
    rgb = qRgba(r,g,b,  tr);
    return rgb;
}
//--------------------------------------------------------------------------
QRgb  GriddedPlotter::getAltitudeColor(double v, const Altitude &alt, bool smooth) {
    if (!smooth) {
        v = 12*floor(v/12);
    } 
	double k, vmin=0, vmax=20000, vmoy=10000;
	if (alt == Altitude(LV_ISOTHERM0,0)) {
		vmin = 0;  vmax = 5000;
	}
	else if (alt.levelType == LV_ISOBARIC) {
		switch (alt.levelValue) {
			case 200 :
				vmoy = 11800; k=0.06; break;
			case 300 :
				vmoy = 9200; k=0.08; break;
			case 500 :
				vmoy = 5600; k=0.10;  break;
			case 700 :
				vmoy = 3000; k=0.15; break;
			case 850 :
				vmoy = 1460; k=0.25; break;
		}
		vmin = vmoy - k*vmoy;
		vmax = vmoy + k*vmoy;
	}
	double t0 = vmin;  // valeur mini de l'échelle
	double t1 = vmax;  // valeur maxi de l'échelle
	double b0 = 0;    // min beauforts
	double b1 = 12;   // max beauforts
	double eqbeauf = b0 + (v-t0)*(b1-b0)/(t1-t0);
	if (eqbeauf < 0)
		eqbeauf = 0;
	else if (eqbeauf > 12)
		eqbeauf = 12;
	return getWindColor(Util::BeaufortToKmh_F(eqbeauf), smooth);
}
//--------------------------------------------------------------------------
QRgb  GriddedPlotter::getHumidColor(double v, bool smooth) {
    QRgb rgb = 0;
    if (!smooth) {
        v = 10.0*floor(v/10.0);
    }
    
    //v = v*v*v/10000;
    v = v*v/100;
    
    int r = 255 - (int)(2.4*v);
    int g = 255 - (int)(2.4*v);
    int b = 255 - (int)(1.2*v);
    rgb = qRgba(r,g,b,  mapColorTransp);
    return rgb;
}
//--------------------------------------------------------------------------
QRgb  GriddedPlotter::getCAPEColor(double v, bool smooth)
{
	double x = sqrt(v)*70.71;
	double t0 = 0;  // valeur mini de l'échelle
	double t1 = 4000;  // valeur maxi de l'échelle
	double b0 = 0;    // min beauforts
	double b1 = 12;   // max beauforts
	double eqbeauf = b0 + (x-t0)*(b1-b0)/(t1-t0);
	if (eqbeauf < 0)
		eqbeauf = 0;
	else if (eqbeauf > 12)
		eqbeauf = 12;
	return getWindColor(Util::BeaufortToKmh_F(eqbeauf), smooth);
}
//--------------------------------------------------------------------------
QRgb  GriddedPlotter::getTemperatureColor(double v, bool smooth)
{
	// Même échelle colorée que pour le vent
	double x = v-273.15;
	double t0 = -30;  // valeur mini de l'échelle
	double t1 =  40;  // valeur maxi de l'échelle
	double b0 = 0;    // min beauforts
	double b1 = 12;   // max beauforts
	double eqbeauf = b0 + (x-t0)*(b1-b0)/(t1-t0);
	if (eqbeauf < 0)
		eqbeauf = 0;
	else if (eqbeauf > 12)
		eqbeauf = 12;
	return getWindColor(Util::BeaufortToKmh_F(eqbeauf), smooth);
}
//--------------------------------------------------------------------------
QRgb  GriddedPlotter::getPressureColor(double v, bool smooth)
{
	// Même échelle colorée que pour le vent
	double x = v/100.0;	// Pa->hPa
	double t0 = 960;  // valeur mini de l'échelle
	double t1 = 1050;  // valeur maxi de l'échelle
	double b0 = 0;    // min beauforts
	double b1 = 12;   // max beauforts
	double eqbeauf = b0 + (x-t0)*(b1-b0)/(t1-t0);
	if (eqbeauf < 0)
		eqbeauf = 0;
	else if (eqbeauf > 12)
		eqbeauf = 12;
	return getWindColor(Util::BeaufortToKmh_F(eqbeauf), smooth);
}
//--------------------------------------------------------------------------
QRgb  GriddedPlotter::getWindColor(double v, bool smooth) {
    QRgb rgb = 0;
    int beauf;
    if (! smooth) {
        beauf = Util::kmhToBeaufort(v);
        rgb = windColor[beauf].rgba();
    }
    else {
        // Interpolation de couleur
        double fbeauf = Util::kmhToBeaufort_F(v);
        QColor c1 = windColor[ (int) fbeauf ];
        QColor c2 = windColor[ (int) fbeauf +1 ];
        double dcol = fbeauf-floor(fbeauf);
        rgb = qRgba(
            (int)( c1.red()  *(1.0-dcol) + dcol*c2.red()   +0.5),
            (int)( c1.green()*(1.0-dcol) + dcol*c2.green() +0.5),
            (int)( c1.blue() *(1.0-dcol) + dcol*c2.blue()  +0.5),
            mapColorTransp
            );
    }
    return rgb;
}


//==================================================================================
// Flèches de direction du vent
//==================================================================================
void GriddedPlotter::drawTransformedLine( QPainter &pnt,
        double si, double co,int di, int dj, int i,int j, int k,int l)
{
    int ii, jj, kk, ll;
    ii = (int) (i*co-j*si +0.5) + di;
    jj = (int) (i*si+j*co +0.5) + dj;
    kk = (int) (k*co-l*si +0.5) + di;
    ll = (int) (k*si+l*co +0.5) + dj;
    // Clip forcé à cause d'un bug qpixmap sous windows
    int w = pnt.device()->width();
    int h = pnt.device()->height();
    if (       Util::isInRange(ii, 0, w)
            && Util::isInRange(kk, 0, w)
            && Util::isInRange(jj, 0, h)
            && Util::isInRange(ll, 0, h) )
        pnt.drawLine(ii, jj, kk, ll);
}
//-----------------------------------------------------------------------------
void GriddedPlotter::drawWindArrow(QPainter &pnt, int i, int j, double vx, double vy)
{
    double ang = atan2(vy, -vx);
    double si=sin(ang),  co=cos(ang);
    QPen pen( windArrowColor);
    pen.setWidth(2);
    pnt.setPen(pen);
    drawTransformedLine(pnt, si,co, i-windArrowSize/2,j,  0,0, windArrowSize, 0);   // hampe
    drawTransformedLine(pnt, si,co, i-windArrowSize/2,j,  0,0,  5, 2);   // flèche
    drawTransformedLine(pnt, si,co, i-windArrowSize/2,j,  0,0,  5,-2);   // flèche
}

//-----------------------------------------------------------------------------
void GriddedPlotter::drawWindArrowWithBarbs(
			QPainter &pnt,
			int i, int j, double vx, double vy,
			bool south,
			QColor arrowColor
	)
{
    double  vkn = sqrt(vx*vx+vy*vy)*3.6/1.852;
    double ang = atan2(vy, -vx);
    double si=sin(ang),  co=cos(ang);
    
    QPen pen( arrowColor);
    pen.setWidth(2);
    pnt.setPen(pen);
    pnt.setBrush(Qt::NoBrush);
    
    if (vkn < 1)
    {
        int r = 5;     // vent très faible, dessine un cercle
        pnt.drawEllipse(i-r,j-r,2*r,2*r);
    }
    else {
        // Flèche centrée sur l'origine
        int dec = -windBarbuleSize/2;
        drawTransformedLine(pnt, si,co, i,j,  dec,0,  dec+windBarbuleSize, 0);   // hampe
        drawTransformedLine(pnt, si,co, i,j,  dec,0,  dec+5, 2);    // flèche
        drawTransformedLine(pnt, si,co, i,j,  dec,0,  dec+5, -2);   // flèche
        
	 	int b1 = dec+windBarbuleSize -4;  // position de la 1ère barbule
		if (vkn >= 7.5  &&  vkn < 45 ) {
			b1 = dec+windBarbuleSize;  // position de la 1ère barbule si >= 10 noeuds
		}
        
        if (vkn < 7.5) {  // 5 ktn
            drawPetiteBarbule(pnt,south, si,co, i,j, b1);
        }
        else if (vkn < 12.5) { // 10 ktn
            drawGrandeBarbule(pnt,south, si,co, i,j, b1);
        }
        else if (vkn < 17.5) { // 15 ktn
            drawGrandeBarbule(pnt,south, si,co, i,j, b1);
            drawPetiteBarbule(pnt,south, si,co, i,j, b1-4);
        }
        else if (vkn < 22.5) { // 20 ktn
            drawGrandeBarbule(pnt,south, si,co, i,j, b1);
            drawGrandeBarbule(pnt,south, si,co, i,j, b1-4);
        }
        else if (vkn < 27.5) { // 25 ktn
            drawGrandeBarbule(pnt,south, si,co, i,j, b1);
            drawGrandeBarbule(pnt,south, si,co, i,j, b1-4);
            drawPetiteBarbule(pnt,south, si,co, i,j, b1-8);
        }
        else if (vkn < 32.5) { // 30 ktn
            drawGrandeBarbule(pnt,south, si,co, i,j, b1);
            drawGrandeBarbule(pnt,south, si,co, i,j, b1-4);
            drawGrandeBarbule(pnt,south, si,co, i,j, b1-8);
        }
        else if (vkn < 37.5) { // 35 ktn
            drawGrandeBarbule(pnt,south, si,co, i,j, b1);
            drawGrandeBarbule(pnt,south, si,co, i,j, b1-4);
            drawGrandeBarbule(pnt,south, si,co, i,j, b1-8);
            drawPetiteBarbule(pnt,south, si,co, i,j, b1-12);
        }
        else if (vkn < 45) { // 40 ktn
            drawGrandeBarbule(pnt,south, si,co, i,j, b1);
            drawGrandeBarbule(pnt,south, si,co, i,j, b1-4);
            drawGrandeBarbule(pnt,south, si,co, i,j, b1-8);
            drawGrandeBarbule(pnt,south, si,co, i,j, b1-12);
        }
        else if (vkn < 55) { // 50 ktn
            drawTriangle(pnt,south, si,co, i,j, b1-4);
        }
        else if (vkn < 65) { // 60 ktn
            drawTriangle(pnt,south, si,co, i,j, b1-4);
            drawGrandeBarbule(pnt,south, si,co, i,j, b1-8);
        }
        else if (vkn < 75) { // 70 ktn
            drawTriangle(pnt,south, si,co, i,j, b1-4);
            drawGrandeBarbule(pnt,south, si,co, i,j, b1-8);
            drawGrandeBarbule(pnt,south, si,co, i,j, b1-12);
        }
        else if (vkn < 85) { // 80 ktn
            drawTriangle(pnt,south, si,co, i,j, b1-4);
            drawGrandeBarbule(pnt,south, si,co, i,j, b1-8);
            drawGrandeBarbule(pnt,south, si,co, i,j, b1-12);
            drawGrandeBarbule(pnt,south, si,co, i,j, b1-16);
        }
        else { // > 90 ktn
            drawTriangle(pnt,south, si,co, i,j, b1-4);
            drawTriangle(pnt,south, si,co, i,j, b1-12);
        }
    }
}
//---------------------------------------------------------------
void GriddedPlotter::drawPetiteBarbule(QPainter &pnt, bool south,
                    double si, double co, int di, int dj, int b)
{
    if (south)
        drawTransformedLine(pnt, si,co, di,dj,  b,0,  b+2, -5);
    else
        drawTransformedLine(pnt, si,co, di,dj,  b,0,  b+2, 5);
}
//---------------------------------------------------------------
void GriddedPlotter::drawGrandeBarbule(QPainter &pnt, bool south,
                    double si, double co, int di, int dj, int b)
{
    if (south)
        drawTransformedLine(pnt, si,co, di,dj,  b,0,  b+4,-10);
    else
        drawTransformedLine(pnt, si,co, di,dj,  b,0,  b+4,10);
}
//---------------------------------------------------------------
void GriddedPlotter::drawTriangle(QPainter &pnt, bool south,
                    double si, double co, int di, int dj, int b)
{
    if (south) {
        drawTransformedLine(pnt, si,co, di,dj,  b,0,  b+4,-10);
        drawTransformedLine(pnt, si,co, di,dj,  b+8,0,  b+4,-10);
    }
    else {
        drawTransformedLine(pnt, si,co, di,dj,  b,0,  b+4,10);
        drawTransformedLine(pnt, si,co, di,dj,  b+8,0,  b+4,10);
    }
}

//==========================================================================
// Rectangle translucide sur la zone couverte par les données
void GriddedPlotter::draw_CoveredZone 
						(QPainter &pnt, const Projection *proj)
{
	GriddedReader *reader = getReader ();
    if (reader == NULL) {
        return;
    }   
    double x0,y0, x1,y1;
    int i, j, k,l;
    if (reader->getZoneExtension(&x0,&y0, &x1,&y1))
    {
        pnt.setPen(QColor(120, 120,120));
        pnt.setBrush(QColor(255,255,255,40));
        proj->map2screen(x0,y0, &i, &j);
        proj->map2screen(x1,y1, &k, &l);
        pnt.drawRect(i, j, k-i, l-j);
        proj->map2screen(x0-360.0,y0, &i, &j);
        proj->map2screen(x1-360.0,y1, &k, &l);
        pnt.drawRect(i, j, k-i, l-j);
    }
}

//==========================================================================
// draw colored map

//--------------------------------------------------------------------------
// Carte de couleurs générique en dimension 1
//--------------------------------------------------------------------------
void  GriddedPlotter::drawColorMapGeneric_1D (
		QPainter &pnt, const Projection *proj, bool smooth,
		DataCode dtc,
		QRgb (GriddedPlotter::*function_getColor) (double v, bool smooth)
	)
{
	GriddedRecord *rec = getReader()->getRecord (dtc, currentDate);
    if (rec == NULL || !rec->isOk())
        return;
    int i, j;
    double x, y, v;
    int W = proj->getW();
    int H = proj->getH();
    QRgb   rgb;
    QImage *image = new QImage(W,H,QImage::Format_ARGB32);
    image->fill( qRgba(0,0,0,0));
    for (i=0; i<W-1; i+=2) {
        for (j=0; j<H-1; j+=2)
        {
            proj->screen2map(i,j, &x, &y);
            
            if (! rec->isXInMap(x))
                x += 360.0;    // tour complet ?
/*            if (! rec->isXInMap(x))
                x -= 720.0;    // tour complet ?*/
                
            if (rec->isPointInMap(x, y))
            {
                v = rec->getInterpolatedValue (dtc, x, y, mustInterpolateValues);
                if (v != GRIB_NOTDEF)
                {
                    rgb = (this->*function_getColor) (v, smooth);
                    image->setPixel(i,  j, rgb);
                    image->setPixel(i+1,j, rgb);
                    image->setPixel(i,  j+1, rgb);
                    image->setPixel(i+1,j+1, rgb);
                }
            }
        }
    }
	pnt.drawImage(0,0,*image);
    delete image;
}
//--------------------------------------------------------------------------
// Carte de couleurs générique en dimension 2
//--------------------------------------------------------------------------
void  GriddedPlotter::drawColorMapGeneric_2D (
		QPainter &pnt, const Projection *proj, bool smooth,
		DataCode dtcX, DataCode dtcY, 
		QRgb (GriddedPlotter::*function_getColor) (double v, bool smooth)
	)
{
	GriddedRecord *recX = getReader()->getRecord (dtcX, currentDate);
	GriddedRecord *recY = getReader()->getRecord (dtcY, currentDate);
    if (recX == NULL || !recX->isOk() || recY == NULL || !recY->isOk())
        return;
    int i, j;
    double x, y, vx, vy, v;
    int W = proj->getW();
    int H = proj->getH();
    QRgb   rgb;
    QImage *image = new QImage(W,H,QImage::Format_ARGB32);
    image->fill( qRgba(0,0,0,0));
    for (i=0; i<W-1; i+=2) {
        for (j=0; j<H-1; j+=2)
        {
            proj->screen2map(i,j, &x, &y);
            
            if (! recX->isXInMap(x))
                x += 360.0;    // tour complet ?
/*            if (! recX->isXInMap(x))
                x -= 720.0;    // tour complet ?*/
			
            if (recX->isPointInMap(x, y))
            {
                vx = recX->getInterpolatedValue (dtcX, x, y, mustInterpolateValues);
                vy = recY->getInterpolatedValue (dtcY, x, y, mustInterpolateValues);
				
                if (vx != GRIB_NOTDEF && vy != GRIB_NOTDEF)
                {
                    v = sqrt(vx*vx+vy*vy)*3.6;		// m/s => km/h
                    rgb = (this->*function_getColor) (v, smooth);
                    image->setPixel(i,  j, rgb);
                    image->setPixel(i+1,j, rgb);
                    image->setPixel(i,  j+1, rgb);
                    image->setPixel(i+1,j+1, rgb);
                }
            }
        }
    }
	pnt.drawImage(0,0,*image);
    delete image;
}
//--------------------------------------------------------------------------
// Carte de couleurs générique de la différence entre 2 champs
//--------------------------------------------------------------------------
void  GriddedPlotter::drawColorMapGeneric_Abs_Delta_Data (
		QPainter &pnt, const Projection *proj, bool smooth,
		DataCode dtc1, DataCode dtc2, 
		QRgb (GriddedPlotter::*function_getColor) (double v, bool smooth)
	)
{
	GriddedRecord *rec1 = getReader()->getRecord (dtc1, currentDate);
	GriddedRecord *rec2 = getReader()->getRecord (dtc2, currentDate);
    if (rec1 == NULL || !rec1->isOk() || rec2 == NULL || !rec2->isOk())
        return;
    int i, j;
    double x, y, vx, vy, v;
    int W = proj->getW();
    int H = proj->getH();
    QRgb   rgb;
    QImage *image = new QImage(W,H,QImage::Format_ARGB32);
    image->fill( qRgba(0,0,0,0));
    for (i=0; i<W-1; i+=2) {
        for (j=0; j<H-1; j+=2)
        {
            proj->screen2map(i,j, &x, &y);
            
            if (! rec1->isXInMap(x))
                x += 360.0;    // tour complet ?
/*            if (! rec1->isXInMap(x))
                x -= 720.0;    // tour complet ?*/
                
            if (rec1->isPointInMap(x, y))
            {
                vx = rec1->getInterpolatedValue (dtc1, x, y, mustInterpolateValues);
                vy = rec2->getInterpolatedValue (dtc2, x, y, mustInterpolateValues);

                if (vx != GRIB_NOTDEF && vy != GRIB_NOTDEF)
                {
                    v = fabs(vx-vy);
                    rgb = (this->*function_getColor) (v, smooth);
                    image->setPixel(i,  j, rgb);
                    image->setPixel(i+1,j, rgb);
                    image->setPixel(i,  j+1, rgb);
                    image->setPixel(i+1,j+1, rgb);
                }
            }
        }
    }
	pnt.drawImage(0,0,*image);
    delete image;
}


//-------------------------------------------------------------------------
void GriddedPlotter::draw_listIsolines (
						std::list<IsoLine *> & listIsolines,
						QPainter &pnt, const Projection *proj)
{
    std::list<IsoLine *>::iterator it;
    for(it=listIsolines.begin(); it!=listIsolines.end(); it++)
    {
        (*it)->drawIsoLine (pnt, proj);
    }
}
//--------------------------------------------------------------------------
void GriddedPlotter::draw_listIsolines_labels (
						std::list<IsoLine *>  & listIsolines,
						double coef,
						QColor &color,
						QPainter &pnt, const Projection *proj)
{
    std::list<IsoLine *>::iterator it;
    int nbseg = 0;
    for(it=listIsolines.begin(); it!=listIsolines.end(); it++)
    {
        nbseg += (*it)->getNbSegments();
    }
    int nbpix, density, first;
    nbpix = proj->getW()*proj->getH();
    if (nbpix == 0)
        return;
    double r = (double)nbseg/nbpix *1000;
    double dens = 10;
    density =  (int) (r*dens +0.5);
    if (density < 20)
        density = 20;
    first = 0;
    for(it=listIsolines.begin(); it!=listIsolines.end(); it++)
    {
        first += 20;
        (*it)->drawIsoLineLabels(pnt, color, proj, density, first, coef);
    }
}
//----------------------------------------------------
void GriddedPlotter::complete_listIsolines (
				std::list<IsoLine *> *listIsolines,
				DataCode dtc,
				double dataMin, double dataMax, double dataStep
		)
{
    Util::cleanListPointers (*listIsolines);    
	GriddedReader *reader = getReader ();
    if (reader == NULL) {
        return;
    }   
    GriddedRecord *rec = reader->getRecord (dtc, currentDate);
    if (rec == NULL)
        return;
	IsoLine *iso;
	for (double val=dataMin; val<=dataMax; val += dataStep)
	{
		iso = new IsoLine (dtc, val, rec);
		if (iso != NULL) {
			if (iso->getNbSegments()>0)
				listIsolines->push_back (iso);
			else
				delete iso;
		}
	}
}

//======================================================================
void GriddedPlotter::draw_DATA_Labels (
				DataCode dtc, 
				QFont 	 labelsFont,
				QColor   labelsColor,
				QString  (formatLabelFunction) (float v, bool withUnit),
				QPainter &pnt, const Projection *proj)
{
	GriddedReader *reader = getReader();
    if (reader == NULL)
        return;
	GriddedRecord *rec = reader->getRecord (dtc, currentDate);
	if (rec == NULL)
		return;
    QFontMetrics fmet (labelsFont);
    pnt.setFont (labelsFont);
    pnt.setPen  (labelsColor);

	double x, y, v;
    int i, j, dimin, djmin;
    dimin = 50;
    djmin = 30;
    for (j=0; j<proj->getH(); j+= djmin) {
        for (i=0; i<proj->getW(); i+= dimin) {
            proj->screen2map(i,j, &x,&y);
            v = rec->getInterpolatedValue (dtc, x, y, mustInterpolateValues);
            if (v!= GRIB_NOTDEF) {
                QString strtemp = formatLabelFunction (v,false);
                pnt.drawText(i-fmet.width("XXX")/2, j+fmet.ascent()/2, strtemp);
            }
        }
    } 
} 
//------------------------------------------------------------
void GriddedPlotter::draw_DATA_MinMax (
						DataCode dtc, 
						double   meanValue,
						QString  minSymbol,
						QString  maxSymbol,
						QFont 	 labelsFont,
						QColor   labelsColor,
						QPainter &pnt, const Projection *proj)
{
	GriddedReader *reader = getReader();
    if (reader == NULL)
        return;
	GriddedRecord *rec = reader->getRecord (dtc, currentDate);
	if (rec == NULL)
		return;
    QFontMetrics fmet (labelsFont);
    pnt.setFont (labelsFont);
    pnt.setPen  (labelsColor);

    int i, j, Ni, Nj, pi,pj;
    double x, y, v;
          
    Ni = rec->getNi();
    Nj = rec->getNj();

    for (j=1; j<Nj-1; j++) {     // !!!! 1 to end-1
        for (i=1; i<Ni-1; i++) {
            v = rec->getValueOnRegularGrid (dtc, i, j );
            if ( v < meanValue
                   && v < rec->getValueOnRegularGrid (dtc, i-1, j-1 )  // Minima local ?
                   && v < rec->getValueOnRegularGrid (dtc, i-1, j   )
                   && v < rec->getValueOnRegularGrid (dtc, i-1, j+1 )
                   && v < rec->getValueOnRegularGrid (dtc, i  , j-1 )
                   && v < rec->getValueOnRegularGrid (dtc, i  , j+1 )
                   && v < rec->getValueOnRegularGrid (dtc, i+1, j-1 )
                   && v < rec->getValueOnRegularGrid (dtc, i+1, j   )
                   && v < rec->getValueOnRegularGrid (dtc, i+1, j+1 )
            ) {
                x = rec->getX(i);
                y = rec->getY(j);
                proj->map2screen(x,y, &pi, &pj);
                pnt.drawText(pi-fmet.width('L')/2, pj+fmet.ascent()/2, minSymbol);
                proj->map2screen(x-360.0,y, &pi, &pj);
                pnt.drawText(pi-fmet.width('L')/2, pj+fmet.ascent()/2, minSymbol);
            }
            else if ( v > meanValue
                   && v > rec->getValueOnRegularGrid (dtc, i-1, j-1 )  // Maxima local ?
                   && v > rec->getValueOnRegularGrid (dtc, i-1, j   )
                   && v > rec->getValueOnRegularGrid (dtc, i-1, j+1 )
                   && v > rec->getValueOnRegularGrid (dtc, i  , j-1 )
                   && v > rec->getValueOnRegularGrid (dtc, i  , j+1 )
                   && v > rec->getValueOnRegularGrid (dtc, i+1, j-1 )
                   && v > rec->getValueOnRegularGrid (dtc, i+1, j   )
                   && v > rec->getValueOnRegularGrid (dtc, i+1, j+1 )
            ) {
                x = rec->getX(i);
                y = rec->getY(j);
                proj->map2screen(x,y, &pi, &pj);
                pnt.drawText(pi-fmet.width('H')/2, pj+fmet.ascent()/2, maxSymbol);
                proj->map2screen(x-360.0,y, &pi, &pj);
                pnt.drawText(pi-fmet.width('H')/2, pj+fmet.ascent()/2, maxSymbol);
            }
        }
    }
}

//------------------------------------------------------------
void GriddedPlotter::setCurrentDateClosestFromNow ()
{
	GriddedReader *reader = getReader();
    if (reader == NULL || ! reader->isOk())
        return;
	time_t date = reader->getClosestDateFromNow ();
	if (date > 0) {
		setCurrentDate (date);
	}
}
//------------------------------------------------------------
std::set<Altitude> GriddedPlotter::getAllAltitudes (int dataType) const
{
	std::set<Altitude> empty;
	GriddedReader *reader = getReader();
    if (reader == NULL || ! reader->isOk())
        return empty;
	else
		return getReader()->getAllAltitudes (dataType);
}
//------------------------------------------------------------
std::set<DataCode> GriddedPlotter::getAllDataCode () const
{
	std::set<DataCode> empty;
	GriddedReader *reader = getReader();
    if (reader == NULL || ! reader->isOk())
        return empty;
	else
		return getReader()->getAllDataCode ();
}

